/*
 *    pgfe
 *
 *    Copyright(C) 2002 Adam Siegel
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 */

#include <gtk/gtk.h>
#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include <libpq-fe.h>
#include <errno.h>

#define MODE_ADD 0
#define MODE_MODIFY 1
#define DEL_INDEX 20
#define DEL_ROW 21
#define DEL_TABLE 22
#define DEL_VIEW 23

typedef struct
{
     char* alias;
     char* host;
     char* database;
     char* username;
     char* password;
} host_rec_t;

typedef struct 
{
     char* host;
     char* database;
     char* table;
     char* sort;
     char* filter;
} hist_rec_t;

static GHashTable* host_hash;
static GList* history_list = NULL;
static int history_index = 0;
static PGconn* conn = NULL;
static GtkWidget* table_data = NULL;
static char* selected_table = NULL;
static char* selected_view = NULL;
static char* selected_index = NULL;
static int num_fields;
static int table_data_row = 0;
static int edit_mode;
static char* sort;
static char* filter;
static char* database;
static char* host;
static GList* adhoc_list = NULL;
static GList* alias_list = NULL;
static int confirm = 0;

extern GtkWidget* window1;
extern GtkWidget* window2;
extern GtkWidget* window3;
extern GtkWidget* window4;
extern GtkWidget* window5;
extern GtkWidget* window6;
extern GtkWidget* window7;
extern GtkWidget* window8;
extern GtkWidget* window9;

static void updateForeignKeyList();
static void updateIndexList();
static void updateViewList();
static void updateTableList();
static void updateTableDataList();
static void updateTablePrivsList();
static void updateViewDataList();
static void updateColumnList();
static void updateOperatorList();
static void updateAggregateList();
static void updateSequenceList();
static void updateFunctionList();
static void updateTypeList();
static void updateUserList();
static void log(char* s);
static void insertRow();
static void updateRow();
static void deleteRow();
static void deleteIndex();
static void deleteTable();
static void deleteView();
static void updateHistory();
static GtkWidget* createResults(char* sql);
static void showConfirm(int id, char* msg);
static void logSQL(char* sql);
static void saveHostRecs(char* alias, host_rec_t* rec, FILE* fp);
static void on_data_select_row(GtkCList* clist,
                               gint row,
                               gint column,
                               GdkEvent *event,
                               gpointer user_data);

/*****************************************************************************/
void done()
{
     PQfinish(conn);
     savePrefs();
     exit(0);
}

/*****************************************************************************/
void
on_exit1_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
     done();
}

/*****************************************************************************/
void
on_connect_1_activate                  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
     gtk_widget_show(window2);
}

/*****************************************************************************/
void
on_button1_clicked                     (GtkButton       *button,
                                        gpointer         user_data)
{
     char* alias =
          gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(
               lookup_widget(window2, "combo2"))->entry));
     char* username = 
          gtk_entry_get_text(GTK_ENTRY(lookup_widget(window2, "entry3")));
     char* password = 
          gtk_entry_get_text(GTK_ENTRY(lookup_widget(window2, "entry4")));
     char* connstr[256];
     char msg[128];
     host_rec_t* host_rec;
     GList* new_items = NULL;

     host = gtk_entry_get_text(GTK_ENTRY(lookup_widget(window2, "entry1")));
     database = gtk_entry_get_text(GTK_ENTRY(lookup_widget(window2, "entry2")));
    
     sprintf((char*)connstr, "host=%s dbname=%s user=%s password=%s",
             host, database, username, password);
    
     if (conn)
     {
          PQfinish(conn);
     }
    
     conn = PQconnectdb((char*)connstr);
    
     if (PQstatus(conn) != CONNECTION_OK)
     {
          sprintf(msg, "ERROR: Unable to connect to %s:%s", 
                  host, database);
          log(msg);
          return;
     }

     if (strlen(alias))
     {
          g_list_free(new_items);

          host_rec = (host_rec_t*)malloc(sizeof(host_rec_t));
          host_rec->alias = (char*)strdup(alias);
          host_rec->host = (char*)strdup(host);
          host_rec->database = (char*)strdup(database);
          host_rec->username = (char*)strdup(username);
          host_rec->password = (char*)strdup(password);
          g_hash_table_insert(host_hash, host_rec->alias, host_rec);

          alias_list = g_list_append(alias_list, host_rec->alias);

          gtk_combo_set_popdown_strings(
               GTK_COMBO(lookup_widget(window2, "combo2")), alias_list);
     }

     gtk_entry_set_text(GTK_ENTRY(lookup_widget(window1, "entry7")), host);
     gtk_entry_set_text(GTK_ENTRY(lookup_widget(window1, "entry8")), database);
     gtk_entry_set_text(GTK_ENTRY(lookup_widget(window1, "entry9")), username);
    
     updateTableList();
     updateViewList();
     updateTypeList();
     updateUserList();
     updateOperatorList();
     updateAggregateList();
     updateSequenceList();
     updateFunctionList();

     gtk_widget_hide(window2);
}

/*****************************************************************************/
void
on_button2_clicked                     (GtkButton       *button,
                                        gpointer         user_data)
{
     gtk_widget_hide(window2);
}

/*****************************************************************************/
void updateViewList()
{
     PGresult* res;
     int i;
     char* cols[1];
     GtkWidget* clist = lookup_widget(window1, "clist6");
    
     res = PQexec(conn, "select viewname from pg_views where viewname "
                  "not like 'pg_%' order by viewname");
    
     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));
    
     for (i = 0; i < PQntuples(res); i++)
     {
          cols[0] = PQgetvalue(res, i, 0);
          gtk_clist_append(GTK_CLIST(clist), cols);
     }
    
     gtk_clist_thaw(GTK_CLIST(clist));
    
     PQclear(res);
}

/*****************************************************************************/
void updateTableList()
{
     PGresult* res;
     int i;
     char* cols[1];
     GtkWidget* clist = lookup_widget(window1, "clist1");
     GtkWidget* cb = lookup_widget(window6, "checkbutton1");
     char sql[256];
     char* table_filter;

     table_filter = gtk_entry_get_text(GTK_ENTRY(lookup_widget(window1, "entry14")));

     if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(cb)))
     {
          if (strlen(table_filter))
          {
               sprintf(sql, "select tablename from pg_tables "
                       "where tablename ilike '%%%s%%' order by tablename",
                       table_filter);
          }
          else
          {
               strcpy(sql, "select tablename from pg_tables "
                      "order by tablename");
          }
     }
     else
     {
          if (strlen(table_filter))
          {
               sprintf(sql, 
                       "select tablename from pg_tables where "
                       "tablename not like 'pg_%%' and tablename ilike '%%%s%%' "
                       "order by tablename", table_filter);
          }
          else
          {
               strcpy(sql, "select tablename from pg_tables where "
                       "tablename not like 'pg_%' order by tablename");
          }
     }

     res = PQexec(conn, sql);

     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));
    
     for (i = 0; i < PQntuples(res); i++)
     {
          cols[0] = PQgetvalue(res, i, 0);
          gtk_clist_append(GTK_CLIST(clist), cols);
     }
    
     gtk_clist_thaw(GTK_CLIST(clist));
    
     PQclear(res);
}

/*****************************************************************************/
static void updateViewDataList()
{
     static GtkWidget* clist = NULL;
     char sql[256];

     if (GTK_CLIST(clist))
     {
         gtk_widget_destroy(clist);
         clist = NULL;
     }

     sprintf(sql, "select * from \"%s\"", selected_view);

     clist = createResults(sql);

     gtk_widget_show(clist);

     gtk_container_add(GTK_CONTAINER(lookup_widget(window1, "scrolledwindow11")), clist);
}

/*****************************************************************************/
GtkWidget* createResults(char* sql)
{
     PGresult* res;
     int num_fields;
     int i;
     int j;
     char* cols[256];
     GtkWidget* clist;

     memset(cols, 0, sizeof(cols));

     logSQL(sql);

     res = PQexec(conn, sql);

     if (PQresultStatus(res) != PGRES_TUPLES_OK)
     {
          log(PQresultErrorMessage(res));
          PQclear(res);
          return NULL;
     }
    
     num_fields = PQnfields(res);
    
     for (i = 0; i < num_fields; i++)
     {
          cols[i] = PQfname(res, i);
     }
     
     clist = gtk_clist_new_with_titles(num_fields + 1, cols);

     for (i = 0; i < num_fields; i++)
     {
          gtk_clist_set_column_auto_resize(GTK_CLIST(clist), i, TRUE);
          gtk_clist_set_column_max_width(GTK_CLIST(clist), i, 200);
     }

     for (i = 0; i < PQntuples(res); i++)
     {
          for (j = 0; j < num_fields; j++)
          {
               cols[j] = PQgetvalue(res, i, j);

               if (strlen(cols[j]) > 256) 
               {
                    cols[j][256] = 0;
               }
          }
        
          gtk_clist_append(GTK_CLIST(clist), cols);
     }
    
     PQclear(res);

     return clist;
}

/*****************************************************************************/
static void updateTableDataList()
{
     PGresult* res;
     char sql[256];
     char* cols[256];
     int i;
     int j;
     char limit[20];
     int maxrows;

     for (i = 0; i < 256; i++)
     {
          cols[i] = NULL;
     }

     if (table_data != NULL)
     {
          gtk_widget_destroy(table_data);
          table_data = NULL;
     }

     sort = gtk_entry_get_text(GTK_ENTRY(lookup_widget(window1, "entry5")));
     filter = gtk_entry_get_text(GTK_ENTRY(lookup_widget(window1, "entry6")));

     if (!strncmp(selected_table, "pg_", 3))
     {
          sprintf(sql, "select 0, * from %s", selected_table);
     }
     else
     {
          sprintf(sql, "select oid, * from %s", selected_table);
     }
    
     if (strlen(filter))
     {
          strcat(sql, " where ");
          strcat(sql, filter); 
     }
    
     if (strlen(sort))
     {
          strcat(sql, " order by ");
          strcat(sql, sort);
     }
    
     maxrows = atoi(gtk_entry_get_text(GTK_ENTRY(lookup_widget(window6, "entry10"))));
    
     if (!maxrows) maxrows = 200;
    
     sprintf(limit, "%i", maxrows);
    
     strcat(sql, " limit ");
     strcat(sql, limit);

     logSQL(sql);

     res = PQexec(conn, sql);
    
     if (PQresultStatus(res) != PGRES_TUPLES_OK)
     {
          log(PQresultErrorMessage(res));
          PQclear(res);
          return;
     }
    
     num_fields = PQnfields(res);
    
     for (i = 0; i < num_fields; i++)
     {
          cols[i] = PQfname(res, i);
     }

     table_data = gtk_clist_new_with_titles(num_fields + 1, cols);

     gtk_signal_connect(GTK_OBJECT(table_data), "select_row",
                        GTK_SIGNAL_FUNC(on_data_select_row),
                        NULL);

     for (i = 0; i <= num_fields; i++)
     {
          gtk_clist_set_column_auto_resize(GTK_CLIST(table_data), i, TRUE);
     }

     for (i = 0; i < PQntuples(res); i++)
     {
          for (j = 0; j < num_fields; j++)
          {
               cols[j] = PQgetvalue(res, i, j);

               if (strlen(cols[j]) > 100) 
               {
                    cols[j][100] = 0;
               }
          }
        
          gtk_clist_append(GTK_CLIST(table_data), cols);
     }
    
     PQclear(res);

     gtk_clist_set_column_visibility(GTK_CLIST(table_data), 0, FALSE);

     gtk_widget_show(table_data);

     gtk_container_add(GTK_CONTAINER(lookup_widget(window1, "scrolledwindow3")), table_data);

}

/*****************************************************************************/
void
on_clist1_select_row                   (GtkCList        *clist,
                                        gint             row,
                                        gint             column,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
     if (!event || (event->type != GDK_2BUTTON_PRESS))
     {
          return;
     }
    
     gtk_clist_get_text(GTK_CLIST(clist), row, 0, &selected_table);
    
     gtk_label_set_text(GTK_LABEL(lookup_widget(window1, "label44")), 
                        selected_table);
     updateForeignKeyList();
     updateIndexList();
     updateColumnList();
     gtk_entry_set_text(GTK_ENTRY(lookup_widget(window1, "entry5")), "");
     gtk_entry_set_text(GTK_ENTRY(lookup_widget(window1, "entry6")), "");
     updateTableDataList();   
     updateTablePrivsList();
     updateHistory();
}

/*****************************************************************************/
static void updateColumnList()
{
     GtkWidget* clist = lookup_widget(window1, "clist3");
     char* cols[5];
     int i;
     int fsize;
     PGresult* res;
     char sql[1024];
     char buf[32];

     sprintf(sql, "select attname, typname, typlen, atttypmod, attnotnull "
             "from pg_attribute, pg_class, pg_type where "
             "pg_attribute.attrelid = pg_class.oid and "
             "pg_class.relname = '%s' and pg_attribute.attnum >= 0 and "
             "pg_attribute.atttypid = pg_type.oid "
             "order by pg_attribute.attnum", selected_table);
       
     res = PQexec(conn, sql);

     if (PQresultStatus(res) != PGRES_TUPLES_OK)
     {
          log(PQresultErrorMessage(res));
          PQclear(res);
          return;
     }
    
     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));
    
     for (i = 0; i < PQntuples(res); i++)
     {
          cols[0] = PQgetvalue(res, i, 0);
          cols[1] = PQgetvalue(res, i, 1);

          if (!strcmp(PQgetvalue(res, i, 2), "-1"))
          {
               if (!strcmp(cols[1], "text"))
               {
                    cols[2] = NULL;
               }
               else
               {
                    fsize = atoi(PQgetvalue(res, i, 3)) - 4;
                    sprintf(buf, "%i", fsize);
                    cols[2] = buf;
               }
          }
          else
          {
               cols[2] = PQgetvalue(res, i, 2);
          }

          if (!strcmp(PQgetvalue(res, i, 4), "t"))
          {
               cols[3] = "No";
          }
          else
          {
               cols[3] = "Yes";
          }

          cols[4] = NULL;

          gtk_clist_append(GTK_CLIST(clist), cols);
     }
    
     gtk_clist_thaw(GTK_CLIST(clist));

     PQclear(res);
}

/*****************************************************************************/
void
on_entry5_activate                     (GtkEditable     *editable,
                                        gpointer         user_data)
{
     updateTableDataList();
     updateHistory();
}

/*****************************************************************************/
void
on_entry6_activate                     (GtkEditable     *editable,
                                        gpointer         user_data)
{
     updateTableDataList();
     updateHistory();

}

/*****************************************************************************/
void
on_button7_clicked                     (GtkButton       *button,
                                        gpointer         user_data)
{
     updateTableDataList();
     updateHistory();
}

/*****************************************************************************/
void log(char* s)
{
     static long rows = 0;
     char* cols[1];
     GtkWidget* clist = lookup_widget(window1, "clist5");
    
     cols[0] = s;
    
     gtk_clist_append(GTK_CLIST(clist), cols);
     gtk_clist_moveto(GTK_CLIST(clist), rows++, 0, 0, 0);
}

/*****************************************************************************/
void
on_button4_clicked                     (GtkButton       *button,
                                        gpointer         user_data)
{
     int i;
     GtkWidget* label;
     GtkWidget* entry;
     GtkWidget* gtable = lookup_widget(window3, "table3");
     GList* children = gtk_container_children(GTK_CONTAINER(gtable));
     GtkWidget* w;

     edit_mode = MODE_ADD;
        
     gtk_window_set_title(GTK_WINDOW(window3), selected_table);
    
     i = 0;
    
     while (w = g_list_nth_data(children, i))
     {
          gtk_widget_destroy(w);
          i++;
     }

     gtk_table_resize(GTK_TABLE(gtable), num_fields, 2);    
        
     for (i = 1; i < num_fields; i++)
     {
          label = gtk_label_new(gtk_clist_get_column_title(GTK_CLIST(table_data), i));
          gtk_widget_show(label);
          gtk_table_attach_defaults(GTK_TABLE(gtable), label, 0, 1, i, i + 1);
          entry = gtk_entry_new();
          gtk_widget_show(entry);
          gtk_table_attach_defaults(GTK_TABLE(gtable), entry, 1, 2, i, i + 1);
     }
    
     gtk_widget_show(window3);
}

/*****************************************************************************/
void
on_button5_clicked                     (GtkButton       *button,
                                        gpointer         user_data)
{
     int i;
     GtkWidget* label;
     GtkWidget* entry;
     GtkWidget* gtable = lookup_widget(window3, "table3");
     GList* children = gtk_container_children(GTK_CONTAINER(gtable));
     GtkWidget* w;
     char* col;
    
     edit_mode = MODE_MODIFY;
    
     gtk_window_set_title(GTK_WINDOW(window3), selected_table);
    
     i = 0;
    
     while (w = g_list_nth_data(children, i))
     {
          gtk_widget_destroy(w);
          i++;
     }

     gtk_table_resize(GTK_TABLE(gtable), num_fields, 2);    
        
     for (i = 1; i < num_fields; i++)
     {
          label = gtk_label_new(gtk_clist_get_column_title(GTK_CLIST(table_data), i));
          gtk_widget_show(label);
          gtk_table_attach_defaults(GTK_TABLE(gtable), label, 0, 1, i, i + 1);
          gtk_clist_get_text(GTK_CLIST(table_data), table_data_row, i, &col);
          entry = gtk_entry_new();
          gtk_entry_set_text(GTK_ENTRY(entry), col);
          gtk_widget_show(entry);
          gtk_table_attach_defaults(GTK_TABLE(gtable), entry, 1, 2, i, i + 1);
     }
    
     gtk_widget_show(window3);

}

/*****************************************************************************/
void
on_button6_clicked                     (GtkButton       *button,
                                        gpointer         user_data)
{
     showConfirm(DEL_ROW, NULL);
}

/*****************************************************************************/
void
on_button8_clicked                     (GtkButton       *button,
                                        gpointer         user_data)
{
     if (edit_mode == MODE_ADD)
     {
          insertRow();
     }
     else if (edit_mode == MODE_MODIFY)
     {
          updateRow();
     }
}

/*****************************************************************************/
static void updateRow()
{
     GList* children = gtk_container_children(GTK_CONTAINER(lookup_widget(window3, "table3")));
     int i = 0;
     GtkWidget* w;
     char sql[1024];
     PGresult* res;
     char* oid;
     char* labels[256];
     char* vals[256];
     int c;
        
     strcpy(sql, "update ");
     strcat(sql, selected_table);
     strcat(sql, " set ");
    
     c = 0;
    
     while (w = g_list_nth_data(children, i))
     {
          if (GTK_IS_LABEL(w))
          {
               labels[c] = GTK_LABEL(w)->label;
               c++;
          }
        
          i++;
     }
    
     c = 0;
     i = 0;
    
     while (w = g_list_nth_data(children, i))
     {
          if (GTK_IS_ENTRY(w))
          {
               vals[c] = gtk_entry_get_text(GTK_ENTRY(w));
               c++;
          }
        
          i++;
     }
    
     for (i = 0; i < num_fields - 1; i++)
     {
          if (strlen(vals[i]) == 0)
          {
               strcat(sql, labels[i]);
               strcat(sql, " = NULL, ");
          }
          else
          {
               strcat(sql, labels[i]);
               strcat(sql, " = '");
               strcat(sql, vals[i]);
               strcat(sql, "', ");
          }
     }

     sql[strlen(sql) - 2] = 0;
            
     strcat(sql, " where oid = ");
     gtk_clist_get_text(GTK_CLIST(table_data), table_data_row, 0, &oid);
     strcat(sql, oid);

     logSQL(sql);
    
     res = PQexec(conn, sql);
    
     if (PQresultStatus(res) != PGRES_COMMAND_OK)
     {
          log(PQresultErrorMessage(res));
     }
     else
     {
          log("Record updated");
     }
    
     PQclear(res);
    
     updateTableDataList();
    
     gtk_widget_hide(window3);
}

/*****************************************************************************/
static void insertRow()
{
     GList* children = gtk_container_children(GTK_CONTAINER(lookup_widget(window3, "table3")));
     int i = 0;
     GtkWidget* w;
     char sql[1024];
     PGresult* res;
        
     strcpy(sql, "insert into ");
     strcat(sql, selected_table);
     strcat(sql, " (");
    
     while (w = g_list_nth_data(children, i))
     {
          if (GTK_IS_LABEL(w))
          {
               strcat(sql, GTK_LABEL(w)->label);
               strcat(sql, ", ");
          }
        
          i++;
     }
    
     sql[strlen(sql) - 2] = 0;
    
     strcat(sql, ") values (");

     i = 0;
        
     while (w = g_list_nth_data(children, i))
     {
          if (GTK_IS_ENTRY(w))
          {
               if (strlen(gtk_entry_get_text(GTK_ENTRY(w))) == 0)
               {
                    strcat(sql, "NULL, ");
               } 
               else
               {
                    strcat(sql, "'");
                    strcat(sql, gtk_entry_get_text(GTK_ENTRY(w)));
                    strcat(sql, "', ");
               }
          }
        
          i++;
     }
    
     sql[strlen(sql) - 2] = 0;

     strcat(sql, ")");

     logSQL(sql);

     res = PQexec(conn, sql);
    
     if (PQresultStatus(res) != PGRES_COMMAND_OK)
     {
          log(PQresultErrorMessage(res));
     }
     else
     {
          log("Record inserted");
     }
        
     PQclear(res);
    
     updateTableDataList();
    
     gtk_widget_hide(window3);
}

/*****************************************************************************/
void
on_button9_clicked                     (GtkButton       *button,
                                        gpointer         user_data)
{
     gtk_widget_hide(window3);
}

/*****************************************************************************/
static
void on_data_select_row               (GtkCList        *clist,
                                      gint             row,
                                      gint             column,
                                      GdkEvent        *event,
                                      gpointer         user_data)
{

     table_data_row = row;
    
     if (!event || (event->type != GDK_2BUTTON_PRESS))
     {
          return;
     }

     on_button5_clicked(NULL, NULL);
}

/*****************************************************************************/
static void deleteRow()
{
     char sql[256];
     char* oid;
     PGresult* res;
        
     gtk_clist_get_text(GTK_CLIST(table_data), table_data_row, 0, &oid);

     if (oid == NULL)
     {
          return;
     }

     sprintf(sql, "delete from %s where oid = %s", selected_table, oid);
    
     logSQL(sql);

     res = PQexec(conn, sql);
    
     if (PQresultStatus(res) != PGRES_COMMAND_OK)
     {
          log(PQresultErrorMessage(res));
     }
     else
     {
          log("Record deleted");
     }
        
     PQclear(res);
    
     updateTableDataList();
}

/*****************************************************************************/
void
on_button10_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     switch (confirm)
     {
     case DEL_ROW:
          deleteRow();
          break;
     case DEL_INDEX:
          deleteIndex();
          break;
     case DEL_TABLE:
          deleteTable();
          break;
     case DEL_VIEW:
          deleteView();
          break;
     }

     gtk_widget_hide(window4);
}

/*****************************************************************************/
void
on_button11_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     gtk_widget_hide(window4);

}

/*****************************************************************************/
void
on_about1_activate                     (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
     gtk_widget_show(window5);
}

/*****************************************************************************/
void
on_button12_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     gtk_widget_hide(window5);
}

/*****************************************************************************/
void
on_button13_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     savePrefs();
     updateTableList();
     gtk_widget_hide(window6);
}

/*****************************************************************************/
void
on_button14_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
    
     gtk_widget_hide(window6);

}

/*****************************************************************************/
void
on_preferences_1_activate              (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
     gtk_widget_show(window6);
}

/*****************************************************************************/
void
on_button18_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
}

/*****************************************************************************/
void
on_clist6_select_row                   (GtkCList        *clist,
                                        gint             row,
                                        gint             column,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{

     if (event->type != GDK_2BUTTON_PRESS)
     {
          return;
     }

     gtk_clist_get_text(clist, row, 0, &selected_view);

     gtk_label_set_text(GTK_LABEL(lookup_widget(window1, "label45")), selected_view);

     updateViewDataList();
}

/*****************************************************************************/
void
on_button19_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     char* name = gtk_entry_get_text(GTK_ENTRY(lookup_widget(window7, "entry12")));
     char* text = gtk_editable_get_chars(GTK_EDITABLE(lookup_widget(window7, "text2")),
                                         0, -1);
     PGresult* res;
     char sql[2048];

     if (strlen(name) == 0)
     {
          log("View name is blank");
          return;
     }

     if (strlen(text) == 0)
     {
          log("View SQL is blank");
          return;
     }

     sprintf(sql, "drop view \"%s\"", name);

     res = PQexec(conn, sql);
     
     PQclear(res);

     sprintf(sql, "create view \"%s\" as %s", name, text);

     res = PQexec(conn, sql);
     
     if (PQresultStatus(res) != PGRES_COMMAND_OK)
     {
          log(PQresultErrorMessage(res));
     }
     else
     {
          log("View inserted");
     }
     
     PQclear(res);
     
     updateViewList();
    
     gtk_widget_hide(window7);

}

/*****************************************************************************/
void
on_button20_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     gtk_widget_hide(window7);

}

/*****************************************************************************/
void
on_button21_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     
     gtk_entry_set_text(GTK_ENTRY(lookup_widget(window7, "entry12")), "");
     gtk_editable_delete_text(GTK_EDITABLE(lookup_widget(window7, "text2")), 0, -1);
     gtk_widget_show(window7);
}

/*****************************************************************************/
void
on_button22_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     PGresult* res;
     char sql[256];
     int position = 0;

     gtk_entry_set_text(GTK_ENTRY(lookup_widget(window7, "entry12")), "");
     gtk_editable_delete_text(GTK_EDITABLE(lookup_widget(window7, "text2")), 0, -1);

     sprintf(sql, "select viewname, definition from pg_views "
             "where viewname = '%s'",
             selected_view);

     res = PQexec(conn, sql);

     if (PQresultStatus(res) != PGRES_TUPLES_OK)
     {
          log(PQresultErrorMessage(res));
          PQclear(res);
          return;
     }

     gtk_entry_set_text(GTK_ENTRY(lookup_widget(window7, "entry12")), 
                        PQgetvalue(res, 0, 0));

     gtk_editable_insert_text(GTK_EDITABLE(lookup_widget(window7, "text2")), 
                              PQgetvalue(res, 0, 1),
                              strlen(PQgetvalue(res, 0, 1)),
                              &position);

     PQclear(res);

     gtk_widget_show(window7);

}

/*****************************************************************************/
void
on_button23_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     showConfirm(DEL_VIEW, NULL);
}

/*****************************************************************************/
static void deleteView()
{
     char sql[256];
     PGresult* res;

     sprintf(sql, "drop view \"%s\"", selected_view);

     res = PQexec(conn, sql);
     
     if (PQresultStatus(res) != PGRES_COMMAND_OK)
     {
          log(PQresultErrorMessage(res));
     }
     else
     {
          log("View dropped");
     }
     
     PQclear(res);
     
     updateViewList();
}

/*****************************************************************************/
gboolean
on_window1_delete_event                (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
     done();
     return FALSE;
}

/*****************************************************************************/
static void updateIndexList()
{
     PGresult* res;
     char sql[256];
     int i;
     char* cols[1];
     GtkWidget* clist = lookup_widget(window1, "clist7");
     GtkWidget* text = lookup_widget(window1, "text3");

     gtk_editable_delete_text(GTK_EDITABLE(text), 0, -1);

     sprintf(sql, "select indexname from pg_indexes where "
             "tablename = '%s' order by indexname", selected_table);

     res = PQexec(conn, sql);
     
     if (PQresultStatus(res) != PGRES_TUPLES_OK)
     {
          log(PQresultErrorMessage(res));
          PQclear(res);
          return;
     }

     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));
     
     for (i = 0; i < PQntuples(res); i++)
     {
          cols[0] = PQgetvalue(res, i, 0);
          gtk_clist_append(GTK_CLIST(clist), cols);
     }

     gtk_clist_thaw(GTK_CLIST(clist));

     PQclear(res);
}

/*****************************************************************************/
void
on_clist7_select_row                   (GtkCList        *clist,
                                        gint             row,
                                        gint             column,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
     /* Show selected index */

     PGresult* res;
     char sql[256];
     GtkWidget* text = lookup_widget(window1, "text3");
     int position = 0;

     gtk_clist_get_text(clist, row, 0, &selected_index);

     sprintf(sql, "select indexdef from pg_indexes where "
             "indexname = '%s' and tablename = '%s'",
             selected_index, selected_table);

     res = PQexec(conn, sql);

     if (PQresultStatus(res) != PGRES_TUPLES_OK)
     {
          log(PQresultErrorMessage(res));
          PQclear(res);
          return;
     }

     gtk_editable_delete_text(GTK_EDITABLE(text), 0, -1);
     gtk_editable_insert_text(GTK_EDITABLE(text),
                              PQgetvalue(res, 0, 0),
                              strlen(PQgetvalue(res, 0, 0)),
                              &position);

     PQclear(res);
}

/*****************************************************************************/
void loadPrefs()
{
     FILE* fp;
     char filename[1024];
     char s[1024];
     hist_rec_t* hist_rec;
     host_rec_t* host_rec;
     char* p;

     sprintf(filename, "%s/.pgferc", getenv("HOME"));
    
     alias_list = g_list_append(alias_list, (char*)strdup(""));
     host_hash = g_hash_table_new(g_str_hash, g_str_equal);

     if (fp = fopen(filename, "r"))
     {
          while (fgets(s, sizeof(s), fp))
          {
               s[strlen(s) - 1] = 0;
               
               if (!strncmp(s, "maxrows=", 8))
               {   
                    gtk_entry_set_text(GTK_ENTRY(lookup_widget(window6, "entry10")), 
                                       s + 8);
               }
               else if (!strncmp(s, "host=", 5))
               {   
                    gtk_entry_set_text(GTK_ENTRY(lookup_widget(window2, "entry1")), 
                                       s + 5);
               }
               else if (!strncmp(s, "database=", 9))
               {   
                    gtk_entry_set_text(GTK_ENTRY(lookup_widget(window2, "entry2")), 
                                       s + 9);
               }
               else if (!strncmp(s, "username=", 9))
               {   
                    gtk_entry_set_text(GTK_ENTRY(lookup_widget(window2, "entry3")), 
                                       s + 9);
               }
               else if (!strncmp(s, "adhoc=", 6))
               {
                    adhoc_list = g_list_append(adhoc_list, (char*)strdup(s + 6));
               } 
               else if (!strncmp(s, "host=", 5))
               {
                    
               }
               else if (!strncmp(s, "showsys=", 8))
               {   
                    if (s[8] == '1')
                    {
                         gtk_toggle_button_set_active(
                              GTK_TOGGLE_BUTTON(lookup_widget(window6, "checkbutton1")), TRUE);
                    }
                    else
                    {
                         gtk_toggle_button_set_active(
                              GTK_TOGGLE_BUTTON(lookup_widget(window6, "checkbutton1")), FALSE);
                    }
               }
               else if (!strncmp(s, "showsql=", 8))
               {
                    if (s[8] == '1')
                    {
                         gtk_toggle_button_set_active(
                              GTK_TOGGLE_BUTTON(lookup_widget(window6, "checkbutton2")), TRUE);
                    }
                    else
                    {
                         gtk_toggle_button_set_active(
                              GTK_TOGGLE_BUTTON(lookup_widget(window6, "checkbutton2")), FALSE);
                    }
               }
               else if (!strncmp(s, "histrec=", 8))
               {
                    hist_rec = (hist_rec_t*)malloc(sizeof(hist_rec_t));

                    hist_rec->database = (char*)strdup(strtok(s + 8, "|"));
                    hist_rec->table = (char*)strdup(strtok(NULL, "|"));
                    hist_rec->sort = (char*)strdup(strtok(NULL, "|"));
                    hist_rec->filter = (char*)strdup(strtok(NULL, "\n"));

                    if (!strcmp(hist_rec->sort, "NULL"))
                    {
                         strcpy(hist_rec->sort, "");
                    }

                    if (!strcmp(hist_rec->filter, "NULL"))
                    {
                         strcpy(hist_rec->filter, "");
                    }

                    history_list = g_list_append(history_list, hist_rec);
               }
               else if (!strncmp(s, "host_rec=", 9))
               {
                    host_rec = (host_rec_t*)malloc(sizeof(host_rec_t));
                    host_rec->alias = (char*)strdup(strtok(s + 9, "|"));
                    host_rec->host = (char*)strdup(strtok(NULL, "|"));
                    host_rec->database = (char*)strdup(strtok(NULL, "|"));
                    host_rec->username = (char*)strdup(strtok(NULL, "|"));
                    host_rec->password = (char*)strdup(strtok(NULL, "\n"));

                    if (!strcmp(host_rec->host, "NULL"))
                    {
                         strcpy(host_rec->host, "");
                    }

                    if (!strcmp(host_rec->database, "NULL"))
                    {
                         strcpy(host_rec->database, "");
                    }

                    if (!strcmp(host_rec->username, "NULL"))
                    {
                         strcpy(host_rec->username, "");
                    }

                    if (!strcmp(host_rec->password, "NULL"))
                    {
                         strcpy(host_rec->password, "");
                    }

                    g_hash_table_insert(host_hash, host_rec->alias, host_rec);
                    alias_list = g_list_append(alias_list, host_rec->alias);
               }
          }
        
          fclose(fp);
     }

     history_index = g_list_length(history_list) - 1;

     if (g_list_length(adhoc_list) > 0)
     {
          gtk_combo_set_popdown_strings(
               GTK_COMBO(lookup_widget(window1, "combo1")), adhoc_list);
     }

     gtk_combo_set_popdown_strings(
          GTK_COMBO(lookup_widget(window2, "combo2")), alias_list);
}

/*****************************************************************************/
void savePrefs()
{
     FILE* fp;
     char filename[1024];
     hist_rec_t* hist_rec;
     int i = 0;
     char* adhoc;

     sprintf(filename, "%s/.pgferc", getenv("HOME"));
    
     if (fp = fopen(filename, "w"))
     {
          fprintf(fp, "maxrows=%s\n", 
                  gtk_entry_get_text(GTK_ENTRY(lookup_widget(window6, "entry10"))));
          fprintf(fp, "host=%s\n", 
                  gtk_entry_get_text(GTK_ENTRY(lookup_widget(window2, "entry1"))));
          fprintf(fp, "database=%s\n", 
                  gtk_entry_get_text(GTK_ENTRY(lookup_widget(window2, "entry2"))));
          fprintf(fp, "username=%s\n", 
                  gtk_entry_get_text(GTK_ENTRY(lookup_widget(window2, "entry3"))));

          if (gtk_toggle_button_get_active(
               GTK_TOGGLE_BUTTON(lookup_widget(window6, "checkbutton1"))))
          {
               fprintf(fp, "showsys=1\n");
          }
          else
          {
               fprintf(fp, "showsys=0\n");
          }

          if (gtk_toggle_button_get_active(
               GTK_TOGGLE_BUTTON(lookup_widget(window6, "checkbutton2"))))
          {
               fprintf(fp, "showsql=1\n");
          }
          else
          {
               fprintf(fp, "showsql=0\n");
          }

          i = g_list_length(history_list) - 20;

          if (i < 0)
          {
               i = g_list_length(history_list);
          }

          while (hist_rec = g_list_nth_data(history_list, i))
          {
               fprintf(fp, "histrec=%s|%s|", 
                       hist_rec->database, hist_rec->table);

               if (strlen(hist_rec->sort))
               {
                    fprintf(fp, "%s|", hist_rec->sort);
               }
               else
               {
                    fprintf(fp, "NULL|");
               }

               if (strlen(hist_rec->filter))
               {
                    fprintf(fp, "%s\n", hist_rec->filter);
               }
               else
               {
                    fprintf(fp, "NULL\n");
               }

               i++;
          }

          i = 0;

          while ((i < 25) && (adhoc = g_list_nth_data(adhoc_list, i)))
          {
               fprintf(fp, "adhoc=%s\n", adhoc); 
               i++;
          }

          g_hash_table_foreach(host_hash, (GHFunc)saveHostRecs, fp);

          fclose(fp);
     }
}

/*****************************************************************************/
void init()
{
     GtkWidget* clist;

     clist = lookup_widget(window1, "clist3");
     gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
     gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE);
     gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 2, TRUE);

     clist = lookup_widget(window1, "clist8");
     gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
     gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE);
     gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 2, TRUE);
     gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 3, TRUE);

     gtk_signal_connect(GTK_OBJECT(GTK_COMBO(lookup_widget(window2, "combo2"))->popwin),
                        "hide", GTK_SIGNAL_FUNC(on_combo2_changed), NULL);
}

/*****************************************************************************/
void
on_button25_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     hist_rec_t* hist_rec;
     int row = 0;
     char* col;
     GtkWidget* clist = lookup_widget(window1, "clist1");

     while (hist_rec = g_list_nth_data(history_list, --history_index))
     {
          if (!strcmp(hist_rec->database, database))
          {
               break;
          }
     }

     if (history_index < 0)
     {
          history_index = 0;
          return;
     }

     gtk_entry_set_text(GTK_ENTRY(lookup_widget(window1, "entry5")), hist_rec->sort);
     gtk_entry_set_text(GTK_ENTRY(lookup_widget(window1, "entry6")), hist_rec->filter);
     selected_table = hist_rec->table;
     gtk_label_set_text(GTK_LABEL(lookup_widget(window1, "label44")), selected_table);
     updateForeignKeyList();
     updateIndexList();
     updateColumnList();
     updateTableDataList();
}

/*****************************************************************************/
void
on_button26_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     hist_rec_t* hist_rec;
     int row = 0;
     char* col;
     GtkWidget* clist = lookup_widget(window1, "clist1");
     
     while (hist_rec = g_list_nth_data(history_list, ++history_index))
     {
          if (!strcmp(hist_rec->database, database))
          {
               break;
          }
     }

     if (history_index >= g_list_length(history_list))
     {
          history_index = g_list_length(history_list) - 1;
          return;
     }
     
     gtk_entry_set_text(GTK_ENTRY(lookup_widget(window1, "entry5")), hist_rec->sort);
     gtk_entry_set_text(GTK_ENTRY(lookup_widget(window1, "entry6")), hist_rec->filter);
     selected_table = hist_rec->table;
     gtk_label_set_text(GTK_LABEL(lookup_widget(window1, "label44")), selected_table);
     updateForeignKeyList();
     updateIndexList();
     updateColumnList();
     updateTableDataList();
}

/*****************************************************************************/
void updateHistory()
{
     hist_rec_t* hist_rec;

     hist_rec = (hist_rec_t*)malloc(sizeof(hist_rec_t));

     hist_rec->database = (char*)strdup(database);
     hist_rec->table = (char*)strdup(selected_table);
     hist_rec->sort = (char*)strdup(sort);
     hist_rec->filter = (char*)strdup(filter);

     history_list = g_list_append(history_list, hist_rec);
     history_index = g_list_length(history_list) - 1;
}

/*****************************************************************************/
void updateForeignKeyList()
{
     char sql[1024];
     PGresult* res;
     char* cols[5];
     char* name;
     char* type;
     char* field1;
     char* field2;
     char* table1;
     char* table2;
     int i;
     char ref1[1024];
     char ref2[1024];
     GtkWidget* clist = lookup_widget(window1, "clist8");

     sprintf(sql, "select distinct tgargs from pg_trigger where "
             "tgrelid = (select oid from pg_class where "
             "relname = '%s' order by tgargs)", selected_table);

     res = PQexec(conn, sql);

     if (PQresultStatus(res) != PGRES_TUPLES_OK)
     {
          log(PQresultErrorMessage(res));
          PQclear(res);
          return;
     }

     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));
     
     for (i = 0; i < PQntuples(res); i++)
     {
          name = (char*)strtok(PQgetvalue(res, i, 0), "\\000");
          table1 = (char*)strtok(NULL, "\\000");
          table2 = (char*)strtok(NULL, "\\000");
          type = (char*)strtok(NULL, "\\000");

          cols[0] = name;
          cols[1] = NULL;
          cols[2] = NULL;
          cols[3] = NULL;
          cols[4] = NULL;

          gtk_clist_append(GTK_CLIST(clist), cols);

          while(field1 = (char*)strtok(NULL, "\\000")) 
          {
               field2 = (char*)strtok(NULL, "\\000");

               sprintf(ref1, "%s.%s", table1, field1);
               sprintf(ref2, "%s.%s", table2, field2);

               cols[0] = NULL;
               cols[1] = ref1;
               cols[2] = "->";
               cols[3] = ref2;
               cols[4] = NULL;
                   
               gtk_clist_append(GTK_CLIST(clist), cols);
          }

     }

     gtk_clist_thaw(GTK_CLIST(clist));

     PQclear(res);
     
}

/*****************************************************************************/
static void updateTypeList()
{
     PGresult* res;
     char* cols[3];
     GtkWidget* clist = lookup_widget(window1, "clist9");
     int i;
     char sql[256];

     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));

     gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
     gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE);

     strcpy(sql, "select pg_type.typname, pg_description.description "
            "from pg_type, pg_description where pg_type.oid = "
            "pg_description.objoid order by pg_type.typname");
     
     res = PQexec(conn, sql);

     for (i = 0; i < PQntuples(res); i++)
     {
          cols[0] = PQgetvalue(res, i, 0);
          cols[1] = PQgetvalue(res, i, 1);
          cols[2] = NULL;
          gtk_clist_append(GTK_CLIST(clist), cols);
     }

     gtk_clist_thaw(GTK_CLIST(clist));
     
     PQclear(res);
}

/*****************************************************************************/
static void updateUserList()
{
     PGresult* res;
     char* cols[3];
     GtkWidget* clist = lookup_widget(window1, "clist11");
     int i;
     char sql[256];

     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));

     gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
     gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE);

     strcpy(sql, "select usesysid, usename from pg_shadow "
            "order by usesysid");
     
     res = PQexec(conn, sql);

     for (i = 0; i < PQntuples(res); i++)
     {
          cols[0] = PQgetvalue(res, i, 0);
          cols[1] = PQgetvalue(res, i, 1);
          cols[2] = NULL;
          gtk_clist_append(GTK_CLIST(clist), cols);
     }

     gtk_clist_thaw(GTK_CLIST(clist));
     
     PQclear(res);
}

/*****************************************************************************/
static void updateOperatorList()
{
     PGresult* res;
     char* cols[6];
     GtkWidget* clist = lookup_widget(window1, "clist13");
     int i;
     char sql[512];

     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));

     for (i = 0; i < 6; i++)
     {
          gtk_clist_set_column_auto_resize(GTK_CLIST(clist), i, TRUE);
     }

     strcpy(sql,
            "select o.oprname, "
            "case when o.oprkind = 'l' then null else "
            "format_type(o.oprleft, null) end, "
            "case when o.oprkind = 'r' then null else "
            "format_type(o.oprright, null) end, "
            "format_type(o.oprresult, null), "
            "obj_description(o.oprcode, 'pg_proc') "
            "from pg_operator o order by o.oprname");

     res = PQexec(conn, sql);

     for (i = 0; i < PQntuples(res); i++)
     {
          cols[0] = PQgetvalue(res, i, 0);
          cols[1] = PQgetvalue(res, i, 1);
          cols[2] = PQgetvalue(res, i, 2);
          cols[3] = PQgetvalue(res, i, 3);
          cols[4] = PQgetvalue(res, i, 4);
          cols[5] = NULL;

          gtk_clist_append(GTK_CLIST(clist), cols);
     }

     gtk_clist_thaw(GTK_CLIST(clist));
     
     PQclear(res);
}

/*****************************************************************************/
static void updateAggregateList()
{
     PGresult* res;
     char* cols[6];
     GtkWidget* clist = lookup_widget(window1, "clist14");
     int i;
     char sql[512];

     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));

     for (i = 0; i < 6; i++)
     {
          gtk_clist_set_column_auto_resize(GTK_CLIST(clist), i, TRUE);
     }

     strcpy(sql,
            "select a.aggname, "
            "format_type(a.aggbasetype, NULL), "
            "obj_description(a.oid, 'pg_aggregate') "
            "from pg_aggregate a order by a.aggname");

     res = PQexec(conn, sql);

     for (i = 0; i < PQntuples(res); i++)
     {
          cols[0] = PQgetvalue(res, i, 0);
          cols[1] = PQgetvalue(res, i, 1);
          cols[2] = PQgetvalue(res, i, 2);
          cols[3] = NULL;

          gtk_clist_append(GTK_CLIST(clist), cols);
     }

     gtk_clist_thaw(GTK_CLIST(clist));
     
     PQclear(res);
}

/*****************************************************************************/
static void updateSequenceList()
{
     PGresult* res;
     PGresult* res1;
     char* cols[9];
     GtkWidget* clist = lookup_widget(window1, "clist12");
     int i;
     char sql[512];

     strcpy(sql, "select relname from pg_class where relkind = 'S' "
            "order by relname");

     res = PQexec(conn, sql);

     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));

     for (i = 0; i < 6; i++)
     {
          gtk_clist_set_column_auto_resize(GTK_CLIST(clist), i, TRUE);
     }

     for (i = 0; i < PQntuples(res); i++)
     {
          cols[0] = PQgetvalue(res, i, 0);

          sprintf(sql, "select * from %s", PQgetvalue(res, i, 0));

          res1 = PQexec(conn, sql);

          cols[1] = PQgetvalue(res1, 0, 1);
          cols[2] = PQgetvalue(res1, 0, 2);
          cols[3] = PQgetvalue(res1, 0, 3);
          cols[4] = PQgetvalue(res1, 0, 4);
          cols[5] = PQgetvalue(res1, 0, 5);
          cols[6] = PQgetvalue(res1, 0, 6);
          cols[7] = PQgetvalue(res1, 0, 7);
          cols[8] = PQgetvalue(res1, 0, 8);

          gtk_clist_append(GTK_CLIST(clist), cols);

          PQclear(res1);
     }

     gtk_clist_thaw(GTK_CLIST(clist));
     
     PQclear(res);
}

/*****************************************************************************/
void
on_combo_entry1_activate               (GtkEditable     *editable,
                                        gpointer         user_data)
{
}

/*****************************************************************************/
void
on_button61_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     showConfirm(DEL_INDEX, NULL);
}

/*****************************************************************************/
static void showConfirm(int id, char* msg)
{
     confirm = id;
     gtk_widget_show(window4);
}

/*****************************************************************************/
static void deleteIndex()
{

     PGresult* res;
     char sql[256];

     if (!selected_index)
     {
          log("ERROR: No index selected");
          return;
     }

     sprintf(sql, "drop index %s", selected_index);

     res = PQexec(conn, sql);

     if (PQresultStatus(res) != PGRES_COMMAND_OK)
     {
          log(PQresultErrorMessage(res));
          PQclear(res);
          return;
     }

     PQclear(res);

     selected_index = NULL;

     updateIndexList();

     log("Index dropped");
}

/*****************************************************************************/
void
on_button29_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     showConfirm(DEL_TABLE, NULL);
}

/*****************************************************************************/
static void deleteTable()
{
     PGresult* res;
     char sql[256];

     if (!selected_table)
     {
          log("ERROR: No table selected");
          return;
     }

     sprintf(sql, "drop table %s", selected_table);

     res = PQexec(conn, sql);

     if (PQresultStatus(res) != PGRES_COMMAND_OK)
     {
          log(PQresultErrorMessage(res));
          PQclear(res);
          return;
     }

     PQclear(res);

     selected_table = NULL;

     updateTableList();

     log("table dropped");
}

/*****************************************************************************/
void
on_button68_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     /* vacuum table */

     char sql[64];

     if (!selected_table)
     {
          log("No table selected");
          return;
     }

     sprintf(sql, "vacuum analyze %s", selected_table);

     PQclear(PQexec(conn, sql));

     log("Vacuum complete");
}

/*****************************************************************************/
void
on_button69_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     /* reindex table */

     char sql[64];

     if (!selected_table)
     {
          log("No table selected");
          return;
     }

     sprintf(sql, "reindex table %s", selected_table);

     PQclear(PQexec(conn, sql));

     log("Reindex complete");
}

/*****************************************************************************/
static void logSQL(char* sql)
{
     if (gtk_toggle_button_get_active(
          GTK_TOGGLE_BUTTON(lookup_widget(window6, "checkbutton2"))))
     {
          log(sql);
     }
}

/*****************************************************************************/
static void updateTablePrivsList()
{
     PGresult* res;
     char* cols[6];
     GtkWidget* clist = lookup_widget(window1, "clist16");
     int i;
     char sql[512];
     char* buff;
     char* p;

     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));

     sprintf(sql,
             "select relacl from pg_class "
             "where relkind in ('r', 'v', 'S') and "
             "relname = '%s'", selected_table);
     
     res = PQexec(conn, sql);

     for (i = 0; i < PQntuples(res); i++)
     {
          if (strlen(PQgetvalue(res, i, 0)))
          {
               buff = (char*)strdup(PQgetvalue(res, i, 0));
               
               buff[strlen(buff) - 1] = 0;

               p = (char*)strtok(buff + 1, ",");
               
               while (p)
               {
                    cols[0] = p;
                    gtk_clist_append(GTK_CLIST(clist), cols);
                    p = (char*)strtok(NULL, ",");
               }
               
               free(buff);
          }
     }

     gtk_clist_thaw(GTK_CLIST(clist));
     
     PQclear(res);
}

/*****************************************************************************/
static void updateFunctionList()
{
     PGresult* res;
     char* cols[6];
     GtkWidget* clist = lookup_widget(window1, "clist15");
     int i;
     char sql[512];

     gtk_clist_freeze(GTK_CLIST(clist));
     gtk_clist_clear(GTK_CLIST(clist));

     for (i = 0; i < 6; i++)
     {
          gtk_clist_set_column_auto_resize(GTK_CLIST(clist), i, TRUE);
     }

     strcpy(sql,
            "select format_type(p.prorettype, NULL), "
            "p.proname, oidvectortypes(p.proargtypes) "
            "from pg_proc p "
            "where p.prorettype <> 0 AND "
            "(pronargs = 0 OR oidvectortypes(p.proargtypes) <> '') "
            "order by 2, 1, 3");

     res = PQexec(conn, sql);

     for (i = 0; i < PQntuples(res); i++)
     {
          cols[0] = PQgetvalue(res, i, 0);
          cols[1] = PQgetvalue(res, i, 1);
          cols[2] = PQgetvalue(res, i, 2);
          cols[3] = NULL;

          gtk_clist_append(GTK_CLIST(clist), cols);
     }

     gtk_clist_thaw(GTK_CLIST(clist));
     
     PQclear(res);
}

/*****************************************************************************/
static void saveHostRecs(char* alias, host_rec_t* rec, FILE* fp)
{
     fprintf(fp, "host_rec=%s|", alias);
     fprintf(fp, "%s|", strlen(rec->host) ? rec->host : "NULL");
     fprintf(fp, "%s|", strlen(rec->database) ? rec->database : "NULL");
     fprintf(fp, "%s|", strlen(rec->username) ? rec->username : "NULL");
     fprintf(fp, "%s\n", strlen(rec->password) ? rec->password : "NULL");
}

/*****************************************************************************/
void on_combo2_changed(GtkCombo* combo, gpointer user_data)
{
     char* alias = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(lookup_widget(window2, "combo2"))->entry));
     host_rec_t* host_rec = g_hash_table_lookup(host_hash, alias);

     if (host_rec)
     {
          gtk_entry_set_text(GTK_ENTRY(lookup_widget(window2, "entry1")), host_rec->host);
          gtk_entry_set_text(GTK_ENTRY(lookup_widget(window2, "entry2")), host_rec->database);
          gtk_entry_set_text(GTK_ENTRY(lookup_widget(window2, "entry3")), host_rec->username);
          gtk_entry_set_text(GTK_ENTRY(lookup_widget(window2, "entry4")), host_rec->password);
          
     }
}

/*****************************************************************************/
void
on_entry14_changed                     (GtkEditable     *editable,
                                        gpointer         user_data)
{
     updateTableList();
}


/*****************************************************************************/
void
on_button27_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     /* Show table editor */

     gtk_widget_show(window8);
}


/*****************************************************************************/
void
on_button79_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     /* Show column editor */

     gtk_widget_show(window9);
}


/*****************************************************************************/
void
on_button80_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     /* Delete a column from table editor */
}


/*****************************************************************************/
void
on_button83_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     gtk_widget_hide(window8);
}


/*****************************************************************************/
void
on_button84_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     gtk_widget_hide(window8);
}


/*****************************************************************************/
void
on_button81_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     gtk_widget_hide(window9);
}


/*****************************************************************************/
void
on_button82_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     gtk_widget_hide(window9);

}

/*****************************************************************************/
void
on_button85_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
     /* Execute adhoc query */

     GtkWidget* combo = lookup_widget(window1, "combo1");
     char* sql = (char*)strdup(gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(combo)->entry)));
     static GtkWidget* clist = NULL;

     adhoc_list = g_list_prepend(adhoc_list, sql);
     gtk_combo_set_popdown_strings(GTK_COMBO(combo), adhoc_list);
     gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), "");

     if (clist)
     {
          gtk_widget_destroy(clist);
          clist = NULL;
     }

     clist = createResults(sql);

     if (!clist)
     {
         return;
     }

     gtk_widget_show(clist);

     gtk_container_add(GTK_CONTAINER(
          lookup_widget(window1, "scrolledwindow20")), clist);
}

