
/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2000 William W. Wong
 *  williamw@jps.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package phex;


import java.util.*;
import java.lang.reflect.*;
import java.applet.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.plaf.FontUIResource;
import javax.swing.text.*;

import phex.actions.base.*;
import phex.config.*;


/**
*    The class BaseFrame provides a generic framework to support menu, toolbar,
*    popup menu, and action.  An action can have multiple visual components
*    associated with it, such as a menu item, a popup menu item, or a toolbar
*    button.  When these visual components are pressed, the action is fired.
*    BaseFrame has distilled all the definition of menu and toolbar and action
*    into the TriLife.resource file.  This allows extreme flexibility and
*    extendibility, enabling on-the-fly adding of new operations and new
*    menu items.
*
*/
public class BaseFrame extends JFrame
{
    private static Vector    sFrames = new Vector();

    private Applet            mApplet = null;
    private String            mFrameType;
    protected Hashtable        mActions;
    protected Hashtable        mAccelerators = new Hashtable();


    /**
    *    Default constructor.  Disabled.
    */
    protected BaseFrame()
    {
        // Disabled.
    }


    /**
    *    Constructor.  Sets up the default Look and Feel, the title,
    *    the window icon, the menu bar, the toolbar, and load the actions.
    *
    *    @param frameType    The frame type name used to look up the menu,
    *                        toolbar, and actions from the resource file.
    */
    public BaseFrame(String frameType, Applet applet)
    {
        super();

        // Add to frame list.
        sFrames.addElement(this);

        mFrameType = frameType;
        mApplet = applet;
        mActions = new Hashtable();

        // Set up Look and Feel.
        try
        {
            UIManager.setLookAndFeel(javax.swing.UIManager.getSystemLookAndFeelClassName());
        }
        catch (Exception exc)
        {
            System.err.println("Error loading L&F: " + exc);
        }

        // Set up frame window title, background color, and icon.
        setTitle(Res.getStr(mFrameType + ".Title"));
        setBackground(Color.lightGray);
        ImageIcon    frameIcon = Res.getIcon(mFrameType + ".Icon");
        if (frameIcon != null)
        {
            setIconImage(frameIcon.getImage());
        }

        // Set up menu and toolbar
        JMenuBar    menubar = createMenubar();
        setJMenuBar(menubar);
        JToolBar    toolbar = createToolbar();
        if (toolbar != null)
            getContentPane().add(BorderLayout.NORTH, toolbar);
    }

    public void dispose()
    {
        // Remove from frame list.
        sFrames.removeElement(this);

        super.dispose();
    }

    /**
    *    Load an action class and create an action object from it.  Cache
    *    the action object in memory for later invocation.
    *
    *    @param itemName        The menu item name prefix that denotes the
    *                        action related resource.  It is concatenated
    *                        with ".ActionID" to load the action ID of the
    *                        action, with ".Label" to load the menu item
    *                        label of the action, with ".Icon" to load the
    *                        icon of the action for toolbar button, with
    *                        ".ActionClass" to load the action class.
    *
    *    @return                The action object, or null if something goes wrong.
    *
    */
    private ActionBase loadAction(String itemName)
    {
        // Get the action related resources.
        String        actionID = Res.getStr(itemName + ".ActionID");
        String        label = Res.getStr(itemName + ".Label");
        ImageIcon    icon = Res.getIcon(itemName + ".Icon");
        String        actionClassName = Res.getStr(itemName + ".ActionClass");

        if (actionID == null)
        {
            System.out.println("Missing " + itemName + ".ActionID");
            return null;
        }
        if (actionClassName == null)
        {
            System.out.println("Missing " + itemName + ".ActionClass");
            return null;
        }
        if (mActions.get(actionID) != null)
        {
//            System.out.println("Duplicated actionID: " + actionID + " class: " + actionClassName);
            return (ActionBase)mActions.get(actionID);
        }

        try
        {
            // Set up the parameters for calling the action class's constructor.
            Class[]        classParams = new Class[3];
            Object[]    params = new Object[3];

            classParams[0] = this.getClass().getSuperclass();
            classParams[1] = Class.forName("java.lang.String");
            classParams[2] = Class.forName("javax.swing.Icon");
            params[0] = this;
            params[1] = label;
            params[2] = icon;

            // Load the action class, get it's constructor, and create new obj.
            Class        actionClassObj = Class.forName(actionClassName);
            Constructor    constructor = actionClassObj.getDeclaredConstructor(classParams);
            ActionBase    action = (ActionBase)constructor.newInstance(params);

            mActions.put(actionID, action);
            return action;
        }
        catch (Exception e)
        {
            System.out.println("Fail to load " + actionClassName + ".  " + e);
        }

        return null;
    }


    /**
    *    Create the menu bar for the frame window.  Get the list of menus of
    *    the menu bar from the resource file based on the frame type of this
    *    frame window.  Create each menu in turn and attach it to the menubar.
    *
    *    @return            The JMenuBar object.
    *
    */
    private JMenuBar createMenubar()
    {
        JMenuBar        menuBar = new JMenuBar();
        String            menuBarList = Res.getStr(mFrameType + ".MenuBar");

        if (menuBarList != null)
        {
            StringTokenizer    enumMenuNames = new StringTokenizer(menuBarList);

            // Go through the menu items
            while (enumMenuNames.hasMoreTokens())
            {
                String    menuName = enumMenuNames.nextToken();

                // Get the label of the menu.
                String    label = Res.getStr(menuName + ".Label");
                if (label == null)
                {
                    continue;
                }

                JMenu    menu = new JMenu(label);

                // Get the mnemoic of the menu.
                String    menuMnemonic = Res.getStr(menuName + ".Mnemonic");
                if (menuMnemonic != null)
                {
                    menu.setMnemonic(menuMnemonic.charAt(0));
                }

                // Fill up the menu with it's menu items.
                populateMenu(menu, menuName);
                menuBar.add(menu);
            }
        }

        return menuBar;
    }


    /**
    *    Populate the menu with its menu items.
    *
    *    @param menu            The JMenu object to populate.
    *    @param menuName        The resource name for the menu.  The value of the
    *                        resource is a list of menu item names of the menu.
    *
    */
    private void populateMenu(JMenu menu, String menuName)
    {
        String                menuItemList = Res.getStr(menuName);
        StringTokenizer        enumItemNames = new StringTokenizer(menuItemList);
        String                oldGroupName = "";
        ButtonGroup            group = null;

        // Go through all the menu items and create each one at a time.
        while (enumItemNames.hasMoreTokens())
        {
            String    itemName = enumItemNames.nextToken();
            if (itemName.equals("-"))
            {
                menu.addSeparator();
            }
            else if (Res.getStr(itemName + ".SubMenu") != null)
            {
                // See if it's a submenu
                JMenu        submenu = new JMenu(Res.getStr(itemName + ".Label"));
                String        menuMnemonic = Res.getStr(itemName + ".Mnemonic");
                submenu.setMnemonic(menuMnemonic.charAt(0));
                populateMenu(submenu, itemName);
                menu.add(submenu);
            }
            else
            {
                String        groupName = Res.getStr(itemName + ".Radio");

                // See if menu item belongs to a radio button group.
                if (groupName == null)
                {
                    // Not a radio button at all.
                    group = null;
                    oldGroupName = "";
                }
                else if (!groupName.equals(oldGroupName) || group == null)
                {
                    // It is a new group.  Create it.
                    group = new ButtonGroup();
                    oldGroupName = groupName;
                }

                // Create the menu item.
                JMenuItem    item = createMenuItem(itemName, group);
                if (item != null)
                {
                    menu.add(item);
                }
            }
        }
    }


    /**
    *    Create the menu bar for the frame window.  Get the list of menus of
    *    the menu bar from the resource file based on the frame type of this
    *    frame window.  Create each menu in turn and attach it to the menubar.
    *
    *    @param popupMenuName    The name of the popup menu resource name.
    *
    *    @return            The JMenuBar object.
    *
    */
    public JPopupMenu createPopupMenu(String popupMenuName)
    {
        JPopupMenu    popupMenu = new JPopupMenu();
        populatePopupMenu(popupMenu, popupMenuName);
        return popupMenu;
    }


    /**
    *    Populate the popup menu with its menu items.  This basically does
    *    the same thing as populateMenu() except the passed in parameter is
    *    a JPopupMenu instead of JMenu.  The problem of incompatible menu
    *    classes. Doh!
    *
    *    @param menu            The JPopupMenu object to populate.
    *    @param menuName        The resource name for the menu.  The value of the
    *                        resource is a list of menu item names of the menu.
    *
    */
    public void populatePopupMenu(JPopupMenu menu, String menuName)
    {
        String                menuItemList = Res.getStr(menuName);

        if (menuItemList == null)
        {
            return;
        }

        StringTokenizer        enumItemNames = new StringTokenizer(menuItemList);
        String                oldGroupName = "";
        ButtonGroup            group = null;

        // Go through all the menu items and create each one at a time.
        while (enumItemNames.hasMoreTokens())
        {
            String    itemName = (String)enumItemNames.nextToken();
            if (itemName.equals("-"))
            {
                menu.addSeparator();
            }
            else
            {
                String        groupName = Res.getStr(itemName + ".Radio");

                // See if menu item belongs to a radio button group.
                if (groupName == null)
                {
                    // Not a radio button at all.
                    group = null;
                    oldGroupName = "";
                }
                else if (!groupName.equals(oldGroupName) || group == null)
                {
                    // It is a new group.  Create it.
                    group = new ButtonGroup();
                    oldGroupName = groupName;
                }

                // Create the menu item.
                JMenuItem    item = createMenuItem(itemName, group);
                if (item != null)
                {
                    menu.add(item);
                }
            }
        }
    }


    /**
    *    Create a new menu item from the resource file.
    *
    *    @param itemName        The resource name for the menu item.  It's used
    *                        as prefix of resource name to retrieve different
    *                        resources of the menu item.
    *    @param radioGroup    The radio button group object if the menu item
    *                        belongs to a radio button group.
    *
    *    @return                Return the menu item.
    */
    private JMenuItem createMenuItem(String itemName, ButtonGroup radioGroup)
    {
        String        actionID = Res.getStr(itemName + ".ActionID");

        if (actionID == null)
        {
            System.out.println("Missing " + itemName + ".ActionID");
            return null;
        }

        // Get the various resources for the menu item.
        String        label = Res.getStr(itemName + ".Label");
        ImageIcon    icon = Res.getIcon(itemName + ".Icon");
        boolean        isCheckBox = (Res.getStr(itemName + ".CheckBox") != null);
        String        itemMnemonic = Res.getStr(itemName + ".Mnemonic");
        KeyStroke    accel = Res.getAccelerator(itemName + ".Accelerator");
        boolean        radioSelected = (Res.getStr(itemName + ".Radio.Selected") != null);
        ActionBase    action = loadAction(itemName);
        JMenuItem    item;


        if (radioGroup != null)
        {
            // Create a radio button menu item.
            item = new JRadioButtonMenuItem(label);
            radioGroup.add(item);
            if (radioSelected)
            {
                item.setSelected(true);
            }
        }
        else if (isCheckBox)
        {
            // Create a checkbox menu item.
            item = new JCheckBoxMenuItem(label);
        }
        else
        {
            // Create a regular menu item.
            item = new JMenuItem(label);
        }

        if (action != null)
        {
            // Attach to action object and add itself to the action.
            item.addActionListener(action);
            ((ActionBase)action).addRefreshComponent(item);
        }
        else
        {
            // No action available.  Disable the menu item.
            item.setEnabled(false);
        }

        if (itemMnemonic != null)
        {
            item.setMnemonic(itemMnemonic.charAt(0));
        }

        if (accel != null)
        {
            item.setAccelerator(accel);
            mAccelerators.put(accel, accel);
        }

        if (icon != null && !isCheckBox)
        {
            item.setIcon(icon);
            item.setHorizontalTextPosition(SwingConstants.RIGHT);
        }

        return item;
    }


    /**
    *    Create a new tool bar, with all its buttons.  Note that the toolbar
    *    has to be created after the menu bar is created.  Because the menu
    *    bar loads the actions as it sets up the menu item.
    *
    *    @return                Return the toolbar object.
    */
    private JToolBar createToolbar()
    {
        JToolBar            toolbar = new JToolBar();
        int                    menuCount = 0;
        String                toolbarList = Res.getStr(mFrameType + ".ToolBar");

        if (toolbarList != null)
        {
            StringTokenizer        enum = new StringTokenizer(toolbarList);

            // Go through all the buttons on the toolbar.
            while (enum.hasMoreTokens())
            {
                menuCount++;
                String    item = (String)enum.nextToken();
                if (item.equals("-"))
                {
                    toolbar.addSeparator();
                }
                else
                {
                    boolean        isCheckBox = (Res.getStr(item + ".CheckBox") != null);
                    ActionBase    action = loadAction(item);
                    if (action == null)
                    {
                        System.out.println("No action class for " + item);
                        continue;
                    }

                    AbstractButton    button;

                    if (isCheckBox)
                    {
                        ImageIcon        icon = Res.getIcon(item + ".Icon");
                        ImageIcon        iconoff = Res.getIcon(item + ".Icon.Selected");
                        JToggleButton    jbutton = new JToggleButton(iconoff);
                        jbutton.setSelectedIcon(icon);
                        jbutton.addActionListener(action);
                        toolbar.add(jbutton);
                        button = jbutton;
                    }
                    else
                    {
                        button = toolbar.add(action);
                    }

                    action.addRefreshComponent(button);
                    button.setMargin(new Insets(0, 0, 0, 0));
                    button.setToolTipText(Res.getStr(item + ".Tooltip"));
                    button.setText(null);
                }
            }
        }

        if (menuCount > 0)
        {
            toolbar.add(Box.createHorizontalGlue());
            return toolbar;
        }
        else
        {
            return null;
        }
    }


    /**
    *    Refresh all the action objects.  Call each action object's refresh()
    *    method.  This gives each action object a chance to update its
    *    associated visual components.
    *
    */
    public void refreshAllActions()
    {
        Enumeration    enum = mActions.elements();

        while (enum.hasMoreElements())
        {
            ActionBase    action = (ActionBase)enum.nextElement();
            action.refresh();
        }
    }


    public void refreshAction(String actionID)
    {
        ActionBase    action = (ActionBase)mActions.get(actionID);
        if (action != null)
            action.refresh();
    }


    /**
    *    Perform an action, by its actionID.
    *
    *    @param source        The source of the action.
    *    @param actionID        The ID of the action to perform.
    */
    protected void doAction(Component source, String actionID)
    {
        Action        action = (Action)mActions.get(actionID);

        if (action == null)
        {
            JOptionPane.showMessageDialog(this, "No action defined for " + actionID, "Error", JOptionPane.ERROR_MESSAGE);
            return;
        }

        action.actionPerformed(new ActionEvent(source,
                                               ActionEvent.ACTION_PERFORMED,
                                               actionID));
    }


    public void addRefreshComponent(String actionID, JComponent compToAdd)
    {
        ActionBase    action = (ActionBase)mActions.get(actionID);
        if (action != null)
            action.addRefreshComponent(compToAdd);
    }



    /**
    *    Set the global fonts for different components
    *
    */
    public void setFonts()
    {
        FontUIResource tableFontResource = new FontUIResource( ServiceManager.sCfg.mFontTable );
        UIManager.getDefaults().put("Table.font", tableFontResource );
        UIManager.getDefaults().put("TextPane.font", tableFontResource );
        UIManager.getDefaults().put("TextArea.font", tableFontResource );
        UIManager.getDefaults().put("TextField.font", tableFontResource );
        UIManager.getDefaults().put("PasswordField.font", tableFontResource );
        UIManager.getDefaults().put("EditorPane.font", tableFontResource );
        UIManager.getDefaults().put("ProgressBar.font", tableFontResource );

        FontUIResource menuFontResource = new FontUIResource( ServiceManager.sCfg.mFontMenu );
        UIManager.getDefaults().put("MenuBar.font", menuFontResource );
        UIManager.getDefaults().put("Menu.font", menuFontResource );
        UIManager.getDefaults().put("MenuItem.font", menuFontResource );
        UIManager.getDefaults().put("PopupMenu.font", menuFontResource );
        UIManager.getDefaults().put("CheckBoxMenuItem.font", menuFontResource );
        UIManager.getDefaults().put("RadioButtonMenuItem.font", menuFontResource );

        FontUIResource labelFontResource = new FontUIResource( ServiceManager.sCfg.mFontLabel );
        UIManager.getDefaults().put("CheckBox.font", labelFontResource );
        UIManager.getDefaults().put("ComboBox.font", labelFontResource );
        UIManager.getDefaults().put("Button.font", labelFontResource );
        UIManager.getDefaults().put("Tree.font", labelFontResource );
        UIManager.getDefaults().put("ScrollPane.font", labelFontResource );
        UIManager.getDefaults().put("TabbedPane.font", labelFontResource );
        UIManager.getDefaults().put("TitledBorder.font", labelFontResource );
        UIManager.getDefaults().put("OptionPane.font", labelFontResource );
        UIManager.getDefaults().put("ToolBar.font", labelFontResource );
        UIManager.getDefaults().put("RadioButton.font", labelFontResource );
        UIManager.getDefaults().put("ToggleButton.font", labelFontResource );
        UIManager.getDefaults().put("ToolTip.font", labelFontResource );
        UIManager.getDefaults().put("TableHeader.font", labelFontResource );
        UIManager.getDefaults().put("Panel.font", labelFontResource );
        UIManager.getDefaults().put("List.font", labelFontResource );
        UIManager.getDefaults().put("ColorChooser.font", labelFontResource );
        UIManager.getDefaults().put("Label.font", labelFontResource );
        UIManager.getDefaults().put("Viewport.font", labelFontResource );

        updateComponentsUI();
    }

    public static void setAllFramesLookAndFeel( LookAndFeel laf )
    {
        try
        {
            if ( laf.getID().equals( UIManager.getLookAndFeel().getID() ) )
            {
                // don't do it if already set...
                return;
            }

            UIManager.setLookAndFeel( laf );
            updateComponentsUI();
        }
        catch (Exception e)
        {
            e.printStackTrace();
            JOptionPane.showMessageDialog(
                ServiceManager.getManager().getMainFrame(),
                "Error loading Look & Feel " + e,
                "Error",
                JOptionPane.ERROR_MESSAGE);
        }
    }

    private static void updateComponentsUI()
    {
        for (int i = 0; i < sFrames.size(); i++)
        {
            BaseFrame frame = (BaseFrame)sFrames.elementAt(i);
            SwingUtilities.updateComponentTreeUI( frame );
            frame.refreshAllActions();

            // go through child windows
            Window[] windows = frame.getOwnedWindows();
            for ( int j = 0; j < windows.length; j++ )
            {
                SwingUtilities.updateComponentTreeUI( windows[j] );
            }
        }
    }

    /**
     * Reset the frame to different look and feel
     * @deprecated try to use setAllFramesLookAndFeel() if possible...
     */
    public void resetLookAndFeel()
    {
        try
        {
            UIManager.setLookAndFeel(ServiceManager.sCfg.mLFClassName);
            SwingUtilities.updateComponentTreeUI( this );
        }
        catch (Exception e)
        {
            JOptionPane.showMessageDialog(
                this,
                "Error loading Look & Feel " + e,
                "Error",
                JOptionPane.ERROR_MESSAGE);
        }

        refreshAllActions();
    }

    /**
     * @deprecated try to use setAllFramesLookAndFeel() if possible...
     */
    public static void resetAllFramesLookAndFeel()
    {
        for (int i = 0; i < sFrames.size(); i++)
        {
            BaseFrame frame = (BaseFrame)sFrames.elementAt(i);
            frame.resetLookAndFeel();
        }
    }



    /**
    *    See if we are running as an applet.
    *
    *    @return        true = is an applet, false = not an applet.
    */
    public boolean isApplet()
    {
        return (mApplet != null);
    }


    /**
    *    Center the window on the screen
    *
    *    @param win        The window object to position.
    *    @param offset    The amount to offset from the center of the screen.
    */
    public static void centerWindow(Window win, Point offset)
    {
        Dimension    screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        Dimension    winSize = win.getSize();
        Rectangle    rect = new Rectangle(
                                         (screenSize.width - winSize.width) / 2 + offset.x,
                                         (screenSize.height - winSize.height) / 2 + offset.y,
                                         winSize.width, winSize.height);

        win.setBounds(rect);
    }


    // Center Window on screen
    static void centerAndSizeWindow(Window win, int fraction, int base)
    {
        Dimension    screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        int            width  = screenSize.width * fraction / base;
        int            height = screenSize.height * fraction / base;
        Rectangle    rect = new Rectangle((screenSize.width - width) / 2,
                                         (screenSize.height - height) / 2,
                                         width, height);
        win.setBounds(rect);
    }
}
