/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2000 Alexander Werth
 *  alexander.werth@gmx.de
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package phex;

import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.OutputStreamWriter;
import java.text.DateFormat;
import java.util.Date;

/**
* This class provides a sophisticated debug system that is flexible and easy
* to use.
* <p>
* To setup the debug system the <code>setDebug(String)</code> class method
* is used. It is allowed to call this method more than once.
* <p>
* The string delivered to the setDebug method is parsed and consists of
* letters optionally followed by a single digit. We are case sensitive so
* there should be enough possibilities for now.
* <p>
* Example: -d d2us <br>
* Means: Level 2 debug for download and level 1 debug for
* upload and search. 
* <p>
* Example: gsg0s0 <br>
* Means: gui and search is set to one but set back to 0 afterwards.
* <p>
* t is reserved for a time option.
* <p>
* In Your program You would write something like this: <br>
* if (Debug.DbgDownload()) Debug.msg("Debug output") 
* <p>
* This gets powerfull when combined like this: <br>
* if (Debug.DbgAlex() && Debug.DbgDownload(2)) {...
* <p>
* Feel free to add new debug slots. You have to add 6 lines of code for each: 
* The levelSlotname variable itself, the 3 lines for the case in setDebug()
* and the two lines for the DbgSlotname() methods used by the conditions.
*/
public class Debug
{
        
        /** Don't let anyone instantiate this class */
        private Debug(){}

        /**
         * Standard output stream. This stream could be used for printing
         * debug messages. It could be redirected if needed. I suggest a
         * DebugStream class that keeps track of the amount of data writen
         * and eventually purges part of the debug logs.
         */
        public static PrintWriter out= new PrintWriter(new OutputStreamWriter(System.out));

        /**
         * Special output stream for user relevant information. Some users
         * may want to watch phex working. This shouldn't be redirected.
         * Redirection of this stream would be the users job.
         */
        public static PrintStream verbose= System.out;

        /**
         * These fields hold the debug level selected at startup.
         * If there is a debug message these values are compared
         * against the debug level of the message;
         */
        private static int levelDownload= 0;
        private static int levelUpload= 0;
        private static int levelSearch= 0;
        private static int levelHost= 0;
        private static int levelGui= 0;
        private static int levelVerbose= 0;
        // For things that don't make sense to anyone else
        private static int levelKonrad= 0;
        private static int levelGregor= 0;        
        private static int levelMark= 0;   
        private static int levelAlex= 0;   

        /** 
         * Sets the values acording to the specified String.
         * The String is parsed and consists of letters optionally
         * followed by a single digit. We are case sensitive so there should
         * be enough possibilities for now.
         * <p>
         * Example: -d d2us <br>
         * Means: Level 2 debug for download and level 1 debug for
         * upload and search. 
         * <p>
         * t is reserved for a time option.
         * @return -1 if there was an error
         */
        public static int setDebug(String debugStr)
        {
                int pos=0;
                char c;
                int level;
                if(DbgVerbose(3)) verbose.println("Debug setup string: "+debugStr);
                while(pos<debugStr.length())
                {
                        c= debugStr.charAt(pos);
                        if (c=='t') 
                        {
                                if(DbgVerbose(2)) verbose.println("Debug setup time not yet implemented.");

                                return -1; //continue;
                        }
                        if (pos+1<debugStr.length())
                        {
                                char levelc= debugStr.charAt(pos+1);
                                level="0123456789".indexOf(levelc);
                                if (level<0) level= 1;
                                else pos++;
                        } 
                        else level= 1;
                        pos++;
                        if(DbgVerbose(4)) verbose.println("Debug setup argument: "+c+"  level: "+level);

                        switch (c) {
                        case 'd':
                                levelDownload= level;
                                break;
                        case 'u':
                                levelUpload= level;
                                break;
                        case 's':
                                levelSearch= level;
                                break;
                        case 'h':
                                levelHost= level;
                                break;
                        case 'g':
                                levelGui= level;
                                break;
                        case 'v':
                                levelVerbose= level;
                                break;
                        case 'K':
                                levelKonrad= level;
                                break;
                        case 'G':
                                levelGregor= level;
                                break;
                        case 'M':
                                levelMark= level;
                                break;
                        case 'A':
                                levelAlex= level;
                                break;
                        default:
                                if(DbgVerbose(2)) verbose.println("Debug setup unknown argument");
                                return -1;
                        }
                }
                return 0;
        }

        /**
         * In Your code You could use something like this: <br>
         * if (Debug.DbgDownload()) Debug.msg("Debug output") 
         * <p>
         * This gets powerfull when combined like this: <br>
         * if (Debug.DbgAlex() && Debug.DbgDownload(2)) {...
         */
        public static boolean DbgDownload() { return(levelDownload>0);}
        public static boolean DbgDownload(int i) { return(levelDownload>=i);}

        public static boolean DbgUpload() { return(levelUpload>0);}
        public static boolean DbgUpload(int i) { return(levelUpload>=i);}

        public static boolean DbgSearch() { return(levelSearch>0);}
        public static boolean DbgSearch(int i) { return(levelSearch>=i);}

        public static boolean DbgHost() { return(levelHost>0);}
        public static boolean DbgHost(int i) { return(levelHost>=i);}

        public static boolean DbgGui() { return(levelGui>0);}
        public static boolean DbgGui(int i) { return(levelGui>=i);}

        public static boolean DbgVerbose() { return(levelVerbose>0);}
        public static boolean DbgVerbose(int i) { return(levelVerbose>=i);}

        public static boolean DbgKonrad() { return(levelKonrad>0);}
        public static boolean DbgKonrad(int i) { return(levelKonrad>=i);}

        public static boolean DbgGregor() { return(levelGregor>0);}
        public static boolean DbgGregor(int i) { return(levelGregor>=i);}

        public static boolean DbgMark() { return(levelMark>0);}
        public static boolean DbgMark(int i) { return(levelMark>=i);}

        public static boolean DbgAlex() { return(levelAlex>0);}
        public static boolean DbgAlex(int i) { return(levelAlex>=i);}

        /**
         * Just for convenience. Outputs msg to the debug stream.
         */
        public static void msg(String msg)
        {
                out.println(DateFormat.getTimeInstance(DateFormat.MEDIUM).format(new Date())+": "+msg);
                out.flush();
        }
    
        /**
         * Just for convenience. If level is higher or equal to 
         * the internal level for verbose messages, msg is written 
         * to verbose stream. Usually this is the System.out stream.
         */
        public static void verbose(int level, String msg)
        {
                if (DbgVerbose(level)) verbose.println(msg);
        }
}
