
/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2000 William W. Wong
 *  williamw@jps.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package phex;

// resizeable size (fast) circular queue.
public class FQueue
{
    private int				mMax;
    private int				mCount;
    private int				mHeadIndex;
    private int				mTailIndex;
    private Object[]		mElements;

    private FQueue()
    {
    }

    public FQueue(int maxsize)
    {
        mMax = maxsize;
        mCount = 0;
        mHeadIndex = 0;
        mTailIndex = 0;
        mElements = new Object[mMax];
    }

    public int getCount()
    {
        return mCount;
    }

    public int getMax()
    {
        return mMax;
    }

    public synchronized void resize( int size )
    {
        if ( size == mMax )
        {
            return;
        }
        mMax = size;
        mElements = new Object[ size ];
        mCount = 0;
        mHeadIndex = 0;
        mTailIndex = 0;
    }

    public boolean isEmpty()
    {
        return (mHeadIndex == mTailIndex);
    }


    public synchronized int add(Object obj)
    {
        if ( mElements.length == 0 )
        {
            return -1;
        }
        mHeadIndex = nextPos(mHeadIndex);
        if (mHeadIndex == mTailIndex)
        {
            // Overwrite the last element.
            mTailIndex = nextPos( mTailIndex );
            mCount--;
        }
        mElements[mHeadIndex] = obj;
        mCount++;
        return mHeadIndex;
    }


    public synchronized void addToHead(Object obj)
    {
        mElements[mTailIndex] = obj;
        mTailIndex = prevPos(mTailIndex);
        if (mTailIndex == mHeadIndex)
        {
            // Overwrite the last element.
            mHeadIndex = prevPos(mHeadIndex);
            mCount--;
        }
        mCount++;
    }


    public synchronized Object remove()
    {
        if (mTailIndex == mHeadIndex)
            return null;

        mTailIndex = nextPos(mTailIndex);
        mCount--;
        return mElements[mTailIndex];
    }


    // Get object by 0-based index
    public Object getAt(int index)
    {
        index = mTailIndex + 1 + index;
        if (index >= mMax)
        {
            index = index - mMax;
        }
        return mElements[index];
    }


    private final int nextPos(int pos)
    {
        pos++;
        if ( pos > mMax )
        {
            return -1;
        }
        else if ( pos == mMax )
        {
            return 0;
        }
        else
        {
            return pos;
        }
    }


    private final int prevPos(int pos)
    {
        if (pos == 0)
            return mMax - 1;
        else
            return pos - 1;
    }
}

