
/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2000 William W. Wong
 *  williamw@jps.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package phex;


import java.io.*;
import java.net.*;

import phex.config.*;
import phex.host.*;
import phex.interfaces.*;
import phex.utils.StatisticTracker;

public class SendWorker implements Runnable
{
    private SendManager	mManager;
    private boolean		mRequestToDie = false;

    private SendWorker()
    {
        // disable
    }

    public SendWorker(SendManager manager)
    {
        mManager = manager;
    }


    public void startup()
    {
        mRequestToDie = false;
        new Thread(this, "SendWorker-" + Integer.toHexString(hashCode())).start();
    }


    public void shutdown()
    {
        mRequestToDie = true;
    }


    public void run()
    {
        FlexBuf		flexBuf = new FlexBuf();
        HostManager	hostMgr = ServiceManager.getHostManager();
        StatisticTracker statisticTracker = ServiceManager.getStatisticTracker();


        while (!mRequestToDie)
        {
            // Sleep and get next message to send.
            HostMsg hostMsg = mManager.getNextHostMsg();

            if (mRequestToDie)
            {
                // I am asked to die.  I can't send the message anymore.
                // Re-queue the msg for another worker to handle.
                mManager.queueMsgToSend(hostMsg);
                break;
            }

            Host host = hostMsg.getHost();
            IMsg msg = hostMsg.getMsg();

            // See if it's dummy message.
            if (host == null || msg == null)
            {
                continue;
            }

            // Test and acquire a lock on the host to send.
            if (!host.acquireSendLockOrQueueMsg(hostMsg))
            {
                // Host is busy.  hostMsg has been placed in its private queue.
                continue;
            }

            try
            {
                OutputStream	os = host.getOs();
                if (os != null)
                {
                    msg.computeHeaderLen();
                    int lenToSend = msg.getSize();
                    byte[] buf = flexBuf.getBuf(lenToSend);
                    msg.serialize(buf, 0);

                    int		lenSent = 0;
                    int		offset = 0;
                    int		len = 0;
                    
                    while (lenSent < lenToSend)
                    {
                        len = lenToSend - lenSent;
                        
                        if (len > 1024)
                            len = 1024;
                        
                        os.write(buf, lenSent, len);
                        
                        lenSent += len;
                        
                        statisticTracker.incBytesCount(len);
                        
                        // Keep track of the fact that we've done some reading.
                        hostMgr.throttleControl( len );
                    }

                    host.incSentCount();
                }
                else
                {
                    // Connection got closed.
//					System.out.println("SendWorker: Connection closed.  Msg discarded.  " + host + ": " + msg);
                }
            }
            catch ( IOException exp )
            {
                host.setStatus(Host.sStatusError, exp.getMessage());
            }
            catch (Exception e)
            {
                e.printStackTrace();
//				host.log("Send error: " + e);
            }
            finally
            {
                host.releaseSendLock();
            }

            // Recyle HostMsg
            mManager.freeHostMsg(hostMsg);
        }

    }

}


