/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.common;

/**
 * This interface is used to define a standard set of method for all kind
 * of classes that provide information of transfered data like UploadFile and
 * DownloadFile.
 */
 // TODO integrate common transfer status for transfer files
public interface TransferDataProvider
{
    public static final short TRANSFER_RUNNING = 10;
    public static final short TRANSFER_NOT_RUNNING = 11;
    public static final short TRANSFER_COMPLETED = 12;
    public static final short TRANSFER_ERROR = 13;

    /**
     * Returns the size of the data that already has been transfered.
     */
    public long getTransferredDataSize();

    /**
     * Returns the length of time, in seconds, that this transfer has
     * been TRANSFER_RUNNING so that one may calculate average
     * transfer rates.
     *
     * WARNING: This method may return 0.
     */
    public long getTransferTimeInSeconds();

    /**
     * Returns the total size of the data that has to be transfered.
     */
    public long getTransferDataSize();

    /**
     * Return the data transfer status.
     * It can be TRANSFER_RUNNING, TRANSFER_NOT_RUNNING, TRANSFER_COMPLETED,
     * TRANSFER_ERROR.
     */
    public short getDataTransferStatus();

    /**
     * Returns the data transfer rate. The rate should depend on the transfer
     * rate timestamp.
     * An implementing class should provide an implementation like this
     * together with the setTransferRateTimestamp() method:
     *
     * <code>
     * public int getDataTransferRate()
     * {
     *    if ( transferRateTimestamp != 0 )
     *    {
     *       double sec = (System.currentTimeMillis() - transferRateTimestamp) / 1000;
     *       // don't drop transfer rate to 0 if we just have a new timestamp and
     *       // no bytes transfered
     *       if ( transferRateBytes > 0 || sec > 1 )
     *       {
     *           transferRate = (int) ( transferRateBytes / sec );
     *       }
     *    }
     *    return transferRate;
     * }
     * </code>
     */    
    public int getDataTransferRate();

    /**
     * To be able to provide a constantly valid data transfer rate the time
     * to calculate the data rate from needs to be updated.
     * If you want your timestamp to be updated regularly you need to register
     * your TransferDataProvider at the TransferRateService.
     * An implementing class should provide an implementation like this
     * together with the getDataTransferRate() method:
     *
     * <code>
     * /**
     *  * holds the timestamp.
     *  *\/
     * private long transferRateTimestamp;
     * /**
     *  * holds the transfered bytes since the last timestamp.
     *  *\/
     * private int transferRateBytes;
     * public void setTransferRateTimestamp( long timestamp )
     * {
     *    transferRateTimestamp = timestamp;
     *    transferRateBytes = 0;
     * }
     * </code>
     */
    public void setTransferRateTimestamp( long timestamp );
}