
/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *                     Mark Saltzman (marks@marksaltzman.com)
 *  Copyright (C) 2000 William W. Wong
 *  williamw@jps.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package phex.connection;


import java.io.*;
import java.net.*;
import java.util.*;

import phex.*;
import phex.host.HostAddress;
import phex.config.*;
import phex.utils.*;


public class ConnectionManager
{
    public ConnectionManager()
    {
    }

    /**
     * @deprecated try to use HostAdress instead
     */
    public Socket connect(String hostPort)
        throws Exception
    {
        return connect( new HostAddress( hostPort ) );
    }

    public Socket connect( HostAddress address )
        throws Exception
    {
        return connect( address.getHostName(), address.getPort() );
    }

    public Socket connect( HostAddress address, int timeout )
        throws IOException
    {
        return connect( address.getHostName(), address.getPort(), timeout );
    }


    public Socket connect(String host, int port)
            throws IOException
    {
        return connect( host, port, 0 );
    }

    /**
     * Opens a socket with a timeout in millis
     */
    public Socket connect( String host, int port, int timeout )
        throws IOException
    {
        if (port < 0 || port > 0xFFFF)
        {
            throw new IOException("Wrong host address (port out of range: "
                + port + " )" );
        }
        if (ServiceManager.sCfg.mProxyUse)
        {
            return connectSock5(host, port);
        }

        SocketFactory factory = new SocketFactory( host, port, timeout );
        return factory.createSocket();
    }

    private Socket connectSock5(String host, int port)
            throws IOException
    {
        Socket sock = null;
        InputStream is = null;
        OutputStream os = null;

        try
        {
            sock = new Socket(ServiceManager.sCfg.mProxyHost, ServiceManager.sCfg.mProxyPort);
            setSocketTimeout( sock, sock.getInetAddress().getAddress() );
            is = sock.getInputStream();
            os = sock.getOutputStream();

            byte[] buf = new byte[600];
            int len;

            buf[0] = (byte)0x05;
            buf[1] = (byte)0x01;
            buf[2] = (byte)0x00;
            len = 3;
            if (ServiceManager.sCfg.mProxyUserName.length() > 0)
            {
                buf[1] = (byte)0x02;
                buf[3] = (byte)0x02;
                len = 4;
            }

            os.write(buf, 0, len);
            int c = is.read();
            if (c != 5)
            {
                StringBuffer buffer = new StringBuffer( );
                buffer.append( (char) c );
                while ( c != -1 )
                {
                    c = is.read();
                    buffer.append( (char) c );
                }
                throw new IOException("Invalid response from Socks5 proxy server: " +
                    buffer.toString() );
            }

            byte method = (byte)is.read();
            if (method == (byte)0xFF)
            {
                throw new IOException("No acceptable authentication method selected by Socks5 proxy server.");
            }
            if (method == 0x00)
            {
            }
            else if (method == 0x02)
            {
                authenticateUserPassword(is, os, buf);
            }
            else
            {
                throw new IOException("Unknown authentication method selected by Socks5 proxy server.");
            }

            len = 0;
            buf[len++] = (byte)0x05;
            buf[len++] = (byte)0x01;
            buf[len++] = (byte)0x00;
            buf[len++] = (byte)0x01;
            len = IOUtil.serializeIP(host, buf, len);
            buf[len++] = (byte)(port >> 8);
            buf[len++] = (byte)(port);
            os.write(buf, 0, len);

            int version = is.read();
            int status = is.read();
            is.read();
            int atype = is.read();

            if (atype == 1)
            {
                is.read();
                is.read();
                is.read();
                is.read();
            }
            else if (atype == 3)
            {
                len = is.read();
                if (len < 0)
                    len += 256;
                while (len > 0)
                {
                    is.read();
                    len--;
                }
            }
            else if (atype == 4)
            {
                for (int i = 0; i < 16; i++)
                    is.read();
            }
            else
            {
                throw new IOException("Invalid return address type for Socks5");
            }

            is.read();
            is.read();

            if (version == 0x05 && status == 0)
            {
                return sock;
            }
            throw new IOException("Connection failed via Socks5 proxy server.  Proxy status=" + status);
        }
        catch (Exception e)
        {
            if (is != null)
            {
                try
                {
                    is.close();
                }
                catch (Exception e2)
                {
                }
            }
            if (os != null)
            {
                try
                {
                    os.close();
                }
                catch (Exception e2)
                {
                }
            }
            if (sock != null)
            {
                try
                {
                    sock.close();
                }
                catch (Exception e2)
                {
                }
            }
            throw new IOException("Proxy server: " + e.getMessage());
        }
    }


    private void authenticateUserPassword(InputStream is, OutputStream os, byte[] buf)
            throws IOException
    {
        int		len = 0;

        buf[len++] = (byte)0x01;
        buf[len++] = (byte)ServiceManager.sCfg.mProxyUserName.length();
        len = IOUtil.serializeString(ServiceManager.sCfg.mProxyUserName, buf, len);
        buf[len++] = (byte)ServiceManager.sCfg.mProxyPassword.length();
        len = IOUtil.serializeString(ServiceManager.sCfg.mProxyPassword, buf, len);
        os.write(buf, 0, len);

        if (is.read() == 1 && is.read() == 0)
        {
            return;
        }

        throw new IOException("Proxy server authentication failed.");
    }

    /**
     * Sets the socket timeout depending on the ip of the host
     * private and invalid IP's get a short time out because they should
     * be reachable very fast in a LAN or are not valid.
     * Unfortunatly this is only for reading not for connecting!
     */
    // TODO check again for jdk 1.4 there are more options to the socket connection
    public void setSocketTimeout( Socket socket, byte[] ip )
        throws IOException
    {
        // TODO imporve this would interrupt readings too soon.
        // the whole read worker should be reworked to make this efficient
        /*if ( IPUtils.isPrivateIP( ip ) || IPUtils.isInvalidIP( ip ) )
        {
            //System.out.println( "short " + new String( (byte)ip[0] + "." + (int)ip[1] +
            //    "." + (int)ip[2] + "." + (int)ip[3] ));
            // set short timeout
            socket.setSoTimeout( ServiceManager.sCfg.privateSocketTimeout );
        }
        else*/
        {
            //System.out.println( "long " + new String( (int)ip[0] + "." + (int)ip[1] +
            //    "." + (int)ip[2] + "." + (int)ip[3] ));
            // set standard timeout
            socket.setSoTimeout( ServiceManager.sCfg.mSocketTimeout );
        }
    }


    private class SocketFactory
    {
        private String host;
        private int port;
        /**
         * Timeout in millis.
         */
        private int timeout;

        /**
         * @param timeout - Timeout in millis.
         */
        public SocketFactory( String aHost, int aPort, int aTimeout )
        {
            host = aHost;
            port = aPort;
            timeout = aTimeout;
        }

        public Socket createSocket()
            throws IOException
        {
            if ( timeout == 0 )
            {
                // TODO this is still the old way of connection to a socket
                // with a timeout of 0
                Socket socket = new Socket( host, port );
                // TODO this setting of timeout method should be reworked and
                // a better integration into the socket factory with timeout should
                // be done.
                setSocketTimeout( socket, socket.getInetAddress().getAddress() );
                return socket;
            }
            OpenSocketThread openSocketThread = new OpenSocketThread( host, port, this );
            openSocketThread.start();
            try
            {
                this.wait( timeout );
            }
            catch (InterruptedException exp)
            {
                openSocketThread.setTimedOut( true );
                throw new IOException( "Timeout while opening Socket" );
            }
            Socket socket = openSocketThread.acquireSocketOrTimeout();
            if ( socket == null )
            {
                throw new IOException( "Timeout while opening Socket" );
            }
            return socket;
        }
    }

    private class OpenSocketThread extends Thread
    {
        private String host;
        private int port;
        private boolean isTimedOut;
        private Socket socket;
        private SocketFactory notifyFactory;
        private IOException connectException;

        /**
         * @param timeout - Timeout in millis.
         * @param factory - The SocketFactory to notify when finished creating socket.
         *        Also used for locking.
         */
        public OpenSocketThread( String aHost, int aPort, SocketFactory factory )
        {
            host = aHost;
            port = aPort;
            notifyFactory = factory;
        }

        public Socket acquireSocketOrTimeout()
            throws IOException
        {
            synchronized ( notifyFactory )
            {
                // throw exception if there was one on connecting
                if ( connectException != null )
                {
                    throw connectException;
                }
                if ( socket != null )
                {
                    return socket;
                }
                setTimedOut( true );
            }
            return null;
        }

        public void setTimedOut( boolean state )
        {
            synchronized ( notifyFactory )
            {
                isTimedOut = state;
                if ( socket == null )
                {
                    return;
                }

                if ( isTimedOut )
                {
                    closeSocket();
                }
            }
        }

        public void run()
        {
            Socket tempSocket = null;
            try
            {
                tempSocket = new Socket( host, port );
            }
            catch ( IOException exp )
            {
                connectException = exp;
            }

            synchronized ( notifyFactory )
            {
                socket = tempSocket;
                if ( isTimedOut )
                {
                    closeSocket();
                }
                else
                {
                    notifyFactory.notify();
                }
            }
        }

        private void closeSocket()
        {
            if ( socket == null )
            {
                return;
            }
            try
            {
                socket.close();
            }
            catch (IOException e)
            { }
        }
    }
}