/*
 *  PHEX - The pure-java Gnutella-servent.
 *  CopyRight (C) 2001 Konrad Haenel ( alterego1@gmx.at )
 *                     Gregor Koukkoullis ( gregor@kouk.de )
 *  Copyright (C) 2000 William W. Wong ( williamw@jps.net )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.download;

import java.io.*;
import java.net.*;
import java.util.*;
import javax.swing.*;

import phex.*;
import phex.config.*;
import phex.connection.*;
import phex.download.*;
import phex.host.*;
import phex.utils.*;


public class DownloadWorker implements Runnable
{
    private DownloadManager mManager;
    private DownloadFile mDownload = null;
    private Host mHost = null;

    private DownloadWorker()
    {
        // disable
    }


    public DownloadWorker( DownloadManager dm )
    {
        mManager = dm;
        new Thread( this, "DownloadWorker-" + Integer.toHexString(hashCode()) ).start();
    }

    /**
     * Closes the connection to the remote host
     */
    public void stopDownload()
    {
        if (mHost != null)
        {
            mHost.disconnect();
        }
    }


    public void run()
    {
        byte[] buf = new byte[1024];
        final int bufSize = buf.length;
        RandomAccessFile rFile = null;
        int len;

        while ( true )
        {
            // Sleep and get the next file to download.
            mDownload = mManager.getNextFileToDownload(this);
            if (mDownload == null)
            {
                ServiceManager.log("DownloadWorker terminated.");
                break;
            }

            String saveFilename = mDownload.getFullLocalFilename();
            String downloadFilename = mDownload.getDownloadName();

            try
            {
                mManager.incDownloadingCount();

                mDownload.appendLog("Start download.");

                // Check for same file name.
                if (!deleteExistingFile(saveFilename))
                {
                    mDownload.setStatus(DownloadFile.sStopped);
                    continue;
                }

                // <filename>.download-<seq>
                File dfile = new File(downloadFilename);
                boolean	append = false;
                int startPos = 0;
                long lngStartPos = 0;

                // Check for existence of previous partial download file.
                if (dfile.exists())
                {
                    append = true;
                    lngStartPos = dfile.length() - 1024; // PH: Why this magic 1024?
                    if ( lngStartPos < 0 )
                    {
                        lngStartPos = 0;
                    }
                    Long tempLong = new Long(lngStartPos);
                    startPos = tempLong.intValue();
                }

                mDownload.appendLog("position to read=" + startPos);

                // Open the file for save.
                try
                {
                    mDownload.appendLog("Download name=" + downloadFilename);
                    rFile = new RandomAccessFile(downloadFilename, "rw");
                }
                catch (Exception e)
                {
                    String		msg = "Failed to create the file '";
                    msg += saveFilename + "' to save the download content.  ";
                    msg += e.getMessage();
                    JOptionPane.showMessageDialog(ServiceManager.getManager().getMainFrame(),
                        msg, "Failed To Create File", JOptionPane.ERROR_MESSAGE);
                    throw e;
                }

                mDownload.appendLog("Connect " + mDownload.getCurrentRemoteFile().getURL());

                // Open connection to remote host.
                URL url = new URL(mDownload.getCurrentRemoteFile().getURL());
                HostAddress hostAddress = new HostAddress(
                    url.getHost(), url.getPort() );
                mHost = new Host( );
                mHost.setType(Host.sTypeDownload);
                try
                {
                    Socket sock = ServiceManager.getConnectionManager().
                        connect( url.getHost(), url.getPort());
                    mDownload.appendLog("Normal connection ok");

                    mHost.setSock(sock);
                    mHost.setOs(sock.getOutputStream());
                    mHost.setIs(sock.getInputStream());
                }
                catch (Exception e)
                {
                    mDownload.appendLog("Normal connection failed.  " + e.getMessage());
                    if ( ServiceManager.sCfg.isBehindFirewall )
                    {
                        // I am behind firewall as well. Push request won't work.
                        mDownload.removeCurrentCandidate();
                        mDownload.appendLog("Download failed.  " + e.toString());
                        throw new IOException("Download failed.  " + e.toString());
                    }

                    mDownload.setStatus(DownloadFile.sError, e.getMessage());
                    Thread.sleep(5000);

                    // Try push request.
                    mDownload.appendLog("Try push request.");

                    // Normal connection failed.  Try PushRequest download.
                    mHost = mManager.pushRequestTransfer(mHost, mDownload);
                    if (mHost == null)
                    {
                        // dont remove candidate here otherwise two candidates
                        // would be removed.
                        throw new IOException("Time out on requesting push transfer.");
                    }
                }

                // User might have stopped download during the long wait during connection.
                if (mDownload.getStatus() == DownloadFile.sStopping ||
                     mDownload.getStatus() == DownloadFile.sRemoved)
                {
                    // Download was stopped between getNextFileToDownload() in above
                    // to here.  Check for it.
                    throw new IOException("Download stopped.");
                }

                // Send download initial handshake.
                String request = mDownload.getCurrentRemoteFile().getGetRequest() +
                    "User-Agent: " + StrUtil.getAppNameVersion() + "\r\n" +
                    mDownload.getRangeHeader((int)startPos) +
                    "\r\n";
                mDownload.appendLog("Send download handshake: " + request);
                len = IOUtil.serializeString(request, buf, 0);

                // not happening anymore
                //try
                {
                    mHost.getOs().write(buf, 0, len);
                }
                /*catch ( NullPointerException exp )
                {
                    // ich glaub die connection ist beim push host nicht mehr verfgbar.
                    System.out.println( "Null " + mHost + "  " + mHost.getOs() + "  "
                    + mHost.getSock() + "  " + buf);
                    throw exp;
                }*/

                InputStream		is = mHost.getIs();
                StringBuffer	strBuf;
                String			str;

                // Look for good response.
                len = IOUtil.readToCRLF(is, buf, bufSize, 0);
                strBuf = new StringBuffer();
                IOUtil.deserializeString(buf, 0, len, strBuf);
                str = strBuf.toString().toUpperCase();
                mDownload.appendLog("Remote host replies: " + str);
                if ( !str.startsWith("HTTP") )
                {
                    mDownload.removeCurrentCandidate();
                    throw new Exception("Invalid response from remote host.");
                }
                else if ( str.indexOf("503") != -1 )
                {
                    mDownload.setStatus( DownloadFile.HOST_BUSY );
                    mDownload.appendLog("Remote host is busy.");
                    continue;
                }
                // if 404 Not Found or 410 Gone then remove host...
                else if ( str.indexOf( "404" ) != -1 || str.indexOf( "410" ) != -1 )
                {
                    mDownload.appendLog("File not available. Host removed.");
                    mDownload.removeCurrentCandidate();
                    continue;
                }
                else if ( (str.indexOf("200 OK") == -1) && (str.indexOf("206 OK") == -1) )
                {
                    mDownload.removeCurrentCandidate();
                    throw new Exception("Negative response to download request.");
                }

                // Look for good header.
                String			server = null;
                boolean			expectedHeader = false;
                while (true)
                {
                    len = IOUtil.readToCRLF(is, buf, bufSize, 0);
                    if (len == 0)
                        break;
                    strBuf = new StringBuffer();
                    IOUtil.deserializeString(buf, 0, len, strBuf);
                    str = strBuf.toString().toUpperCase();
                    mDownload.appendLog(str);

                    if (str.startsWith("CONTENT-LENGTH:"))
                        expectedHeader = true;
                    if (str.startsWith("CONTENT-RANGE:"))
                        expectedHeader = true;
                    if (str.startsWith("SERVER:"))
                        server = str.substring(7).trim();
                }
                mDownload.setRemoteAppName(server);

                if (!expectedHeader)
                {
                    mDownload.removeCurrentCandidate();
                    throw new Exception("Invalid header from remote host.");
                }

                if ( startPos > 0 && !hasServerResumeSupport( server ) )
                {
                    mDownload.removeCurrentCandidate();
                    throw new IOException("Candidate doesn't support resuming.");
                }

                int count = 0; // We have 0 bytes transfered thus far.

                mDownload.appendLog( "Downloading..." );
                mDownload.setStartSize( startPos );
                mDownload.setStartingTime( System.currentTimeMillis() );
                mDownload.setStatus( DownloadFile.sDownloading );

                // We will need to throttle the bandwidth in the download loop.
                ThrottleController bandwidthThrottle = ThrottleController.acquireThrottle();

                try
                {
                    // Download loop. Seek to the starting position in the file.
                    rFile.seek( lngStartPos );



                    // TODO: Make this a final variable rather than calling mDownload.getTransferDataSize()
                    //       all the time for something which won't change.
                    // Loop until we download everything.
                    while ( count < mDownload.getTransferDataSize() )
                    {
                        // Read from our input stream upto the size of our buffer.
                        len = is.read(buf);

                        // Error on read?
                        if (len == -1)
                            break;

                        // Write what we read.
                        rFile.write(buf, 0, len);

                        // Keep track of the fact that we've just downloaded something
                        count += len;
                        mDownload.setTransferredDataSize(count);

                        // Throttle our bandwidth usage appropriately.
                        bandwidthThrottle.setRate( ServiceManager.sCfg.mDownloadMaxBandwidth / mManager.getDownloadingCount() );
                        bandwidthThrottle.controlThrottle( len );
                    }
                }
                catch (Exception e2)
                {
                    // Sometime the close of connection comes as an exception.
                    // Check whether I have finished downloading to decide
                    // whether to throw exception.
                    if ( count < mDownload.getTransferDataSize() )
                    {
                        mDownload.appendLog( "Transfered ended on exception with only " + count + " of "
                                             + mDownload.getTransferDataSize() + " bytes transfered!" );
                        throw e2;
                    }
                }
                finally
                {
                    // Release the throttle correctly.
                    ThrottleController.releaseThrottle( bandwidthThrottle );
                    bandwidthThrottle = null;
                }

                if ( count < mDownload.getTransferDataSize() )
                {
                    mDownload.appendLog("Transfered ended with only " + count + "/" + mDownload.getTransferDataSize() + " transfered!");
                    throw new IOException("Disconnected from remote host");
                }

                mDownload.setTransferredDataSize(count);
                mDownload.setStatus(DownloadFile.sCompleted);
                mDownload.setStoppingTime( System.currentTimeMillis() );
                mDownload.appendLog("Download completed");

                StatisticTracker statTracker = ServiceManager.getStatisticTracker();
                statTracker.incStatDownloadCount(1);
            }
            catch ( Exception e )
            {
                if ( !(e instanceof IOException ) )
                {
                    e.printStackTrace();
                }

                if (mDownload.getStatus() == DownloadFile.sStopping)
                {
                    mDownload.setStatus(DownloadFile.sStopped);
                    mDownload.setStoppingTime( System.currentTimeMillis() );
                }
                else if (mDownload.getStatus() == DownloadFile.sRemoved)
                {
                    // nothing
                }
                else
                {
                    mDownload.setStatus(DownloadFile.sError, e.getMessage());
                    mDownload.setStoppingTime( System.currentTimeMillis() );
                    mDownload.appendLog("Error: " + e);
                }

                // no host = no connection, so we can kick this candidate
                if ( mHost == null )
                {
                    mDownload.removeCurrentCandidate();
                }
            }
            finally
            {
                mManager.decDownloadingCount();

                if (mHost != null)
                {
                    mHost.disconnect();
                    mHost = null;
                }

                if ( rFile != null )
                {
                    try
                    {
                        rFile.close();
                    }
                    catch (IOException e)
                    {
                    }

                    if (mDownload.getStatus() == DownloadFile.sCompleted)
                    {
                        (new File(downloadFilename)).renameTo(new File(saveFilename));
                        mDownload.appendLog("Rename '" + downloadFilename + "' to '" + saveFilename + "'");
                    }
                }
                rFile = null;

                // Remove intermediate file after it has been closed.
                File f = new File(mDownload.getDownloadName());
                if (f.exists() && ( mDownload.getStatus() == DownloadFile.sRemoved ) )
                {
                    f.delete();
                }

                mDownload.setDownloadWorker(null);
                mDownload = null;
            }
        }
    }

    private boolean deleteExistingFile(String saveFilename)
            throws Exception
    {
        File	file = new File(saveFilename);

        if (file.exists())
        {
            int	option = JOptionPane.showConfirmDialog(
                ServiceManager.getManager().getMainFrame(),
                "The file '" + saveFilename + "' exists already.  Overwrite it with the download file?",
                "Confirmation",
                JOptionPane.YES_NO_CANCEL_OPTION,
                JOptionPane.INFORMATION_MESSAGE);

            if (option != JOptionPane.YES_OPTION)
            {
                // Don't proceed.
                return false;
            }
            if (file.delete() == false)
            {
                String	msg = "Failed to delete the old file '";
                msg += saveFilename + ".'";
                JOptionPane.showMessageDialog(ServiceManager.getManager().getMainFrame(),
                                              msg, "Failed To Delete File", JOptionPane.ERROR_MESSAGE);
                throw new Exception("Failed To Delete File");
            }
            mDownload.appendLog("Old file deleted.");
        }

        return true;
    }

    /**
     * Limewire 1.3 dosn't support download resume!
     */
    private boolean hasServerResumeSupport( String server )
    {
        if ( server == null )
        {// unsure what kind of server this is... to be safe return false
            return false;
        }
        int idxID = server.indexOf("LIMEWIRE");
        if ( idxID != -1 )
        {
            int idxDot = server.indexOf(".");
            // find version seperator
            if ( idxDot == -1 )
            {
                // no version separator we assume version smaller then 1.4
                return false;
            }
        }
        return true;
    }
}
