
/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *  Copyright (C) 2000 William W. Wong
 *  williamw@jps.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.download;


import java.io.*;
import java.net.*;
import java.util.*;

import phex.*;
import phex.xml.XMLRemoteFile;

public class RemoteFile implements Serializable
{
    private GUID mRemoteClientID;
    private int mFileIndex;
    private String mFilename;
    private String mShortname;
    private String mFileExt = null;
    private long mFileSize;
    private Long mFileSizeLong;
    private String mRemoteHost;
    private int mSpeed;
    private Integer mSpeedInt;
    private boolean isInDownloadQueue;
    private String localFilename;
    private String searchTerm;

    /**
     * The score defines how well the search term matches the filename.
     */
    private Short score;


    private RemoteFile()
    {
    }

    public RemoteFile( XMLRemoteFile xFile )
        throws Exception
    {
        this( GUID.createFromHexString( xFile.getGuid() ), xFile.getFileIndex(),
            xFile.getFileName(), xFile.getFileSize(), xFile.getRemoteHost(),
            xFile.getSpeed() );
    }

    public RemoteFile(GUID remoteClientID, int fileIndex, String fileName,
        long fileSize, String remoteHost, int speed )
    {
        this( remoteClientID, fileIndex, fileName, fileSize, remoteHost, speed,
            (short)-1 );
    }

    /**
     * @param aScore - The score defines how well the search term matches the
     *                 filename.
     */
    public RemoteFile(GUID remoteClientID, int fileIndex, String fileName,
        long fileSize, String remoteHost, int speed, short aScore )
    {
        mRemoteClientID = remoteClientID;
        mFileIndex = fileIndex;
        mShortname = mFilename = fileName;
        mFileSize = fileSize;
        mFileSizeLong = new Long( fileSize );
        mRemoteHost = remoteHost;
        mSpeed = speed;
        mSpeedInt = new Integer(speed);
        isInDownloadQueue = false;
        score = new Short( aScore );

        // Some servents just return the path along with the filename.
        // Path info is platform-dependent.
        int	i = mFilename.lastIndexOf("/");
        if (i != -1)
        {
            mShortname = mFilename.substring(i + 1);
        }
        else
        {
            i = mFilename.lastIndexOf("\\");
            if (i != -1)
            {
                mShortname = mFilename.substring(i + 1);
            }
        }

    }

    public RemoteFile(RemoteFile b)
    {
        copy( b );
    }

    public void copy(RemoteFile b)
    {
        mRemoteClientID = b.getRemoteClientID();
        mFileIndex = b.getFileIndex();
        mFilename = b.getFilename();
        mShortname = b.getShortname();
        mFileSize = b.getFileSize();
        mFileSizeLong = b.getFileSizeLong();
        mRemoteHost = b.getRemoteHost();
        mSpeed = b.getSpeed();
        mSpeedInt = b.getSpeedInt();
        isInDownloadQueue = b.isInDownloadQueue();
        score = b.getScore();
    }


    public GUID getRemoteClientID()
    {
        return mRemoteClientID;
    }


    public int getFileIndex()
    {
        return mFileIndex;
    }

    public String getFilename()
    {
        return mFilename;
    }

    public String getShortname()
    {
        return mShortname;
    }

    public String getFileExt()
    {
        if (mFileExt != null)
            return mFileExt;

        int		index = mFilename.lastIndexOf(".");

        if (index > -1)
        {
            mFileExt = mFilename.substring(index + 1, mFilename.length());
        }
        else
        {
            mFileExt = "";
        }

        return mFileExt;
    }


    public long getFileSize()
    {
        return mFileSize;
    }


    public Long getFileSizeLong()
    {
        return mFileSizeLong;
    }


    public String getRemoteHost()
    {
        return mRemoteHost;
    }


    public int getSpeed()
    {
        return mSpeed;
    }


    public Integer getSpeedInt()
    {
        return mSpeedInt;
    }


    public boolean isInDownloadQueue()
    {
        return isInDownloadQueue;
    }


    public void setInDownloadQueue(boolean inDownloadQueue)
    {
        isInDownloadQueue = inDownloadQueue;
    }


    public String getURL()
    {
        return "http://" + mRemoteHost + "/get/" + mFileIndex + "/" + mFilename;
    }


    public String getGetRequest()
    {
        // GET /get/1/foo.txt HTTP/1.1\r\n
        return "GET /get/" + mFileIndex + "/" + mFilename + " HTTP/1.0\r\n";

        // Some servents can't handle URL encoded string.
        // return "GET /get/" + mFileIndex + "/" + URLEncoder.encode(mFilename) + " HTTP/1.0\r\n";
    }

    /**
     * The score defines how well the search term matches the filename.
     */
    public Short getScore()
    {
        return score;
    }

    public boolean equals( Object obj )
    {
        if ( !(obj instanceof RemoteFile) )
        {
            return false;
        }
        RemoteFile b = (RemoteFile) obj;
        return mRemoteHost.equals( b.mRemoteHost )
            && mFileIndex == b.mFileIndex;
    }

    public XMLRemoteFile createXMLRemoteFile()
    {
        XMLRemoteFile xfile = new XMLRemoteFile();
        xfile.setFileIndex( mFileIndex );
        xfile.setGuid( mRemoteClientID.toHexString() );
        xfile.setRemoteHost( mRemoteHost );
        xfile.setSpeed( mSpeed );
        xfile.setFileName( mFilename );
        xfile.setFileSize( mFileSize );
        return xfile;
    }

    public void readFromOIS(java.io.DataInputStream dis, boolean firstTime)
            throws IOException, ClassNotFoundException
    {
        if (firstTime)
        {
            dis.skipBytes(244);
        } else {
            dis.skipBytes(5);
        }
        byte[] b={dis.readByte(),dis.readByte(),dis.readByte(),dis.readByte()} ;
        ServiceManager.log("Reading RemoteFile after: "+b[0]+" "+b[1]+" "+b[2]+" "+b[3]);
        int	version = dis.readInt();
        ServiceManager.log("Reading RemoteFile Version: "+version);
        switch (version)
        {
        case 1:
            deserialize1(dis, firstTime);
            break;

            default:
            throw new IOException("Fail to deserialize remotefile.  Unknown verison.");
        }
    }



    private void deserialize1(DataInputStream dis, boolean firstTime)
            throws IOException, ClassNotFoundException
    {
        mRemoteClientID = new GUID();
        mRemoteClientID.readFromOIS(dis, firstTime);
        byte[] b={dis.readByte(),dis.readByte(),dis.readByte()} ;
        if (b[1]== 122)
        {
            ServiceManager.log("found long entry exception");
            dis.skipBytes(3);
        }

        mFileIndex = dis.readInt();
        ServiceManager.log("Reading RemoteFile Index: "+mFileIndex);
        mFilename = dis.readUTF();
        ServiceManager.log("Reading RemoteFile Item: "+mFilename);
        mShortname = dis.readUTF();
        ServiceManager.log("Reading RemoteFile Shortname: "+mShortname);
        mFileSize = dis.readInt();
        ServiceManager.log("Reading RemoteFile FileSize: "+mFileSize);
        if (firstTime)
        {
            dis.skipBytes(79);
        }
        else
        {
            if (dis.readByte() == 113)
            {
                ServiceManager.log("found known FileSize: skipping");
                dis.skipBytes(6);
            }
            else
            {
                dis.skipBytes(11);
            }
        }
        //byte[] b={dis.readByte(),dis.readByte(),dis.readByte(),dis.readByte()} ;
        //ServiceManager.log("Reading RemoteFile after: "+b[0]+" "+b[1]+" "+b[2]+" "+b[3]);
        mRemoteHost = dis.readUTF();
        ServiceManager.log("Reading RemoteFile RemoteHost: "+mRemoteHost);
        mSpeed = dis.readInt();
        ServiceManager.log("Reading RemoteFile Speed: "+mSpeed);
        mSpeedInt = new Integer(mSpeed);
        if (dis.readByte() == 113)
        {
            ServiceManager.log("found known Speed: skipping");
            dis.skipBytes(4);
        }
        else
        {
            dis.skipBytes(9);
        }
    }

    public String toString()
    {
        StringBuffer buffer = new StringBuffer( super.toString() );
        buffer.append( "  " );
        buffer.append( mRemoteHost );
        buffer.append( "  " );
        buffer.append( mFilename );
        buffer.append( "  " );
        buffer.append( mFileIndex );
        return buffer.toString();
    }
}