/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.gui.dialogs.options;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import phex.ServiceManager;
import phex.gui.common.GUIUtils;
import phex.share.FileAdministration;
import phex.utils.Localizer;
import phex.utils.DirectoryOnlyFileFilter;

public class DirectoriesPane extends OptionsSettingsPane
{
    private static final String DIRECTORY_ERROR_KEY = "DirectoryErrorKey";
    private static final String DOWNLOAD_DIRECTORY_KEY = "DownloadDirectory";
    private static final String SHARE_DIRECTORIES_KEY = "ShareDirectories";

    private static final String DOWNLOAD_DIR_SHARED = "DownloadDirShared";
    private static final String NO_DIRECTORY_SHARED = "NoDirectoryShared";
    private static final String CANT_CREATE_DOWNLOAD_DIR = "CantCreateDownloadDir";

    private JTextField downloadDirectoryTF;
    private JList sharedDirectoryList;
    private JCheckBox shareSubDirectoriesChkBx;
    private JTextField includeFilesTF;
    private JTextField excludeFilesTF;

    public DirectoriesPane()
    {
        super( "Directories" );
    }

    /**
     * Called when preparing this settings pane for display the first time. Can
     * be overriden to implement the look of the settings pane.
     */
    protected void prepareComponent()
    {
        GridBagConstraints constraints;
        setLayout( new GridBagLayout() );

        JPanel downloadPanel = new JPanel( new GridBagLayout() );
        downloadPanel.setBorder( BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(),
            Localizer.getString( "DownloadDirectorySettings" ) ) );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.BOTH;
            constraints.weightx = 1;
        add( downloadPanel, constraints );

        JLabel label = new JLabel( Localizer.getString( "DownloadDirectory" ) + ": " );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 0, 0 );
        downloadPanel.add( label, constraints );

        JPanel panel = new JPanel( new GridBagLayout() );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 1;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.weightx = 1;
            constraints.weighty = 1;
        downloadPanel.add( panel, constraints );

        downloadDirectoryTF = new JTextField( ServiceManager.sCfg.mDownloadDir );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.WEST;
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.insets = new Insets( 0, 0, 0, 5 );
            constraints.weightx = 1;
        panel.add( downloadDirectoryTF, constraints );

        JButton button = new JButton( Localizer.getString( "SetDirectory" ) );
        button.addActionListener( new SetDownlodDirectoryListener() );
            constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
        panel.add( button, constraints );

        JPanel uploadPanel = new JPanel( new GridBagLayout() );
        uploadPanel.setBorder( BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(),
            Localizer.getString( "SharingDirectorySettings" ) ) );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 1;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.BOTH;
            constraints.weightx = 1;
            constraints.weighty = 1;
        add( uploadPanel, constraints );

        label = new JLabel( Localizer.getString( "SharedDirectories" ) + ": " );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 0, 0 );
        uploadPanel.add( label, constraints );

        // take the list from the file administration since its already split
        FileAdministration fileAdmin =
            ServiceManager.getShareManager().getFileAdministration();
        Object[] dirs = fileAdmin.getSharedDirectories();
        DefaultListModel model = new DefaultListModel();
        // TODO in JDK 1.3 there is no better way to fill the DefaultListModel :-(
        for ( int i = 0; i < dirs.length; i++ )
        {
            model.addElement( dirs[i] );
        }
        sharedDirectoryList = new JList( model );
        sharedDirectoryList.setVisibleRowCount( 2 );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 1;
            constraints.gridheight = 2;
            constraints.anchor = GridBagConstraints.WEST;
            constraints.fill = GridBagConstraints.BOTH;
            constraints.insets = new Insets( 0, 0, 0, 5 );
            constraints.weightx = 1;
            constraints.weighty = 1;
        uploadPanel.add( new JScrollPane( sharedDirectoryList ), constraints );

        button = new JButton( Localizer.getString( "Add" ) );
        button.addActionListener( new AddUploadDirectoryListener() );
            constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 1;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 5, 0 );
        uploadPanel.add( button, constraints );

        button = new JButton( Localizer.getString( "Remove" ) );
        button.addActionListener( new ActionListener()
            {
                public void actionPerformed( ActionEvent e )
                {
                    int idx = sharedDirectoryList.getSelectedIndex();
                    if ( idx < 0 )
                    {
                        return;
                    }
                    DefaultListModel model = (DefaultListModel)sharedDirectoryList.getModel();
                    model.removeElementAt( idx );
                }
            } );
            constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 2;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
        uploadPanel.add( button, constraints );

        shareSubDirectoriesChkBx = new JCheckBox(
            Localizer.getString( "ShareSubDirectories"),
            ServiceManager.sCfg.mUploadScanRecursively );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 3;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 5, 0);
            constraints.weightx = 1;
        uploadPanel.add( shareSubDirectoriesChkBx, constraints );

        panel = new JPanel( new GridBagLayout() );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 4;
            constraints.gridwidth = 2;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.insets = new Insets( 5, 0, 5, 0);
            constraints.weightx = 1;
        uploadPanel.add( panel, constraints );

        label = new JLabel( Localizer.getString( "IncludeFiles" ) + ": ");
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.WEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 5, 5);
            constraints.weightx = 0;
        panel.add( label, constraints );

        includeFilesTF = new JTextField( ServiceManager.sCfg.mUploadFileInclusions );
            constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.weightx = 1;
        panel.add( includeFilesTF, constraints );

        panel = new JPanel( new GridBagLayout() );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 5;
            constraints.gridwidth = 2;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.insets = new Insets( 0, 0, 5, 0);
            constraints.weightx = 1;
        uploadPanel.add( panel, constraints );

        label = new JLabel( Localizer.getString( "ExcludeFiles" ) + ": ");
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.WEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 5, 5);
            constraints.weightx = 0;
        panel.add( label, constraints );

        excludeFilesTF = new JTextField( ServiceManager.sCfg.mUploadFileExclusions );
            constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.weightx = 1;
        panel.add( excludeFilesTF, constraints );
    }

    /**
     * Override this method if you like to verify inputs before storing them.
     * A input dictionary is given to the pane. It can be used to store values
     * like error flags or prepared values for saving. The dictionary is given
     * to every settings pane checkInput(), displayErrorMessage() and
     * saveAndApplyChanges() method.
     * When the input has been flaged as invalid with the method setInputValid()
     * the method displayErrorMessage() is called directly after return of
     * checkInput() and the focus is given to settings pane.
     * After checking all settings pane without any error the method
     * saveAndApplyChanges() is called for all settings panes to save the
     * changes.
     */
    public void checkInput( HashMap inputDic )
    {
        String downloadDirPath = downloadDirectoryTF.getText();
        File downloadDir = new File( downloadDirPath );

        boolean shareSubDirs = shareSubDirectoriesChkBx.isSelected();
        ListModel model = sharedDirectoryList.getModel();
        int size = model.getSize();
        if ( size == 0 )
        {
            inputDic.put( DIRECTORY_ERROR_KEY, NO_DIRECTORY_SHARED );
            setInputValid( inputDic, false );
            return;
        }

        File[] shareDirs = new File[ size ];
        for ( int i = 0; i < size; i++ )
        {
            String dir = (String)model.getElementAt( i );
            File shareDir = new File( dir );

            // check if is download dir... we don't like sharing download dir...
            boolean isDownloadDirShared = downloadDir.equals( shareDir );
            if ( shareSubDirs && !isDownloadDirShared)
            {
                isDownloadDirShared = isSubDirOf( downloadDir, shareDir );
            }
            // download dir is shared...
            if ( isDownloadDirShared )
            {
                inputDic.put( DIRECTORY_ERROR_KEY, DOWNLOAD_DIR_SHARED );
                setInputValid( inputDic, false );
                return;
            }
            shareDirs[i] = shareDir;
        }

        if ( !downloadDir.exists() || !downloadDir.isDirectory())
        {
            boolean succ = downloadDir.mkdirs();
            if ( !succ )
            {
                inputDic.put( DIRECTORY_ERROR_KEY, CANT_CREATE_DOWNLOAD_DIR );
                setInputValid( inputDic, false );
                return;
            }
        }
        inputDic.put( DOWNLOAD_DIRECTORY_KEY, downloadDir );
        inputDic.put( SHARE_DIRECTORIES_KEY, shareDirs );
        setInputValid( inputDic, true );
    }

    /**
     * When isInputValid() returns a false this method is called.
     * The input dictionary should contain the settings pane specific information
     * of the error.
     * The settings pane should override this method to display a error
     * message. Before calling the method the focus is given to the
     * settings pane.
     */
    public void displayErrorMessage( HashMap inputDic )
    {
        Object error = inputDic.get( DIRECTORY_ERROR_KEY );
        if ( error == null )
        {// no error
            return;
        }

        if ( error.equals( DOWNLOAD_DIR_SHARED ) )
        {
            downloadDirectoryTF.requestFocus();
            downloadDirectoryTF.selectAll();
            GUIUtils.showErrorMessage(
                Localizer.getString( DOWNLOAD_DIR_SHARED ),
                Localizer.getString( "DirectoryError" ) );
        }
        else if ( error.equals( NO_DIRECTORY_SHARED ) )
        {
            GUIUtils.showErrorMessage(
                Localizer.getString( NO_DIRECTORY_SHARED ),
                Localizer.getString( "DirectoryError" ) );
        }
        else if ( error.equals( CANT_CREATE_DOWNLOAD_DIR ) )
        {
            downloadDirectoryTF.requestFocus();
            downloadDirectoryTF.selectAll();
            String[] params = { downloadDirectoryTF.getText() };
            GUIUtils.showErrorMessage(
                Localizer.getFormatedString( CANT_CREATE_DOWNLOAD_DIR, params ),
                Localizer.getString( "DirectoryError" ) );
        }
    }

    /**
     * Override this method if you like to apply and save changes made on
     * settings pane. To trigger saving of the configuration if any value was
     * changed call triggerConfigSave().
     */
    public void saveAndApplyChanges( HashMap inputDic )
    {
        File downloadDir = (File)inputDic.get( DOWNLOAD_DIRECTORY_KEY );
        String downloadDirPath = downloadDir.getAbsolutePath();
        if ( !ServiceManager.sCfg.mDownloadDir.equals( downloadDirPath ) )
        {
            ServiceManager.sCfg.mDownloadDir = downloadDirPath;
            OptionsSettingsPane.triggerConfigSave( inputDic );
        }

        File[] shareDirs = (File[])inputDic.get( SHARE_DIRECTORIES_KEY );
        StringBuffer shareDirsPaths = new StringBuffer();
        for ( int i = 0; i < shareDirs.length; i++ )
        {
            shareDirsPaths.append( shareDirs[i].getAbsolutePath() );
            shareDirsPaths.append( ';' );
        }
        if ( !ServiceManager.sCfg.mUploadDir.equals( shareDirsPaths.toString() ) )
        {
            ServiceManager.sCfg.mUploadDir = shareDirsPaths.toString();
            OptionsSettingsPane.triggerConfigSave( inputDic );
            OptionsSettingsPane.triggerSharedFilesRescan( inputDic );
        }

        String inclusions = includeFilesTF.getText();
        if ( !ServiceManager.sCfg.mUploadFileInclusions.equals( inclusions ) )
        {
            ServiceManager.sCfg.mUploadFileInclusions = inclusions;
            OptionsSettingsPane.triggerConfigSave( inputDic );
            OptionsSettingsPane.triggerSharedFilesRescan( inputDic );
        }

        String exclusions = excludeFilesTF.getText();
        if ( !ServiceManager.sCfg.mUploadFileExclusions.equals( exclusions ) )
        {
            ServiceManager.sCfg.mUploadFileExclusions = exclusions;
            OptionsSettingsPane.triggerConfigSave( inputDic );
            OptionsSettingsPane.triggerSharedFilesRescan( inputDic );
        }
    }

    private boolean isSubDirOf( File subDir, File maybeParentDir )
    {
        // it can't be a sub dir if they don't start the same way...
        if ( !subDir.getAbsolutePath().startsWith( maybeParentDir.getAbsolutePath() ) )
        {
            return false;
        }
        return isSubDirOfInternal( subDir, maybeParentDir );
    }

    private boolean isSubDirOfInternal( File subDir, File maybeParentDir )
    {
        File parent = subDir.getParentFile();
        // no parent dir... cant be sub
        if ( parent == null )
        {
            return false;
        }
        // parent equals we have a sub dir
        if ( parent.equals( maybeParentDir ) )
        {
            return true;
        }
        // go up one level and check again
        return isSubDirOfInternal( parent, maybeParentDir );
    }

    private class AddUploadDirectoryListener implements ActionListener
    {
        public void actionPerformed( ActionEvent e )
        {
            JFileChooser chooser = new JFileChooser();
            DefaultListModel model = (DefaultListModel)sharedDirectoryList.getModel();
            if ( model.getSize() > 0 )
            {
                String path = (String)model.getElementAt( model.getSize() - 1 );
                chooser.setSelectedFile( new File( path ) );
            }
            chooser.setAcceptAllFileFilterUsed( false );
            chooser.setFileFilter( new DirectoryOnlyFileFilter() );
            chooser.setFileSelectionMode( JFileChooser.DIRECTORIES_ONLY );
            chooser.setMultiSelectionEnabled( true );
            chooser.setDialogTitle(
                Localizer.getString( "AddUploadDirectories" ) );
            chooser.setApproveButtonText( Localizer.getString( "Add" ) );
            chooser.setApproveButtonMnemonic(
                Localizer.getChar( "AddDirectoryMnemonic" ) );
            int returnVal = chooser.showDialog( DirectoriesPane.this, null );
            if( returnVal == JFileChooser.APPROVE_OPTION )
            {
                File[] files = chooser.getSelectedFiles();
                for ( int i = 0; i < files.length; i++ )
                {
                    model.addElement( files[i].getAbsolutePath() );
                }
            }
        }
    }

    private class SetDownlodDirectoryListener implements ActionListener
    {
        public void actionPerformed( ActionEvent e )
        {
            JFileChooser chooser = new JFileChooser();
            chooser.setSelectedFile( new File( downloadDirectoryTF.getText() ) );
            chooser.setAcceptAllFileFilterUsed( false );
            chooser.setFileFilter( new DirectoryOnlyFileFilter() );
            chooser.setFileSelectionMode( JFileChooser.DIRECTORIES_ONLY );
            chooser.setMultiSelectionEnabled( false );
            chooser.setDialogTitle(
                Localizer.getString( "SelectDownloadDirectory" ) );
            chooser.setApproveButtonText( Localizer.getString( "Select" ) );
            chooser.setApproveButtonMnemonic(
                Localizer.getChar( "SelectMnemonic" ) );
            int returnVal = chooser.showDialog( DirectoriesPane.this, null );
            if( returnVal == JFileChooser.APPROVE_OPTION )
            {
                String directory = chooser.getSelectedFile().getAbsolutePath();
                downloadDirectoryTF.setText( directory );
            }
        }
    }
}