/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.gui.dialogs.options;

import java.awt.*;
import java.util.*;
import javax.swing.*;
import phex.ServiceManager;
import phex.BaseFrame;
import phex.gui.renderer.LAFListCellRenderer;
import phex.utils.Localizer;

public class GeneralUIPane extends OptionsSettingsPane
{
    private static final String INSTANCIATED_LAF_KEY = "InstanciatedLAF";
    private static final String LAF_ERROR_KEY = "LAFError";

    private static final String LAF_NOT_SUPPORTED = "LAFNotSupported";
    private static final String LAF_NOT_FOUND = "LAFNotFound";
    private static final String LAF_ACCESS_ERROR = "LAFAccessError";
    private static final String LAF_INSTANTIATION_ERROR = "LAFInstantiationError";

    private static final String[] STYLE_LIST = {"Plain", "Bold", "Italic"};
    // initialized in static block
    private static final String[] SIZE_LIST;

    static
    {
        // build size list
        int sizeStart = 6;
        int sizeEnd = 30;
        SIZE_LIST = new String[ sizeEnd - sizeStart + 1];
        for ( int i = 0; i < SIZE_LIST.length; i++ )
        {
            SIZE_LIST[i] = Integer.toString( i + sizeStart );
        }
    }


    private JComboBox lafBox;

    private JComboBox menuFontCoBx;
    private JComboBox labelFontCoBx;
    private JComboBox contentFontCoBx;
    private JComboBox menuStyleCoBx;
    private JComboBox labelStyleCoBx;
    private JComboBox contentStyleCoBx;
    private JComboBox menuSizeCoBx;
    private JComboBox labelSizeCoBx;
    private JComboBox contentSizeCoBx;

    private JCheckBox displayTooltipChkbx;
    private JCheckBox showUpdateNotificationChkbx;


    public GeneralUIPane()
    {
        super( "General" );
    }

    /**
     * Override this method if you like to verify inputs before storing them.
     * A input dictionary is given to the pane. It can be used to store values
     * like error flags or prepared values for saving. The dictionary is given
     * to every settings pane checkInput(), displayErrorMessage() and
     * saveAndApplyChanges() method.
     * When the input has been flaged as invalid with the method setInputValid()
     * the method displayErrorMessage() is called directly after return of
     * checkInput() and the focus is given to settings pane.
     * After checking all settings pane without any error the method
     * saveAndApplyChanges() is called for all settings panes to save the
     * changes.
     */
    public void checkInput( HashMap inputDic )
    {
        // clear the reference to remove last change...
        inputDic.remove( INSTANCIATED_LAF_KEY );
        inputDic.remove( LAF_ERROR_KEY );

        // verify if LAF is supported on plattform.
        UIManager.LookAndFeelInfo lafInfo =
            ( UIManager.LookAndFeelInfo ) lafBox.getSelectedItem();
        try
        {
            Class lnfClass = Class.forName( lafInfo.getClassName() );
            LookAndFeel laf = (LookAndFeel)lnfClass.newInstance();
            if ( !laf.isSupportedLookAndFeel() )
            {
                inputDic.put( LAF_ERROR_KEY, LAF_NOT_SUPPORTED );
                setInputValid( inputDic, false );
                return;
            }
            // this is a valid instance... safe reference to apply it later.
            inputDic.put( INSTANCIATED_LAF_KEY, laf );
        }
        catch ( ClassNotFoundException exp )
        {
            inputDic.put( LAF_ERROR_KEY, LAF_NOT_FOUND );
            setInputValid( inputDic, false );
            return;
        }
        catch ( IllegalAccessException exp )
        {
            inputDic.put( LAF_ERROR_KEY, LAF_ACCESS_ERROR );
            setInputValid( inputDic, false );
            return;
        }
        catch ( InstantiationException exp )
        {
            inputDic.put( LAF_ERROR_KEY, LAF_INSTANTIATION_ERROR );
            setInputValid( inputDic, false );
            return;
        }
        setInputValid( inputDic, true );
    }

    /**
     * When isInputValid() returns a false this method is called.
     * The input dictionary should contain the settings pane specific information
     * of the error.
     * The settings pane should override this method to display a error
     * message. Before calling the method the focus is given to the
     * settings pane.
     */
    public void displayErrorMessage( HashMap inputDic )
    {
        Object value = inputDic.get( LAF_ERROR_KEY );
        if ( value != null )
        {
            JOptionPane.showMessageDialog( this, Localizer.getString( (String)value ),
                Localizer.getString( "LAFError" ), JOptionPane.ERROR_MESSAGE  );
        }
    }

    /**
     * Override this method if you like to apply and save changes made on
     * settings pane. To trigger saving of the configuration if any value was
     * changed call triggerConfigSave().
     */
    public void saveAndApplyChanges( HashMap inputDic )
    {
        // LAF
        LookAndFeel laf = (LookAndFeel) inputDic.get( INSTANCIATED_LAF_KEY );
        if ( !ServiceManager.sCfg.mLFClassName.equals( laf.getClass().getName() ) )
        {
            ServiceManager.sCfg.mLFClassName = laf.getClass().getName();
            BaseFrame.setAllFramesLookAndFeel( laf );
            OptionsSettingsPane.triggerConfigSave( inputDic );
        }

        // FONT
        String name = (String) menuFontCoBx.getSelectedItem();
        int style = menuStyleCoBx.getSelectedIndex();
        int size = Integer.parseInt( (String)menuSizeCoBx.getSelectedItem() );
        Font menuFont = new Font( name, style, size );

        name = (String) labelFontCoBx.getSelectedItem();
        style = labelStyleCoBx.getSelectedIndex();
        size = Integer.parseInt( (String)labelSizeCoBx.getSelectedItem() );
        Font labelFont = new Font( name, style, size );

        name = (String) contentFontCoBx.getSelectedItem();
        style = contentStyleCoBx.getSelectedIndex();
        size = Integer.parseInt( (String)contentSizeCoBx.getSelectedItem() );
        Font tableFont = new Font( name, style, size );

        if ( !ServiceManager.sCfg.mFontMenu.equals( menuFont ) ||
             !ServiceManager.sCfg.mFontLabel.equals( labelFont ) ||
             !ServiceManager.sCfg.mFontTable.equals( tableFont ) )
        {
            ServiceManager.sCfg.mFontMenu = menuFont;
            ServiceManager.sCfg.mFontLabel = labelFont;
            ServiceManager.sCfg.mFontTable = tableFont;
            ServiceManager.getManager().getMainFrame().setFonts();
            OptionsSettingsPane.triggerConfigSave( inputDic );
        }

        // TOOLTIP
        boolean tooltip = displayTooltipChkbx.isSelected();
        if ( ServiceManager.sCfg.mUIDisplayTooltip != tooltip )
        {
            ServiceManager.sCfg.mUIDisplayTooltip = tooltip;
            OptionsSettingsPane.triggerConfigSave( inputDic );
        }

        // update notify
        boolean updateNotify = showUpdateNotificationChkbx.isSelected();
        if ( ServiceManager.sCfg.showUpdateNotification != updateNotify )
        {
            ServiceManager.sCfg.showUpdateNotification = updateNotify;
            OptionsSettingsPane.triggerConfigSave( inputDic );
        }
    }

    /**
     * Called when preparing this settings pane for display the first time. Can
     * be overriden to implement the look of the settings pane.
     */
    protected void prepareComponent()
    {
        GridBagConstraints constraints;
        setLayout( new GridBagLayout() );

        setBorder( BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(),
            Localizer.getString( "GeneralUserInterfaceSettings" ) ) );

        // L&F
        JLabel label = new JLabel( Localizer.getString( "ConfigureLAF" ) );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.weightx = 1;
        add( label, constraints );

        UIManager.LookAndFeelInfo[] lafs = UIManager.getInstalledLookAndFeels();
        lafBox = new JComboBox( lafs );
        int currentLAFIndex = determineCurrentLAFIndex( lafs );
        lafBox.setSelectedIndex( currentLAFIndex );
        lafBox.setRenderer( new LAFListCellRenderer() );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 1;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.weightx = 1;
        add( lafBox, constraints );

        // FONT
        String[] fontList = GraphicsEnvironment.getLocalGraphicsEnvironment().
            getAvailableFontFamilyNames();
        Font font;

        JPanel fontPanel = new JPanel();
        fontPanel.setLayout( new GridBagLayout() );
            constraints.gridx = 0;
            constraints.gridy = 2;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 10, 0, 0, 0 );
            constraints.weightx = 1;
        add( fontPanel, constraints );

        label = new JLabel( Localizer.getString( "MenuFont" ) );
        constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.weightx = 1;
        fontPanel.add( label, constraints );

        label = new JLabel( Localizer.getString( "Style" ) );
        constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.weightx = 1;
        fontPanel.add( label, constraints );

        label = new JLabel( Localizer.getString( "Size" ) );
        constraints = new GridBagConstraints();
            constraints.gridx = 2;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.weightx = 1;
        fontPanel.add( label, constraints );

        font = ServiceManager.sCfg.mFontMenu;
        menuFontCoBx = new JComboBox( fontList );
        menuFontCoBx.setSelectedItem( font.getName() );
        constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 1;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 2, 5 );
            constraints.weightx = 1;
            constraints.weighty = 1;
        fontPanel.add( menuFontCoBx, constraints );

        menuStyleCoBx = new JComboBox( STYLE_LIST );
        menuStyleCoBx.setSelectedIndex( font.getStyle() );
        constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 1;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 0, 5 );
            constraints.weightx = 1;
        fontPanel.add( menuStyleCoBx, constraints );

        menuSizeCoBx = new JComboBox( SIZE_LIST );
        menuSizeCoBx.setSelectedItem( Integer.toString( font.getSize() ) );
        constraints = new GridBagConstraints();
            constraints.gridx = 2;
            constraints.gridy = 1;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.weightx = 1;
        fontPanel.add( menuSizeCoBx, constraints );


        label = new JLabel( Localizer.getString( "LabelFont" ) );
        constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 2;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.weightx = 1;
        fontPanel.add( label, constraints );

        label = new JLabel( Localizer.getString( "Style" ) );
        constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 2;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.weightx = 1;
        fontPanel.add( label, constraints );

        label = new JLabel( Localizer.getString( "Size" ) );
        constraints = new GridBagConstraints();
            constraints.gridx = 2;
            constraints.gridy = 2;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.weightx = 1;
        fontPanel.add( label, constraints );

        font = ServiceManager.sCfg.mFontLabel;
        labelFontCoBx = new JComboBox( fontList );
        labelFontCoBx.setSelectedItem( font.getName() );
        constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 3;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 2, 5 );
            constraints.weightx = 1;
            constraints.weighty = 1;
        fontPanel.add( labelFontCoBx, constraints );

        labelStyleCoBx = new JComboBox( STYLE_LIST );
        labelStyleCoBx.setSelectedIndex( font.getStyle() );
        constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 3;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 0, 5 );
            constraints.weightx = 1;
        fontPanel.add( labelStyleCoBx, constraints );

        labelSizeCoBx = new JComboBox( SIZE_LIST );
        labelSizeCoBx.setSelectedItem( Integer.toString( font.getSize() ) );
        constraints = new GridBagConstraints();
            constraints.gridx = 2;
            constraints.gridy = 3;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.weightx = 1;
        fontPanel.add( labelSizeCoBx, constraints );


        label = new JLabel( Localizer.getString( "ContentFont" ) );
        constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 4;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.weightx = 1;
        fontPanel.add( label, constraints );

        label = new JLabel( Localizer.getString( "Style" ) );
        constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 4;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.weightx = 1;
        fontPanel.add( label, constraints );

        label = new JLabel( Localizer.getString( "Size" ) );
        constraints = new GridBagConstraints();
            constraints.gridx = 2;
            constraints.gridy = 4;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.weightx = 1;
        fontPanel.add( label, constraints );

        font = ServiceManager.sCfg.mFontTable;
        contentFontCoBx = new JComboBox( fontList );
        contentFontCoBx.setSelectedItem( font.getName() );
        constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 5;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 2, 5 );
            constraints.weightx = 1;
            constraints.weighty = 1;
        fontPanel.add( contentFontCoBx, constraints );

        contentStyleCoBx = new JComboBox( STYLE_LIST );
        contentStyleCoBx.setSelectedIndex( font.getStyle() );
        constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 5;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 0, 5 );
            constraints.weightx = 1;
        fontPanel.add( contentStyleCoBx, constraints );

        contentSizeCoBx = new JComboBox( SIZE_LIST );
        contentSizeCoBx.setSelectedItem( Integer.toString( font.getSize() ) );
        constraints = new GridBagConstraints();
            constraints.gridx = 2;
            constraints.gridy = 5;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.weightx = 1;
        fontPanel.add( contentSizeCoBx, constraints );


        // tooltip
        displayTooltipChkbx = new JCheckBox(
            Localizer.getString( "DisplayTooltipText" ),
            ServiceManager.sCfg.mUIDisplayTooltip );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 3;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 10, 0, 0, 0 );
            constraints.weightx = 1;
        add( displayTooltipChkbx, constraints );

        showUpdateNotificationChkbx = new JCheckBox(
            Localizer.getString( "ShowUpdateNotification" ),
            ServiceManager.sCfg.showUpdateNotification );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 4;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 10, 0, 0, 0 );
            constraints.weightx = 1;
            constraints.weighty = 1;
        add( showUpdateNotificationChkbx, constraints );
    }

    private int determineCurrentLAFIndex( UIManager.LookAndFeelInfo[] lafs )
    {
        LookAndFeel laf = UIManager.getLookAndFeel();
        String lafClassName = laf.getClass().getName();
        for ( int i = 0; i < lafs.length; i++ )
        {
            if ( lafClassName.equals( lafs[i].getClassName() ) )
            {
                return i;
            }
        }
        return 0;
    }
}