/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.gui.dialogs.options;

import java.awt.*;
import javax.swing.*;
import java.util.HashMap;
import phex.ServiceManager;
import phex.gui.common.IntegerTextField;
import phex.utils.Localizer;

public class SharingPane extends OptionsSettingsPane
{
    private static final String SIMULTANEOUS_UPLOADS_KEY = "SimultaneousUploads";
    private static final String UPLOADS_PER_HOST_KEY = "UploadsPerHost";
    private static final String RETURNED_SEARCH_LIMIT_KEY = "ReturnedSearchLimit";

    private IntegerTextField simultaneousUploadsTF;
    private IntegerTextField uploadsPerHostTF;
    private IntegerTextField returnedSearchLimitTF;
    private JCheckBox removeCompletedUploadsChkbx;
    private JCheckBox allowBrowsingChkbx;

    public SharingPane()
    {
        super( "Sharing" );
    }

    /**
     * Called when preparing this settings pane for display the first time. Can
     * be overriden to implement the look of the settings pane.
     */
    protected void prepareComponent()
    {
        GridBagConstraints constraints;
        setLayout( new GridBagLayout() );

        JPanel generalPanel = new JPanel( new GridBagLayout() );
        generalPanel.setBorder( BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(),
            Localizer.getString( "GeneralUploadSettings" ) ) );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.BOTH;
            constraints.weightx = 1;
            constraints.weighty = 1;
        add( generalPanel, constraints );

        JPanel panel = new JPanel( new GridBagLayout() );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 3, 0, 5, 0 );
            constraints.weightx = 1;
        generalPanel.add( panel, constraints );

        JLabel label = new JLabel( Localizer.getString( "SimultaneousUploads" ) + ": " );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 0, 5 );
        panel.add( label, constraints );

        simultaneousUploadsTF = new IntegerTextField(
            String.valueOf( ServiceManager.sCfg.mMaxUpload ), 3, 3 );
            constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
        panel.add( simultaneousUploadsTF, constraints );

        panel = new JPanel( new GridBagLayout() );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 1;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 5, 0 );
            constraints.weightx = 1;
        generalPanel.add( panel, constraints );

        label = new JLabel( Localizer.getString( "UploadsPerHost" ) + ": ");
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 0, 5 );
        panel.add( label, constraints );

        uploadsPerHostTF = new IntegerTextField(
            String.valueOf( ServiceManager.sCfg.mMaxUploadPerIP ), 3, 3 );
            constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
        panel.add( uploadsPerHostTF, constraints );

        panel = new JPanel( new GridBagLayout() );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 2;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 5, 0 );
            constraints.weightx = 1;
        generalPanel.add( panel, constraints );

        label = new JLabel( Localizer.getString( "ReturnedSearchLimit" ) + ": ");
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 0, 5 );
        panel.add( label, constraints );

        returnedSearchLimitTF = new IntegerTextField(
            String.valueOf( ServiceManager.sCfg.mUploadMaxSearch ), 4, 4 );
            constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
        panel.add( returnedSearchLimitTF, constraints );

        removeCompletedUploadsChkbx = new JCheckBox(
            Localizer.getString( "AutoRemoveCompletedUploads" ),
            ServiceManager.sCfg.mUploadAutoRemoveCompleted );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 3;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 5, 0 );
            constraints.weightx = 1;
        generalPanel.add( removeCompletedUploadsChkbx, constraints );

        allowBrowsingChkbx = new JCheckBox(
            Localizer.getString( "AllowBrowsingDirectories" ),
            ServiceManager.sCfg.mShareBrowseDir );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 4;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.NONE;
            constraints.insets = new Insets( 0, 0, 5, 0 );
            constraints.weightx = 1;
            constraints.weighty = 1;
        generalPanel.add( allowBrowsingChkbx, constraints );
    }

    /**
     * Override this method if you like to verify inputs before storing them.
     * A input dictionary is given to the pane. It can be used to store values
     * like error flags or prepared values for saving. The dictionary is given
     * to every settings pane checkInput(), displayErrorMessage() and
     * saveAndApplyChanges() method.
     * When the input has been flaged as invalid with the method setInputValid()
     * the method displayErrorMessage() is called directly after return of
     * checkInput() and the focus is given to settings pane.
     * After checking all settings pane without any error the method
     * saveAndApplyChanges() is called for all settings panes to save the
     * changes.
     */
    public void checkInput( HashMap inputDic )
    {
        try
        {
            String simultaneousUploadsStr = simultaneousUploadsTF.getText();
            Integer simultaneousUploads = new Integer( simultaneousUploadsStr );
            inputDic.put( SIMULTANEOUS_UPLOADS_KEY, simultaneousUploads );
        }
        catch ( NumberFormatException exp )
        {
            inputDic.put( NUMBER_FORMAT_ERROR_KEY, simultaneousUploadsTF );
            setInputValid( inputDic, false );
            return;
        }

        try
        {
            String uploadsPerHostStr = uploadsPerHostTF.getText();
            Integer uploadsPerHost = new Integer( uploadsPerHostStr );
            inputDic.put( UPLOADS_PER_HOST_KEY, uploadsPerHost );
        }
        catch ( NumberFormatException exp )
        {
            inputDic.put( NUMBER_FORMAT_ERROR_KEY, uploadsPerHostTF );
            setInputValid( inputDic, false );
            return;
        }

        try
        {
            String returnedSearchLimitStr = returnedSearchLimitTF.getText();
            Integer returnedSearchLimit = new Integer( returnedSearchLimitStr );
            inputDic.put( RETURNED_SEARCH_LIMIT_KEY, returnedSearchLimit );
        }
        catch ( NumberFormatException exp )
        {
            inputDic.put( NUMBER_FORMAT_ERROR_KEY, returnedSearchLimitTF );
            setInputValid( inputDic, false );
            return;
        }

        setInputValid( inputDic, true );
    }

    /**
     * When isInputValid() returns a false this method is called.
     * The input dictionary should contain the settings pane specific information
     * of the error.
     * The settings pane should override this method to display a error
     * message. Before calling the method the focus is given to the
     * settings pane.
     */
    public void displayErrorMessage( HashMap inputDic )
    {
        if ( inputDic.containsKey( NUMBER_FORMAT_ERROR_KEY ) )
        {
            displayNumberFormatError( inputDic );
        }
    }

    /**
     * Override this method if you like to apply and save changes made on
     * settings pane. To trigger saving of the configuration if any value was
     * changed call triggerConfigSave().
     */
    public void saveAndApplyChanges( HashMap inputDic )
    {
        Integer simultaneousUploadsInt = (Integer) inputDic.get(
            SIMULTANEOUS_UPLOADS_KEY );
        int simultaneousUploads = simultaneousUploadsInt.intValue();
        if ( ServiceManager.sCfg.mMaxUpload != simultaneousUploads )
        {
            ServiceManager.sCfg.mMaxUpload = simultaneousUploads;
            OptionsSettingsPane.triggerConfigSave( inputDic );
        }

        Integer uploadsPerHostInt = (Integer) inputDic.get(
            UPLOADS_PER_HOST_KEY );
        int uploadsPerHost = uploadsPerHostInt.intValue();
        if ( ServiceManager.sCfg.mMaxUploadPerIP != uploadsPerHost )
        {
            ServiceManager.sCfg.mMaxUploadPerIP = uploadsPerHost;
            OptionsSettingsPane.triggerConfigSave( inputDic );
        }

        Integer returnedSearchLimitInt = (Integer) inputDic.get(
            RETURNED_SEARCH_LIMIT_KEY );
        int returnedSearchLimit = returnedSearchLimitInt.intValue();
        if ( ServiceManager.sCfg.mUploadMaxSearch != returnedSearchLimit )
        {
            ServiceManager.sCfg.mUploadMaxSearch = returnedSearchLimit;
            OptionsSettingsPane.triggerConfigSave( inputDic );
        }

        boolean removeCompletedUploads = removeCompletedUploadsChkbx.isSelected();
        if ( ServiceManager.sCfg.mUploadAutoRemoveCompleted !=
             removeCompletedUploads )
        {
            ServiceManager.sCfg.mUploadAutoRemoveCompleted = removeCompletedUploads;
            OptionsSettingsPane.triggerConfigSave( inputDic );
        }

        boolean allowBrowsing = allowBrowsingChkbx.isSelected();
        if ( ServiceManager.sCfg.mShareBrowseDir != allowBrowsing )
        {
            ServiceManager.sCfg.mShareBrowseDir = allowBrowsing;
            OptionsSettingsPane.triggerConfigSave( inputDic );
        }
    }
}