/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.gui.models;

import java.util.*;
import javax.swing.table.*;

import phex.ServiceManager;
import phex.download.DownloadManager;
import phex.download.DownloadFile;
import phex.download.RemoteFile;
import phex.event.DownloadCandidatesChangeListener;
import phex.gui.tabs.DownloadTab;
import phex.gui.renderer.*;

public class DownloadCandidateTableModel extends AbstractTableModel
{
    private static final int numColSharingHost = 0;
    private static final int numColFile = 1;
    private static final int numColSize = 2;
    private static final int numColHostSpeed = 3;
    private static final int numColInUse = 4;
    
    private static String[] tableColumns =
    {
        "Sharing Host",
        "File",
        "Size",
        "Host Speed",
        "In Use"
    };
    private DownloadFile downloadFile;
    private DownloadCandidatesListener changeListener;

    public DownloadCandidateTableModel( )
    {
        changeListener = new DownloadCandidatesListener();
    }

    public void setCurrentDownloadFile( DownloadFile file )
    {
        if ( downloadFile != null )
        {
            downloadFile.removeDownloadCandidatesChangeListener( changeListener );
        }
        downloadFile = file;
        if ( downloadFile != null )
        {
            downloadFile.addDownloadCandidatesChangeListener( changeListener );
        }
        fireTableDataChanged();
    }

    public String getColumnName(int col)
    {
        return tableColumns[ col ];
    }

    public int getColumnCount()
    {
        return tableColumns.length;
    }

    public int getRowCount()
    {
        if ( downloadFile == null )
        {
            return 0;
        }
        return downloadFile.getRemoteCandidatesCount();
    }


    public Object getValueAt(int row, int col)
    {
        RemoteFile remoteFile = downloadFile.getRemoteCandidateAt( row );
        if ( remoteFile == null )
        {
            fireTableRowsDeleted( row, row );
            return "";
        }

        switch (col)
        {
            case numColSharingHost:
                return remoteFile.getRemoteHost();
                
            case numColFile:
                return remoteFile.getShortname();
                
            case numColSize:
                return remoteFile.getFileSizeLong();
                
            case numColHostSpeed:
                return remoteFile.getSpeedInt();
                
            case numColInUse:
                return (downloadFile.getCurrentCandidate() == row ? "Current" : "");
        }

        return "";
    }


    public void setValueAt(Object value, int row, int col)
    {
        // Do nothing
    }


    public boolean isCellEditable(int row, int col)
    {
        return false;
    }


    public Class getColumnClass(int col)
    {
        switch ( col )
        {
            case numColSharingHost:
                return IPCellRenderer.class;
                
            case numColFile:
                return String.class;
                
            case numColSize:
                return Long.class;
                
            case numColHostSpeed:
                return Integer.class;
                
            case numColInUse:
                return String.class;
        }
        return String.class;
    }

    private class DownloadCandidatesListener
        implements DownloadCandidatesChangeListener
    {
        /**
         * Called if a download file changed.
         */
        public void downloadCandidateChanged( int position )
        {
            fireTableRowsUpdated( position, position );
        }

        /**
         * Called if a download file was added.
         */
        public void downloadCandidateAdded( int position )
        {
            fireTableRowsInserted( position, position );
        }

        /**
         * Called if a download file was removed.
         */
        public void downloadCandidateRemoved( int position )
        {
            fireTableRowsDeleted( position, position );
        }
    }
}