/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package phex.host;

import java.io.*;
import java.net.*;
import java.util.*;
import javax.swing.SwingUtilities;
import phex.Listener;
import phex.event.CaughtHostsChangeListener;
import phex.ServiceManager;
import phex.utils.IPUtils;

/**
 * Responsible for holding all caught hosts.
 */
public class CaughtHostsContainer
{
    private static final String[] DEFAULT_STARTUP_HOSTS =
        {
            "connect1.gnutellanet.com:6346",
            "connect2.gnutellanet.com:6346",
            "connect3.gnutellanet.com:6346",
            "gnutella-again.hostscache.com:6346",
            "gnutella.hostscache.com:6346",
            "gnotella.fileflash.com:6346",
            "public.bearshare.net:6346"

            //"gnutellahosts.com:6346",
            // out of order
            //"router.limewire.com:6346",
        };

    public static final short HIGH_PRIORITY = 0;
    public static final short LOW_PRIORITY = 1;

    private Listener listener;
    private ArrayList caughtHosts;
    private ArrayList autoConnectHosts;
    private HashMap uniqueCaughtHosts;
    private boolean hasChangedSinceLastSave;
    //private int currentHostIndex;

    /**
     * All listeners interested in events.
     */
    private ArrayList listenerList = new ArrayList( 2 );

    public CaughtHostsContainer()
    {
        listener = ServiceManager.getListener();
        caughtHosts = new ArrayList();
        autoConnectHosts = new ArrayList();
        uniqueCaughtHosts = new HashMap();
    }

    public void initializeCaughtHostsContainer()
    {
        removeAllCaughtHosts();
        removeAllAutoConnectHosts();
        hasChangedSinceLastSave = false;
        //currentHostIndex = 0;
        loadHostsFromFile();
        loadAutoConnectHostsFromFile();
    }

    public synchronized String getNextCaughtHost()
    {
        String host = getCaughtHostAt( 0/*currentHostIndex*/ );
        if ( host == null )
        {
            //if ( currentHostIndex == 0 )
            {
                // host list is empty
                addAutoConnectHosts();
            }
            //currentHostIndex = 0;
            host = getCaughtHostAt( 0/*currentHostIndex*/ );
        }
        removeCaughtHost( host );
        //currentHostIndex ++;
        return host;
    }

    public synchronized int getCaughtHostsCount()
    {
        return caughtHosts.size();
    }

    public synchronized int getAutoConnectHostsCount()
    {
        return autoConnectHosts.size();
    }

    public synchronized String getCaughtHostAt( int index )
    {
        if ( index >= caughtHosts.size() )
        {
            return null;
        }
        return (String) caughtHosts.get( index );
    }

    public synchronized String getAutoConnectHostAt( int index )
    {
        if ( index >= autoConnectHosts.size() )
        {
            return null;
        }
        return (String) autoConnectHosts.get( index );
    }

    public synchronized String[] getCaughtHostsAt( int[] indices )
    {
        int length = indices.length;
        String[] hosts = new String[ length ];
        for ( int i = 0; i < length; i++ )
        {
            hosts[i] = (String)caughtHosts.get( indices[i] );
        }
        return hosts;
    }

    /**
     * Removes all caught hosts and initializes them with the startup hosts.
     */
    public synchronized void resetCaughtHosts()
    {
        removeAllCaughtHosts();
        addAutoConnectHosts();
    }

    public synchronized void removeAllCaughtHosts()
    {
        for ( int i = caughtHosts.size() - 1; i >= 0; i-- )
        {
            removeCaughtHost( (String) caughtHosts.get( i ) );
        }
    }

    public synchronized void removeAllAutoConnectHosts()
    {
        for ( int i = autoConnectHosts.size() - 1; i >= 0; i-- )
        {
            removeAutoConnectHost( (String) autoConnectHosts.get( i ) );
        }
    }

    /**
     * Adds a caught host with given priority.
     */
    public synchronized void addCaughtHost( HostAddress address, short priority )
    {
        if ( address.isLocalHost() )
        {
            return;
        }
        try
        {
            if ( IPUtils.isInvalidIP( address.getHostIP() ) )
            {
                return;
            }
        }
        catch ( UnknownHostException exp )
        {
            // this actually could mean that we have no connection to the internet...
            return;
        }
        // TODO integrate HostAddress object handling...
        insertCaughtHost( address.getHostName() + ":" + address.getPort(), priority );
    }

    /**
     * Adds a auto connect host.
     */
    public synchronized void addAutoConnectHost( String host )
    {
        insertAutoConnectHost( host, autoConnectHosts.size() );
    }

    /**
     * inserts a host and fires the add event..
     */
    private synchronized void insertCaughtHost( String host, short priority )
    {
        if ( caughtHosts.size() > ServiceManager.sCfg.mNetMaxHostToCatch)
        {
            // we remove depending on the priority
            if ( priority == HIGH_PRIORITY )
            {
                // we remove from bottom... not nice... they might be new
                // but actually happens very rarely
                // TODO find better collection model.
                removeCaughtHost( getCaughtHostAt( caughtHosts.size() - 1 ) );
            }
            // all other prioritys
            else
            {
                removeCaughtHost( getCaughtHostAt( 0 ) );
            }
        }

        int position;
        if ( priority == HIGH_PRIORITY )
        {
            position = 0;
        }
        // all other prioritys
        else
        {
            position = caughtHosts.size();
        }

        // if host is not already in the list
        if ( uniqueCaughtHosts.get( host ) == null)
        {
            caughtHosts.add( position, host );
            uniqueCaughtHosts.put( host, host );
            hasChangedSinceLastSave = true;
            fireCaughtHostAdded( position );
        }
    }

    /**
     * inserts a auto connect host and fires the add event..
     */
    private synchronized void insertAutoConnectHost( String host, int position )
    {
        // if host is not already in the list
        if ( !autoConnectHosts.contains( host ) )
        {
            autoConnectHosts.add( position, host );
            hasChangedSinceLastSave = true;
            fireAutoConnectHostAdded( position );
        }
    }

    public synchronized void removeCaughtHost( String host )
    {
        int position = caughtHosts.indexOf( host );
        if ( position >= 0 )
        {
            caughtHosts.remove( position );
            uniqueCaughtHosts.remove( host );
            fireCaughtHostRemoved( position );
            hasChangedSinceLastSave = true;
        }
    }

    public synchronized void removeAutoConnectHost( String host )
    {
        int position = autoConnectHosts.indexOf( host );
        if ( position >= 0 )
        {
            autoConnectHosts.remove( position );
            fireAutoConnectHostRemoved( position );
            hasChangedSinceLastSave = true;
        }
    }

    public synchronized void saveHostsContainer( )
    {
        if ( !hasChangedSinceLastSave )
        {
            return;
        }
        saveCaughtHosts();
        saveAutoConnectHosts();
        hasChangedSinceLastSave = false;
    }

    private synchronized void addAutoConnectHosts()
    {
        if (ServiceManager.sCfg.mCurrentNetwork.equals(ServiceManager.sCfg.sGeneralNetwork))
        {
            int size = autoConnectHosts.size();
            for ( int i = size - 1; i >= 0; i-- )
            {
                insertCaughtHost( (String)autoConnectHosts.get( i ), HIGH_PRIORITY );
            }
            if ( size == 0 )
            {
                addDefaultAutoConnectHosts();
            }
        }
    }

    private synchronized void addDefaultAutoConnectHosts()
    {
        if (ServiceManager.sCfg.mCurrentNetwork.equals(ServiceManager.sCfg.sGeneralNetwork))
        {
            for (int i = DEFAULT_STARTUP_HOSTS.length - 1; i >= 0 ; i--)
            {
                insertCaughtHost( DEFAULT_STARTUP_HOSTS[i], HIGH_PRIORITY );
            }
        }
    }

    private void copyDefaultAutoConnectHosts()
    {
        for ( int i = 0; i < DEFAULT_STARTUP_HOSTS.length; i++ )
        {
            addAutoConnectHost( DEFAULT_STARTUP_HOSTS[i] );
        }
    }

    private void loadHostsFromFile()
    {
        try
        {
            File file = new File(ServiceManager.getHostsFilename());
            BufferedReader br = new BufferedReader(new FileReader(file));
            String line;

            while ( (line = br.readLine()) != null)
            {
                if ( line.startsWith("#") )
                {
                    continue;
                }
                insertCaughtHost( line, LOW_PRIORITY );
            }
            br.close();
        }
        catch ( IOException exp )
        {
        }
    }

    private void loadAutoConnectHostsFromFile()
    {
        try
        {
            File file = new File(ServiceManager.getAutoConnectHostsFilename());
            if ( !file.exists() )
            {
                copyDefaultAutoConnectHosts();
            }
            BufferedReader br = new BufferedReader( new FileReader( file ) );
            String line;

            while ( (line = br.readLine()) != null)
            {
                if ( line.startsWith("#") )
                {
                    continue;
                }
                insertAutoConnectHost( line, autoConnectHosts.size() );
            }
            br.close();
        }
        catch ( IOException exp )
        {
        }
    }

    private synchronized void saveCaughtHosts()
    {
        try
        {
            File file = new File(ServiceManager.getHostsFilename());
            BufferedWriter bw = new BufferedWriter(new FileWriter(file));

            for ( int i = 0; i < caughtHosts.size(); i++ )
            {
                String host = (String)caughtHosts.get(i);
                if ( !isInStartupHosts( host ) )
                {
                    bw.write(host);
                    bw.newLine();
                }
            }
            bw.close();
        }
        catch ( IOException exp )
        {
            exp.printStackTrace();
        }
    }

    private synchronized void saveAutoConnectHosts()
    {
        try
        {
            File file = new File(ServiceManager.getAutoConnectHostsFilename());
            BufferedWriter bw = new BufferedWriter(new FileWriter(file));

            for ( int i = 0; i < autoConnectHosts.size(); i++ )
            {
                String host = (String)autoConnectHosts.get(i);
                bw.write(host);
                bw.newLine();
            }
            bw.close();
        }
        catch ( IOException exp )
        {
            exp.printStackTrace();
        }
    }

    private boolean isInStartupHosts( String host )
    {
        for ( int i = 0; i < DEFAULT_STARTUP_HOSTS.length; i++ )
        {
            if ( host.equals( DEFAULT_STARTUP_HOSTS[i] ) )
            {
                return true;
            }
        }
        int length = autoConnectHosts.size();
        for ( int i = 0; i < length; i++ )
        {
            if ( host.equals( autoConnectHosts.get( i ) ) )
            {
                return true;
            }
        }
        return false;
    }

    ///////////////////// START event handling methods ////////////////////////
    public void addCaughtHostsChangeListener( CaughtHostsChangeListener listener )
    {
        listenerList.add( listener );
    }

    public void removeCaughtHostsChangeListener( CaughtHostsChangeListener listener )
    {
        listenerList.remove( listener );
    }

    private void fireCaughtHostAdded( final int position )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                CaughtHostsChangeListener listener;
                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (CaughtHostsChangeListener)listeners[ i ];
                    listener.caughtHostAdded( position );
                }
            }
        });
    }

    private void fireAutoConnectHostAdded( final int position )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                CaughtHostsChangeListener listener;
                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (CaughtHostsChangeListener)listeners[ i ];
                    listener.autoConnectHostAdded( position );
                }
            }
        });
    }

    private void fireCaughtHostRemoved( final int position )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                CaughtHostsChangeListener listener;
                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (CaughtHostsChangeListener)listeners[ i ];
                    listener.caughtHostRemoved( position );
                }
            }
        });
    }

    private void fireAutoConnectHostRemoved( final int position )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                CaughtHostsChangeListener listener;
                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (CaughtHostsChangeListener)listeners[ i ];
                    listener.autoConnectHostRemoved( position );
                }
            }
        });
    }
    ///////////////////// END event handling methods ////////////////////////
}