
/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package phex.host;


import java.util.*;
import java.io.*;
import phex.*;
import phex.config.*;
import phex.event.CaughtHostsChangeListener;
import phex.event.NetworkHostsChangeListener;
import phex.interfaces.*;
import phex.msg.*;
import phex.utils.*;

/**
 * Responsible for managing caught host and the network neighbourhood.
 */
final public class HostManager
{
    private CaughtHostsContainer caughtHostsContainer;
    private NetworkHostsContainer networkHostsContainer;
    private ThrottleController mBandwidthThrottle;

    public HostManager()
    {
        caughtHostsContainer = new CaughtHostsContainer();
        networkHostsContainer = new NetworkHostsContainer();
        mBandwidthThrottle = ThrottleController.acquireThrottle();
    }
    
    /**
     *
     * Call before getting rid of HostManager.
     *
     * TODO: Convert users of HostManager to use dispose. Should this be called from finalize?
     */
    public void dispose()
    {
        mBandwidthThrottle.dispose();
    }


    /////////////// START CaughtHostsContainer wrapper methods ///////////////////////
    public CaughtHostsContainer getCaughtHostsContainer()
    {
        return caughtHostsContainer;
    }

    public void initializeCaughtHosts()
    {
        caughtHostsContainer.initializeCaughtHostsContainer();
    }

    /**
     * Adds a caught host if the limit is not reached...
     */
    public void addCaughtHost( HostAddress address, short priority )
    {
        caughtHostsContainer.addCaughtHost( address, priority );
    }

    public int getCaughtHostCount()
    {
        return caughtHostsContainer.getCaughtHostsCount();
    }

    public String getCaughtHostAt( int index )
    {
        return caughtHostsContainer.getCaughtHostAt( index );
    }

    public String[] getCaughtHostsAt( int[] indices )
    {
        return caughtHostsContainer.getCaughtHostsAt( indices );
    }

    public void removeCaughtHost( String host )
    {
        caughtHostsContainer.removeCaughtHost( host );
    }

    /**
     * Removes all caught hosts and initializes them with the startup hosts.
     */
    public void resetCaughtHosts()
    {
        caughtHostsContainer.resetCaughtHosts();
    }

    public void saveHostsContainer()
    {
        try
        {
            caughtHostsContainer.saveHostsContainer();
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }

    public void addCaughtHostsChangeListener( CaughtHostsChangeListener listener )
    {
        caughtHostsContainer.addCaughtHostsChangeListener( listener );
    }
    /////////////// END CaughtHostsContainer wrapper methods ///////////////////////

    /////////////// START NetworkHostsContainer wrapper methods ///////////////////////

    public void createOutgoingConnectionToHost( String hostname )
    {
        networkHostsContainer.createOutgoingConnectionToHost( hostname );
    }

    public void addConnectedHost( Host host )
    {
        networkHostsContainer.addConnectedHost( host );
    }

    public void cleanupConnectedHosts()
    {
        networkHostsContainer.cleanupConnectedHosts();
    }

    public void addIncomingHost( Host host )
    {
        networkHostsContainer.addIncomingHost( host );
    }

    public int getConnectedHostCount()
    {
        return networkHostsContainer.getConnectedHostCount();
    }

    public boolean hasIncommingSlotsAvailable()
    {
        return networkHostsContainer.hasIncommingSlotsAvailable();
    }

    public Host getConnectedHostAt( int index )
    {
        return networkHostsContainer.getConnectedHostAt( index );
    }

    public void periodicallyCheckHosts()
    {
        networkHostsContainer.periodicallyCheckHosts();
    }

    public int getNetworkHostCount()
    {
        return networkHostsContainer.getNetworkHostCount();
    }

    public void cleanupNetworkHosts()
    {
        networkHostsContainer.cleanupNetworkHosts();
    }

    public Host getNetworkHostAt( int index )
    {
        return networkHostsContainer.getNetworkHostAt( index );
    }

    public void removeAllNetworkHosts()
    {
        networkHostsContainer.removeAllNetworkHosts();
    }

    public void removeNetworkHosts( Host[] hosts )
    {
        networkHostsContainer.removeNetworkHosts( hosts );
    }

    public void removeNetworkHost( Host host )
    {
        networkHostsContainer.removeNetworkHost( host );
    }


    public void addNetworkHostsChangeListener( NetworkHostsChangeListener listener )
    {
        networkHostsContainer.addNetworkHostsChangeListener( listener );
    }

    public void removeNetworkHostsChangeListener( NetworkHostsChangeListener listener )
    {
        networkHostsContainer.removeNetworkHostsChangeListener( listener );
    }

    public void fireNetworkHostChanged( Host host )
    {
        networkHostsContainer.fireNetworkHostChanged( host );
    }
    /////////////// END NetworkHostsContainer wrapper methods ///////////////////////

    public void autoConnectToNextCaughtHost()
    {
        if (!ServiceManager.sCfg.mAutoConnect)
        {
            return;
        }
        String	hostname = caughtHostsContainer.getNextCaughtHost();
        if (hostname != null)
        {
            networkHostsContainer.createOutgoingConnectionToHost( hostname );
        }
    }

/*
    public void globalThrottleControl( int transferedBytes )
    {
        ThrottleController.setRate( ServiceManager.sCfg.mNetMaxRate );
        ThrottleController.controlThrottle( transferedBytes );
    }
*/
    // TODO move somewhere more appropriate
    public void throttleControl( long bytesTransfered )
    {
        mBandwidthThrottle.setRate( ServiceManager.sCfg.mNetMaxRate );
        mBandwidthThrottle.controlThrottle( bytesTransfered );
    }
}
