/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.query;

import javax.swing.*;
import phex.config.Cfg;
import phex.ServiceManager;
import phex.download.DownloadFile;
import phex.download.DownloadManager;
import phex.download.RemoteFile;
import phex.event.SearchChangeListener;
import phex.event.SearchChangeEvent;

public class ResearchSetting
{
    private long lastResearchStartTime;

    /**
     * The count of research that didn't return any new results.
     */
    private int noNewResultsCount;
    private int totalResearchCount;
    private String searchTerm;
    private Search search;

    /**
     * When the search has new results. This flag is true.
     */
    private boolean hasNewSearchResults;


    // Since currently the only one how uses the research setting is the
    // download file this solution is ok... later we need to find a different
    // way
    private DownloadFile downloadFile;
    private DownloadManager downloadMgr;
    private ResearchChangeListener researchChangeListener;

    public ResearchSetting( DownloadFile file )
    {
        downloadFile = file;
        downloadMgr = ServiceManager.getDownloadManager();
        researchChangeListener = new ResearchChangeListener();
    }

    public long getLastResearchStartTime()
    {
        return lastResearchStartTime;
    }

    public void setLastResearchStartTime( long time )
    {
        lastResearchStartTime = time;
    }

    public int getNoNewResultsCount()
    {
        return noNewResultsCount;
    }

    public String getSearchTerm()
    {
        return searchTerm;
    }

    public void setSearchTerm( String term )
    {
        searchTerm = term;
    }

    public void startSearch( long searchTimeout )
    {
        if ( search != null && search.isSearching() )
        {
            return;
        }

        if ( searchTerm.length() < Cfg.MIN_SEARCH_TERM_LENGTH )
        {
            return;
        }
        hasNewSearchResults = false;
        BackgroundSearchContainer backgroundSearchContainer =
            ServiceManager.getQueryManager().getBackgroundSearchContainer();
        Search.FileSizeConstraints fileSize =
            new Search.FileSizeConstraints( downloadFile.getTotalFileSize(),
                downloadFile.getTotalFileSize() );
        search = backgroundSearchContainer.createSearch( searchTerm, 0,
            fileSize, searchTimeout );
        search.addSearchChangeListener( researchChangeListener );
        totalResearchCount ++;
        long currentTime = System.currentTimeMillis();
        lastResearchStartTime = currentTime;
    }

    public int getTotalResearchCount()
    {
        return totalResearchCount;
    }

    public void stopSearch()
    {
        if ( search == null || !search.isSearching() )
        {
            return;
        }
        search.stopSearching();
    }

    public int getSearchHitCount()
    {
        return search.getQueryHitCount();
    }

    public boolean isSearchRunning()
    {
        if ( search == null )
        {
            return false;
        }
        return search.isSearching();
    }

    public class ResearchChangeListener implements SearchChangeListener
    {
        public void searchChanged( final SearchChangeEvent event )
        {
            SwingUtilities.invokeLater( new Runnable()
            {
                public void run()
                {
                    // after search has stoped check if we found any thing.
                    if ( event.getType() == SearchChangeEvent.SEARCH_STOPED )
                    {
                        if ( hasNewSearchResults == false )
                        {   // no new results...
                            noNewResultsCount ++;
                        }
                        else
                        {
                            noNewResultsCount = 0;
                        }
                    }

                    if ( event.getType() != SearchChangeEvent.SEARCH_HITS_ADDED )
                    {
                        downloadMgr.fireDownloadFileChanged( downloadFile );
                        return;
                    }


                    // Adds a file from a backgroundsearch to the candidates list.
                    Search search = (Search) event.getSource();
                    int startIdx = event.getStartIndex();
                    int endIdx = event.getEndIndex();
                    for ( int i = startIdx; i < endIdx; i++ )
                    {
                        RemoteFile rFile = search.getQueryHit( i );
                        boolean isAdded = downloadFile.addRemoteCandidate( rFile );
                        if ( isAdded )
                        {
                            hasNewSearchResults = true;
                        }
                    }
                    downloadMgr.fireDownloadFileChanged( downloadFile );
                }
            });
        }
    }
}