/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *  Copyright (C) 2000 William W. Wong williamw@jps.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.share;


import java.io.*;
import java.net.*;
import java.util.*;
import java.text.*;

import phex.*;
import phex.msg.GETRequest;
import phex.config.*;
import phex.connection.*;
import phex.host.*;
import phex.interfaces.*;
import phex.utils.*;


public class ShareManager
{
    private static final ShareFile[] VERSION_FILE_ARRAY =
    {
        new ShareFile( null )
    };
    private static final String PHEX_PING = "!`phex~~ping";

    //private IndexerManager	mIndexerManager;
    //private DataChanger		mStatusChangedListener = new DataChanger();

    private FileAdministration fileAdministration;
    private UploadFileContainer uploadFileContainer;

    public ShareManager()
    {
        //mIndexerManager = ServiceManager.getIndexerManager();
        fileAdministration = new FileAdministration( this );
        uploadFileContainer = new UploadFileContainer();
    }

    /*public void addStatusChangedListener(IDataChangedListener listener)
    {
        mStatusChangedListener.addListener(listener);
    }*/

    /**
     * Returns true if all upload slots are filled.
     */
    public boolean isHostBusy()
    {
        if ( uploadFileContainer.getUploadFileCount() >= ServiceManager.sCfg.mMaxUpload )
        {
            return true;
        }
        return false;
    }

    public FileAdministration getFileAdministration()
    {
        return fileAdministration;
    }

    public UploadFileContainer getUploadFileContainer()
    {
        return uploadFileContainer;
    }

    public void addUploadFile(UploadFile file)
    {
        uploadFileContainer.addUploadFile( file );
    }

    public ShareFile[] searchFile( String searchStr )
    {
        if ( ServiceManager.sCfg.mPhexPingResponse && searchStr.equals( PHEX_PING ) )
        {
            return VERSION_FILE_ARRAY;
        }

        String[] terms = new String[10];
        int termCount = 0;
        StringTokenizer tokens = new StringTokenizer( searchStr );

        String term;
        // Build search term, upto 10 terms.
        while (termCount < 10 && tokens.hasMoreElements())
        {
            term = tokens.nextToken();
            // ignore terms with less then 2 char length
            if ( term.length() >= 2 )
            {
                terms[ termCount++ ] = term.toLowerCase();
            }
        }

        ShareFile[] shareFiles = fileAdministration.getSharedFiles();
        // Don't search for empty string. Exception: Index query.
        if (termCount == 0)
        {
            if (searchStr.equals("    "))
            {
                ServiceManager.log( "Index query detected." );
                return shareFiles;
            }
            // return empty map for no search result
            return new ShareFile[0];
        }

        HashMap resultsHash = new HashMap();
        SearchEngine searchEngine = new SearchEngine();

        // searches through the files for each search term. Drops all files that
        // dont match a search term from the possible result list.

        //long start1 = System.currentTimeMillis();
        ArrayList leftFiles = new ArrayList();
        // all files are possible results
        leftFiles.addAll( Arrays.asList( shareFiles ) );
        // go through each term
        for ( int i = 0; i < termCount && leftFiles.size() > 0; i++ )
        {
            searchEngine.setPattern( terms[i], true );

            // go through each left file in the files array
            for ( int j = leftFiles.size() - 1; j >= 0; j-- )
            {
                ShareFile shareFile = (ShareFile)leftFiles.get( j );
                searchEngine.setText( shareFile.getSearchCompareTerm(), true );
                if ( !searchEngine.match() )
                {
                    // a term dosn't match remove possible result.
                    leftFiles.remove( j );
                }
            }
        }
        if ( leftFiles.size() > ServiceManager.sCfg.mUploadMaxSearch )
        {
            // limit list to max contain number of files.
            leftFiles.subList( ServiceManager.sCfg.mUploadMaxSearch - 1,
                leftFiles.size() - 1 ).clear();
        }

        ShareFile[] resultFiles = new ShareFile[ leftFiles.size() ];
        resultFiles = (ShareFile[]) leftFiles.toArray( resultFiles );

        // increment search count for files in list
        for ( int i = 0; i < resultFiles.length; i++ )
        {
            ShareFile shareFile = resultFiles[ i ];
            shareFile.incSearchCount();
        }
        //long end1 = System.currentTimeMillis();
        //int size1 = results.size();
        //System.out.println( size1 + "  " + (end1 - start1) );


        // Content search.  Match the search terms to words of each file.
        /*Hashtable	filenumbers = termsMatchIndex(terms, termCount);
        if (filenumbers != null)
        {
            Enumeration	numbers = filenumbers.keys();
            while (numbers.hasMoreElements())
            {
                ShareFile	shareFile = (ShareFile)mNumbersToFiles.get(numbers.nextElement());

                if (resultsByNumber.get(shareFile.getFileIndex()) != null)
                {
                    // Avoid duplicated result
                    continue;
                }

                shareFile.incSearchCount(1);
                results.addElement(shareFile);
                resultsByNumber.put(shareFile.getFileIndex(), shareFile);
                if (results.size() >= ServiceManager.sCfg.mUploadMaxSearch)
                {
                    // Result set size reaches max.  Don't search anymore.
                    break;
                }
            }
        }*/


        return resultFiles;
    }

    // Called by ReadWorker to handle a HTTP GET request from the remote host.
    public void httpRequestHandler(ConnectionRequest req, Host remoteHost)
    {
        if (!req.readHeaders(remoteHost.getIs()))
        {
            // Error in reading HTTP readers.  Just return and close connection.
            return;
        }

        // GET / HTTP/1.1
        if (req.getUri().equals("/"))
        {
            // The remote host just want the index.html.
            // Return a list of shared files.
            sendFile(req, remoteHost);
            return;
        }

        // GET /get/1/foo doo.txt HTTP/1.1
        GETRequest getRequest = new GETRequest( req.getUri() );

        /*catch (Exception e)
        {
            e.printStackTrace();
            // Didn't get all the expected fields.  Wrong request.
            // Just return a list of shared files.
            sendFile(req, remoteHost);
            return;
        }*/

        if ( !getRequest.getOperation().toLowerCase().equals( "get" ) )
        {
            // Wrong operation.
            sendFile(req, remoteHost);
            return;
        }

        ShareFile shareFile = null;
        // file index is -1 when parsing was wrong
        if ( getRequest.getFileIndex() != -1 )
        {
            shareFile = fileAdministration.getFileByIndex( getRequest.getFileIndex() );
            if ( shareFile != null )
            {
                File file = shareFile.getFile();
                // if filename dosn't fit
                if ( !getRequest.getFileName().equalsIgnoreCase( shareFile.getEffectiveName() ) )
                {
                    shareFile = null;
                }
            }
        }


        if ( shareFile == null )
        {
            // TODO currently this will not work right because the file hash contains
            // the full path name informations of the file. But we only look for the
            // filename.
            shareFile = fileAdministration.getFileByName( getRequest.getFileName() );
            if ( shareFile == null )
            {
                // Can't find file.
                sendString(remoteHost, buildErrorHTTP("404 Not Found", "File not found"));
                return;
            }
        }

        // Finally things look ok.  Go ahead and do upload.
        if ( upload( req, remoteHost, shareFile.getFile(), true ) )
        {
            shareFile.incUploadCount();
        }
    }

    private boolean upload(ConnectionRequest req, Host remoteHost, File file,
        boolean gnutellaUpload)
    {
        if ( !gnutellaUpload )
        {
            System.out.println( "NON Gnutella net upload: " + remoteHost + "  " + file.getName() );
        }

        RandomAccessFile	fis = null;
        UploadFile			localfile = null;
        HostManager			hm = ServiceManager.getHostManager();
        boolean				rc = false;

        // We will need to throttle the bandwidth in the upload loop.
        ThrottleController bandwidthThrottle = ThrottleController.acquireThrottle();

        try
        {
            HttpRange range;
            String rangeValue;
            int	transferLen;

            // Verify that upload resources are available, and that the request range for partial gets
            // is valid. All these conditions must be checked and the the call to addUploadFile made
            // atomically since this method is called by many threads potentially at the same time.
            synchronized (this)
            {
                // Check if allow further upload
                /*if (!ServiceManager.sCfg.mShareEnabled)
                {
                    remoteHost.log("Upload request: upload disabled.  No more upload allowed.");
                    sendString(remoteHost, buildErrorHTTP("404 Not Found", "File sharing disabled"));
                    return rc;
                }*/

                // Control the number of simultaneous uploads.
                if ( uploadFileContainer.getUploadFileCount() >= ServiceManager.sCfg.mMaxUpload)
                {
                    remoteHost.log("Upload request: upload limit reached.");
                    sendString(remoteHost, buildErrorHTTP("503 Service Unavailable", "Server busy.  Too many downloads from this host."));
                    return rc;
                }

                // Control the number of simultaneous uploads per IP.
                int uploadCount = uploadFileContainer.getUploadCountPerIP(
                remoteHost.getHostAddress().getHostName() );
                if ( uploadCount >= ServiceManager.sCfg.mMaxUploadPerIP)
                {
                    remoteHost.log("Upload request: upload limit per host reached.");
                    sendString(remoteHost, buildErrorHTTP("503 Service Unavailable", "You have too many downloads from this host."));
                    return rc;
                }

                // See if it's a partial get.
                range = new HttpRange((int)file.length());
                rangeValue = req.getHeader("RANGE");
                if (rangeValue != null)
                {
                    if (!range.deserialize(rangeValue))
                    {
                        // Invalide range value.
                        remoteHost.log("Upload request: Invalid range value in partial get.  " + rangeValue);
                        sendString(remoteHost, buildErrorHTTP("416 Requested Range Not Satisfiable", "Upload request: Invalid range value in partial get.  " + rangeValue));
                        return rc;
                    }
                    if (range.getSpecCount() == 0)
                    {
                        range.addSpec(0, (int)file.length() - 1);
                    }
                }
                else
                {
                    range.addSpec(0, (int)file.length() - 1);
                }

                transferLen = range.getLastPos() - range.getFirstPos() + 1;

                localfile = new UploadFile(file, remoteHost, transferLen, req.getHeader("USER-AGENT"));
                localfile.setStatus(UploadFile.sUploading);
                localfile.setStartingTime(System.currentTimeMillis());
                addUploadFile(localfile);
            } // end of synchronized (this)

            // The upload is being allowed if we get this far.

            // Prepare HTTP response
            byte[]			outbuf = new byte[512];
            StringBuffer headers = new StringBuffer();
            String			contentType = "application/binary";

            // in the case of a webupload through a browser.
            if (!gnutellaUpload)
            {
                // Mapping file type to content type.
                String ext = StrUtil.getFileExt( localfile.getFilename() );
                contentType = MimeTypeMapping.getMimeTypeForExtension( ext );
            }

            if (transferLen == range.getEntitySize())
            {
                headers.append( "HTTP/1.1 200 OK\r\n" );
            }
            else
            {
                if (gnutellaUpload)
                {
                    // Gnutella client expects non-standard syntax for partial GET.
                    headers.append( "HTTP/1.1 200 OK\r\n" );
                }
                else
                {
                    // Use standard HTTP syntax.
                    headers.append( "HTTP/1.1 206 OK\r\n" );
                }
            }
            headers.append( "Server: " );
            headers.append( StrUtil.getAppNameVersion() );
            headers.append( "\r\n" );

            headers.append( "Accept-Ranges: bytes\r\n" );
            headers.append( "Connection: close\r\n" );

            headers.append( "Content-Type: " );
            headers.append( contentType );
            headers.append( "\r\n" );
            if (transferLen == range.getEntitySize())
            {
                headers.append( "Content-length: " );
                headers.append( range.getEntitySize() );
                headers.append( "\r\n" );
            }
            else
            {
                // Use standard HTTP syntax.
                headers.append( "Content-range:bytes " );
                headers.append( range.getFirstPos() );
                headers.append( "-" );
                headers.append( range.getLastPos() );
                headers.append( "/" );
                headers.append( range.getEntitySize() );
                headers.append( "\r\n" );
                headers.append( "Content-length: " );
                headers.append( transferLen );
                headers.append( "\r\n" );
            }
            headers.append( "\r\n" );

            // Write HTTP response headers
            int len = IOUtil.serializeString(headers.toString(), outbuf, 0);
            OutputStream os = remoteHost.getOs();
            os.write(outbuf, 0, len);

            fis = new RandomAccessFile(file, "r");
            fis.seek(range.getFirstPos());

            long count = 0;

            while (transferLen > 0)
            {
                // Read a chunk from the file
                len = fis.read(outbuf);

                // FIXME:  This is an error condition which will cause the upload to be marked completed not errored!
                if (len == -1)
                    break;

                // Write the just read chunk to the output stream
                os.write(outbuf, 0, len);

                // Keep track of the fact that we've transfered a bit more.
                transferLen -= len;
                count += len;
                localfile.setTransferredDataSize(count);

                // Throttle our bandwidth usage appropriately.
                bandwidthThrottle.setRate( ServiceManager.sCfg.mUploadMaxBandwidth / uploadFileContainer.getUploadFileCount() );
                bandwidthThrottle.controlThrottle( len );
            }

            localfile.setTransferredDataSize(count);
            localfile.setStatus(UploadFile.sCompleted);
            localfile.setStoppingTime( System.currentTimeMillis() );

            StatisticTracker statTracker = ServiceManager.getStatisticTracker();
            statTracker.incStatUploadCount(1);

            rc = true;

            // Wait a bit before closing the connection.
            // Somehow the remote gnutella won't read the last
            // buffer if closing the connection too soon.
            Thread.sleep(2000);
        }
        catch (Exception e)
        {
            rc = false;
//			e.printStackTrace();
            if (localfile != null && localfile.getStatus() != UploadFile.sAborted)
            {
                localfile.setStatus(UploadFile.sError, e.getMessage());
                localfile.setStoppingTime( System.currentTimeMillis() );
            }
        }
        finally
        {
            // Release the throttle correctly.
            ThrottleController.releaseThrottle( bandwidthThrottle );
            bandwidthThrottle = null;
        }

        if (fis != null)
        {
            try
            {
                fis.close();
            }
            catch (IOException e)
            {
            }
        }
        return rc;
    }


    private void sendFile(ConnectionRequest req, Host remoteHost)
    {
        String	uri;

        try
        {
            uri = URLUtil.decodeURL(req.getUri());
        }
        catch (Exception e)
        {
            remoteHost.log("Invalid URI from client.  " + req.getUri());
            sendString(remoteHost, buildErrorHTTP("400 Bad Request", "Invalid URI from client.  " + req.getUri()));
            return;
        }

        if (uri.indexOf("..") != -1)
        {
            remoteHost.log("Invalid URI from client, containing '..'  " + req.getUri());
            sendString(remoteHost, buildErrorHTTP("400 Bad Request", "Invalid URI from client.  " + req.getUri()));
            return;
        }

        Object[] sharedDirs = fileAdministration.getSharedDirectories();
        for (int i = 0; i < sharedDirs.length; i++)
        {
            String dirName = (String) sharedDirs[ i ];
            File dir = new File(dirName);
            String normalizedUri = StrUtil.replaceAllChar(uri, '/', File.separator);
            String path = dir.getAbsolutePath() + normalizedUri;
            File file = new File(path);


            if (!file.exists())
                continue;

            if (!file.isFile())
            {
                File defaultFile;

                defaultFile = new File(file, "index.html");
                if (defaultFile.exists())
                {
                    upload(req, remoteHost, defaultFile, false);
                    return;
                }
                defaultFile = new File(file, "Index.html");
                if (defaultFile.exists())
                {
                    upload(req, remoteHost, defaultFile, false);
                    return;
                }
                defaultFile = new File(file, "default.html");
                if (defaultFile.exists())
                {
                    upload(req, remoteHost, defaultFile, false);
                    return;
                }
                defaultFile = new File(file, "Default.html");
                if (defaultFile.exists())
                {
                    upload(req, remoteHost, defaultFile, false);
                    return;
                }

                if (!ServiceManager.sCfg.mShareBrowseDir)
                {
                    sendString(remoteHost, buildErrorHTTP("403 Forbidden", "Directory browsing is not allowed."));
                }
                else
                {
                    sendString(remoteHost, buildDirListHTTP(file, uri));
                }
                return;
            }

            ShareFile shareFile = (ShareFile) fileAdministration.getFileByName( file.getAbsolutePath() );
            if (shareFile != null)
            {
                File file2 = shareFile.getFile();
                if (upload(req, remoteHost, file2, false))
                {
                    shareFile.incUploadCount();
                }
            }
            else
            {
                sendString(remoteHost, buildErrorHTTP("404 Not Found", "File not found."));
            }
            return;
        }

        sendString(remoteHost, buildErrorHTTP("404 Not Found", "File not found."));
    }


    private void sendString(Host remoteHost, String html)
    {
        try
        {
            // Prepare HTTP response
            byte[]			outbuf = new byte[html.length()];
            int				lenToSend = IOUtil.serializeString(html, outbuf, 0);
            int				lenSent = 0;
            int				len;
            OutputStream	os = remoteHost.getOs();
            HostManager		hm = ServiceManager.getHostManager();
            StatisticTracker statTracker = ServiceManager.getStatisticTracker();

            // Write HTTP response
            while (lenSent < lenToSend)
            {
                len = lenToSend - lenSent;

                if (len > 1024)
                    len = 1024;

                os.write(outbuf, lenSent, len);

                lenSent += len;

                statTracker.incBytesCount(len);

                // Keep track of the fact that we've done some reading.
                hm.throttleControl( len );
            }

            // Wait a bit before closing the connection.
            // Somehow the remote gnutella won't read the last
            // buffer if closing the connection too soon.
            Thread.sleep(2000);
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }


    private String buildDirListHTTP(File dir, String uri)
    {
        StringBuffer		buf = new StringBuffer();
        StringBuffer		body = new StringBuffer();
        File[]				files;
        String				field;
        String				date;
        String				size;
        SimpleDateFormat	dformat = new SimpleDateFormat("yyyy/MM/dd KK:mma");


        // Build the HTTP header
        buf.append("HTTP/1.1 200 OK\r\n");
        buf.append("Server: " + StrUtil.getAppNameVersion() + "\r\n");
        buf.append("Content-Type: text/html\r\n");
        buf.append("Connection: close\r\n");

        // Build the HTTP body
        body.append("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2 Final//EN\">");
        body.append("<HTML>");
        body.append("<HEAD><TITLE>Index of " + uri + "</TITLE></HEAD>");
        body.append("<BODY>");

        body.append("<H3>Index of " + uri + "</H3>\n");
        body.append("<PRE>");

        body.append("Name                                   Last Modified           Size       Bytes\n");
        body.append("<HR>");

        if (uri.startsWith("/") && uri.length() > 1)
        {
            // A subdirectory.
            // Build link to parent directory.
            if (uri.endsWith("/"))
                uri = uri.substring(0, uri.length() - 1);

            int	index = uri.lastIndexOf("/");
            uri = uri.substring(0, index + 1);

            String	encStr = URLUtil.encodeURL(uri);
            body.append("[<A HREF=\"" + encStr + "\">Parent Directory</A>]\n");

            // Get a list of files in the subdirectory.
            files = dir.listFiles();
        }
        else
        {
            // Root directory.  Get a list of files of all the top directories.
            ArrayList list = new ArrayList();
            Object[] sharedDirs = fileAdministration.getSharedDirectories();
            for (int i = 0; i < sharedDirs.length; i++)
            {
                String dirName = (String) sharedDirs[ i ];
                files = new File(dirName).listFiles();
                for (int j = 0; j < files.length; j++)
                {
                    list.add( files[j] );
                }
            }

            files = new File[list.size()];
            for (int i = 0; i < list.size(); i++)
            {
                files[i] = (File)list.get( i );
            }
        }

        // Output the file list to a HTML page.
        for (int j = 0; j < files.length; j++)
        {
            if ( fileAdministration.isFileInvalid(files[j]))
                continue;

            String			filename = files[j].getName();

            if (!files[j].isFile())
            {
                filename += "/";
            }

            String			encStr = URLUtil.encodeURL(filename);
            String			paddedFilename = StrUtil.getPadStr(filename, 39);
            StringBuffer	filenameHTML = new StringBuffer(paddedFilename);
            StrUtil.replaceAllChar(filenameHTML, '&', "&amp;");
            StrUtil.replaceAllChar(filenameHTML, '\"', "&quot;");
            StrUtil.replaceAllChar(filenameHTML, '<', "&lt;");
            StrUtil.replaceAllChar(filenameHTML, '>', "&gt;");

            field = "<A HREF=\"" + encStr + "\">" + filename + "</A>" + filenameHTML;
            body.append(field);

            date = dformat.format(new java.util.Date(files[j].lastModified()));
            body.append(StrUtil.padStringRight(date, 19));

            size = (files[j].isFile() ? StrUtil.formatSizeBytes(files[j].length()) : "-");
            body.append(StrUtil.padStringLeft(size, 9));

            size = (files[j].isFile() ? String.valueOf(files[j].length()) : "-");
            body.append(StrUtil.padStringLeft(size, 12));

            body.append("\n");
        }

        body.append("<HR>");
        body.append("Powered by ");
        body.append( StrUtil.getAppNameVersion() );
        body.append("\r\n<A HREF=\"");
        body.append( Res.getStr("Program.Url") );
        body.append( "\">" );
        body.append( Res.getStr("Program.Url") );
        body.append( "</A>\r\n");
        body.append("</PRE>");
        body.append("</BODY>");
        body.append("</HTML>");

        // Finish up the header and add in the body.
        buf.append("Content-Length: " + body.length() + "\r\n");
        buf.append("\r\n");
        buf.append(body);

        return buf.toString();
    }

    private String buildErrorHTTP(String statusStr, String errMsg)
    {
        StringBuffer buf = new StringBuffer();

        buf.append( "HTTP/1.1 " + statusStr + "\r\n");
        buf.append( "Server: " + StrUtil.getAppNameVersion() + "\r\n");
        buf.append( "Connection: close\r\n");
        buf.append( "Content-Tyep: text/plain\r\n");
        buf.append( "Content-Length: " + errMsg.length() + "\r\n");
        buf.append( "\r\n");
        buf.append( "<html><head><title>PHEX</title></head><body>" );
        buf.append( errMsg );
        buf.append( "<hr>Visit the Phex website at ");
        buf.append( "<a href=\"http://phex.sourceforge.net\">http://phex.sourceforge.net</a>.");
        buf.append( "</body>" );
        buf.append( "</html>" );

        return buf.toString();
    }



/*
    private Hashtable termsMatchIndex(String[] terms, int termCount)
    {
        Hashtable		result1;
        Hashtable		result2;

        result1 = (Hashtable)mIndexedFiles.get(terms[0]);
        if (result1 == null)
            return null;

        for (int i = 1; i < termCount; i++)
        {
            Hashtable	filenumbers = (Hashtable)mIndexedFiles.get(terms[i]);

            if (filenumbers == null)
                return null;

            result2 = new Hashtable();
            Enumeration	numbers = filenumbers.keys();
            while (numbers.hasMoreElements())
            {
                Integer		fileIndex = (Integer)numbers.nextElement();
                if (result1.get(fileIndex) != null)
                {
                    result2.put(fileIndex, "");
                }
            }
            result1 = result2;
        }

        return result1;
    }
*/
/*

    private void indexFile(Properties wordFiles, File file)
    {
        Hashtable		words;

        try
        {
            words = mIndexerManager.index(file, getFileExt(file.getName()));
        }
        catch (Exception e)
        {
            e.printStackTrace();
            return;
        }

        if (words == null)
            return;

        Enumeration		enum = words.keys();
        while (enum.hasMoreElements())
        {
            String		word = (String)enum.nextElement();
            String		filenames = (String)wordFiles.get(word);
            if (filenames == null)
            {
                filenames = "";
            }

            if (filenames.indexOf(file.getName() + ",") == -1)
                filenames += file.getName() + ",";
            wordFiles.put(word, filenames);
        }
    }


    private void addWordIndex(String filename, Integer fileNumber, Properties wordFiles)
    {
        Enumeration		keys = wordFiles.keys();

        while (keys.hasMoreElements())
        {
            String		word = (String)keys.nextElement();
            String		files = (String)wordFiles.get(word);
            if (files.indexOf(filename + ",") != -1)
            {
                Hashtable	filenumbers = (Hashtable)mIndexedFiles.get(word);

                if (filenumbers == null)
                {
                    filenumbers = new Hashtable();
                    mIndexedFiles.put(word, filenumbers);
                }

                filenumbers.put(fileNumber, "");
            }
        }
    }


    private String getFileExt(String filename)
    {
        int		index = filename.lastIndexOf(".");

        if (index > -1)
        {
            return filename.substring(index + 1, filename.length());
        }
        else
        {
            return "";
        }
    }

*/
/*

    private Properties loadAliasFile(File parent)
    {
        Properties	aliases = new Properties();
        try
        {
            FileInputStream	is = new FileInputStream(new File(parent, sAliasFilename));
            aliases.load(is);
            is.close();
        }
        catch (Exception e)
        {
        }
        return aliases;
    }

*/

/*
    public synchronized void removeUploadFile(int index)
    {
        if (index >= mUploadList.size())
            return;

        mUploadList.removeElementAt(index);
    }

*/
}
