/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.utils;

/**
 * The ReadWriteLock is a helper class for synchronized execution of
 * multiple threads on a single resource.
 * This class is responsible for maintaining a locking strategy that
 * allows multiple read but single write locking. The programmers is responsible
 * for using this class wisely and should be aware of possible deadlock situations.
 * Locking a resource should be done like this:
 * <pre>
 * rwLock.readLock();  // or writeLock()
 * try
 * {
 *    // use the resource here
 * }
 * finally
 * {
 *     try{ rwLock.readUnlock(); }  // or writeUnlock()
 *     catch (IllegalAccessException exp )
 *     { exp.printStackTrace(); }
 * }
 * </pre>
 */
public class ReadWriteLock
{
    private Thread writeOwner;
    private int readLockCount;
    private boolean readLock;
    private boolean writeLock;

    public ReadWriteLock()
    {
        readLock = false;
        writeLock = false;
    }

    /**
     * Sets up a read lock. The methods blocks if a write lock is currently
     * set and waits till it's released except when the current thread also
     * owns the write lock.
     */
    public synchronized void readLock( )
    {
        if ( writeOwner == Thread.currentThread() )
        {
            // I already have the write lock
            return;
        }
        while ( writeLock )
        {
            try{ wait(); }
            catch (InterruptedException exp){}
        }
        readLock = true;
        readLockCount ++;
    }

    /**
     * Unlocks a read lock.
     */
    public synchronized void readUnlock()
        throws IllegalAccessException
    {
        if ( writeOwner == Thread.currentThread() )
        {
            // I already have the write lock
            return;
        }
        readLockCount --;
        if ( readLockCount == 0 )
        {
            readLock = false;
        }
        else if ( readLockCount < 0 )
        {
            throw new IllegalAccessException("More read unlocks then locks found");
        }
        notify();
    }

    /**
     * Sets up a write lock. The methods blocks if a write lock is currently
     * set and waits till it's released except when the current thread also
     * owns the write lock.
     * After the write lock is set up no more read locks can be set and when
     * all read locks are unlocked the methods returns control.
     * TODO count the set up write locks and decrease the count here. But
     * actualy for a clean implementation no multiple write locks should be set
     * up by a single thread.
     */
    public synchronized void writeLock( )
    {
        if ( writeOwner == Thread.currentThread() )
        {
            // I already have the write lock
            return;
        }
        while ( writeLock )
        {
            try{ wait(); }
            catch (InterruptedException exp){}
        }
        // set the lock to block other reader...
        writeLock = true;
        writeOwner = Thread.currentThread();
        // wait till all readers are gone...
        while ( readLock )
        {
            try{ wait(); }
            catch (InterruptedException exp){}
        }
    }

    /**
     * Unlocks a write lock. Be aware that a single call to this method unlocks
     * a write lock even though there have been multiple write locks from the
     * same thread!!
     * TODO count the set up write locks and decrease the count here. But
     * actualy for a clean implementation no multiple write locks should be set
     * up by a single thread.
     */
    public synchronized void writeUnlock()
        throws IllegalAccessException
    {
        if ( writeOwner != Thread.currentThread() )
        {
            throw new IllegalAccessException("Current thread not owner.");
        }
        writeLock = false;
        notify();
    }

    private void debug()
    {
        System.out.println( "readLock: " + readLock + "  " + readLockCount +
        " - writeLock: " + writeLock + " - " + Thread.currentThread() );
    }
}