/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package phex.utils;

import phex.interfaces.IDataChangedListener;
import phex.DataChanger;

final public class StatisticTracker
{
    private DataChanger mStatusChangedListener = new DataChanger();
    private int mStatHosts;
    private int mStatFiles;
    private long mStatSize;
    private String mStatSizeStr;
    private int mStatTakenCount;
    private int mStatMsgCount;
    private int mStatDropCount;
    private int mStatUploadCount;
    private int	mStatDownloadCount;
    private long mRateTime0;
    private int mRateBytes = 0;
    private int mOldRate = 0;
    private final long phexStartTimeMillis = System.currentTimeMillis();

    public StatisticTracker()
    {
        resetStat();
    }
    
    /**
     * Return the length of time Phex has been running in seconds
     */
    public long getPhexUptimeInSeconds()
    {
        // Convert difference in start time to present from millis to seconds
        return (System.currentTimeMillis() - phexStartTimeMillis) / 1000L;
    }
    
    public int getStatHosts()
    {
        return mStatHosts;
    }

    public void setStatHosts(int val)
    {
        mStatHosts = val;
    }

    public synchronized void incStatHosts(int amount)
    {
        mStatHosts += amount;
    }

    public int getStatFiles()
    {
        return mStatFiles;
    }

    public synchronized void incStatFiles(int amount)
    {
        // if amount is over 1,000,000,000 ignore it... some auto connector provide it...
        if ( amount < 0 || amount > 1000000000 || mStatFiles == Integer.MAX_VALUE )
        {
            return;
        }
        if ( mStatFiles < Integer.MAX_VALUE - amount )
        {
            mStatFiles += amount;
        }
        else
        {
            mStatFiles = Integer.MAX_VALUE;
        }
    }

    public long getStatSize()
    {
        return mStatSize;
    }


    public void setStatSize(long val)
    {
        mStatSize = val;
        updateStatSizeStr();
    }


    public synchronized void incStatSize(long amount)
    {
        // Sounds bogos.  Ignore
        if (amount < 0)
            return;

        mStatSize += amount;
        updateStatSizeStr();
    }


    public String getStatSizeStr()
    {
        return mStatSizeStr;
    }


    private void updateStatSizeStr()
    {
        mStatSizeStr = StrUtil.formatSizeBytes(mStatSize << 10);
    }


    public int getStatTakenCount()
    {
        return mStatTakenCount;
    }


    public void setStatTakenCount(int val)
    {
        mStatTakenCount = val;
    }


    public synchronized void incStatTakenCount(int amount)
    {
        mStatTakenCount += amount;
    }


    public int getStatMsgCount()
    {
        return mStatMsgCount;
    }


    public void setStatMsgCount(int val)
    {
        mStatMsgCount = val;
    }


    public synchronized void incStatMsgCount(int amount)
    {
        mStatMsgCount += amount;
    }


    public int getStatDropCount()
    {
        return mStatDropCount;
    }


    public void setStatDropCount(int val)
    {
        mStatDropCount = val;
    }


    public synchronized void incStatDropCount(int amount)
    {
        mStatDropCount += amount;
    }


    public int getStatUploadCount()
    {
        return mStatUploadCount;
    }


    public void setStatUploadCount(int val)
    {
        mStatUploadCount = val;
    }


    public synchronized void incStatUploadCount(int amount)
    {
        mStatUploadCount += amount;
    }


    public int getStatDownloadCount()
    {
        return mStatDownloadCount;
    }


    public void setStatDownloadCount(int val)
    {
        mStatDownloadCount = val;
        mStatusChangedListener.dataChanged(null);
    }


    public synchronized void incStatDownloadCount(int amount)
    {
        mStatDownloadCount += amount;
        mStatusChangedListener.dataChanged(null);
    }


    public void resetStat()
    {
        mStatHosts = 0;
        mStatFiles = 0;
        mStatSize = 0;
        updateStatSizeStr(); // Must be done after mStatSize is reset       
        mStatTakenCount = 0;
        mStatMsgCount = 0;
        mStatDropCount = 0;
        mStatUploadCount = 0;
        mStatDownloadCount = 0;
        newRateSamplePeriod();
    }


    public void incBytesCount(int count)
    {
        mRateBytes += count;
    }


    public final int getRate()
    {
        if (mRateBytes == 0)
        {
            // No date in new sampling period.  Use old rate for continuation.
            return mOldRate;
        }
        return mRateBytes * 1000 / (int)(System.currentTimeMillis() - mRateTime0 + 1);
    }


    public void newRateSamplePeriod()
    {
        mOldRate = getRate();
        mRateBytes = 0;
        mRateTime0 = System.currentTimeMillis();
    }

    public void addStatusChangedListener(IDataChangedListener listener)
    {
        mStatusChangedListener.addListener(listener);
    }
}