/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2002 Peter Hunnisett ( hunnise@users.sourceforge.net )
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *                     Peter Hunnisett ( hunnise@users.sourceforge.net )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.utils;

import java.util.*;

import phex.*;


// TODO: A better throttle would also increase the size of transfers allowed to avoid/coalease sleeps. One heuristic
//       would be to attempt to allow only 1 transfer per window. If there was more than 1 transfer in the previous
//       pane, just increase the size of the fetch for the next pane. Might want to stop at 2, to allow for some
//       error. This would need to be combined with prebooking of some sort so as to not massively exceed the
//       throttle.
// TODO: Throttle is an after the fact thing. This means that the throttle can and will be exceeded. It should
//       require prebooking or something.
// TODO: Compensate for time drift in the thread that slides the window.
// TODO: Perhaps provide different types of throttles (pure clamp and clamp with memory).
// TODO: Throttle rates should only need to be set when things change and are setup for the first time. Setup some sort
//       of system to do this
// TODO: Allow reasonable names to be passed into the ThrottleController.
// TODO: Make the ThrottleController the implementation of an interface which can be used to hide certain methods from
//       the outside world.

/**
 * A clamping bandwidth throttle with memory. Bandwidth generally does not exceed set value within a
 * <code>paneLengthInMillis</code> period. Excess bandwidth is partially available for the
 * next period, exceeded bandwidth is fully unavailble for the next period.
 */
final public class ThrottleController implements NotifyThrottleController
{
    /**
     * Returns a ThrottleController object which can be used as a bandwidth throttle.
     */
    public static ThrottleController acquireThrottle()
    {
        return new ThrottleController();
    }

    /**
     * Release any resources associated with the throttle. Once this is called, the throttle
     * should no longer be used and should be dereferenced.
     * Must be called only once for each throttle.
     */
    public static void releaseThrottle( ThrottleController throttle )
    {
        throttle.dispose();
    }

    /**
     * Call to set the desired throttling rate. If the rate is different from
     * the originally set rate all appropriate internal variables will be reset.
     */
    public synchronized void setRate( long bytesPerSecond )
    {
        // logInfo( "setRate( bytesPerSecond = " + bytesPerSecond + " )" );
        maxRate = bytesPerSecond;
    }

    /**
     * Call this after every transfer.
     * If bandwidth is exceeded for the last little bit,
     */
    public void controlThrottle( long transferedBytes )
    {
        boolean transferMaxedOut;

        synchronized(this)
        {
            transferMaxedOut = slidingWindow.addBytesTransfered( transferedBytes );
        }

/* For Debugging
        logInfo( "ThrottleInfo for " + name() );
        logInfo( "Throttle in effect: " + transferMaxedOut + " Remaining time: " + getRemainingTimeThisPane() );
        logInfo( "Tx: " + transferedBytes + " Tx this pane: " + slidingWindow.getBytesTransfered()
                 + " Max Tx: " + slidingWindow.getMaxTransferAllowed() );
        logInfo( "MaxRate: " + maxRate + " Pane length: " + paneLengthInMillis + "\n" );
*/

        // Have we exceeded the number of bytes available this pane?
        if( transferMaxedOut )
        {
            // Throttle for the remainder of the pane duration plus a small safety margin of 3ms to ensure
            // that the window slides.
            long sleepTime = timer.getRemainingTimeThisPeriod() + 3;

            // If we're right at the end of a pane (or have exceeded it) perhaps we need to allow
            // other threads to run in order to advance the window.
            if( sleepTime <= 0 ) sleepTime = 1;

            try
            {
                Thread.sleep( sleepTime );
            }
            catch (Exception e)
            {
            }
        }
    }

    /**
     * Always create through <code>acquireThrottle</code>.
     */
    private ThrottleController()
    {
        // Create a 2 second sliding window composed of 2 panes. At this point
        // there is no value in increasing the number of windows as we do a trivial
        // aging of data inside the window.
        slidingWindow = new SlidingWindow( 2 );

        throttleNumberString = Integer.toHexString( hashCode() );

        setRate( 0L );
        
        timer.registerThrottleController( this );
    }

    /**
     * Purely for debugging purposes.
     */
    protected void finalize() throws Throwable
    {
        // logInfo( "finalize called for " + name() );
        super.finalize();
    }

    /**
     * Indicate that this thread should kill itself. Once the thread kills itself
     * there should be no more references to it anymore so finalize should get called.
     */
    public void dispose()
    {
        timer.deregisterThrottleController( this );
    }

    private static void logInfo( String info )
    {
        Debug.msg( info );
    }
       
    /**
     * Called every second by our thread to reset our transfered counts.
     * TODO: Hide this from the outside world somehow. Should only be accessable to ThrottleResetTimer.
     */
    public void newRatePeriod()
    {
        slidingWindow.slideWindow( maxRate ); // This is wrong if pane size != 1 second (actually maxRate * 1sec/pane length)
    }
    
    public String name()
    {
        return throttleNumberString;
    }
    
    private long maxRate;
    private SlidingWindow slidingWindow;
    
    private final static long oneSecondInMillis = 1000L;
    private final static long paneLengthInMillis = oneSecondInMillis;
    
    private final String throttleNumberString;
        
    // Singleton throttle timer.
    final private static ThrottleResetTimer timer = new ThrottleResetTimer( paneLengthInMillis );
}    

interface NotifyThrottleController
{
  void newRatePeriod();
  String name(); // For debugging purposes only.
}

/**
 * Singleton class which advances the sliding window every <code>paneLengthInMillis</code>.
 * The class is designed to be created and never destroyed as we always have at least one
 * throttle going at a time.
 */
final class ThrottleResetTimer extends Thread
{
    /**
     * Called to have a throttle controller notified every <code>paneLengthInMillis</code>.
     */
    public void registerThrottleController( NotifyThrottleController newThrottleToNotify )
    {
        // TODO: This should throw if adding fails.
        // Add the new throttle to the callback list
        synchronized( throttles )
        {
            throttles.add( newThrottleToNotify );
        }
        
        //logInfo( "Adding NotifyThrottleController interface " + newThrottleToNotify.name() );
    }
    
    /**
     * Called to stop a throttle controller from being notified
     */
    public synchronized boolean deregisterThrottleController( NotifyThrottleController throttleToRemove )
    {
        //logInfo( "Removing NotifyThrottleController interface " + throttleToRemove.name() );
                
        synchronized( throttles )
        {
            return throttles.remove( throttleToRemove );
        }
    }
    
    public ThrottleResetTimer( long resetPeriod )
    {
        super( "ThrottleResetTimer" );
        
        resetPeriodInMillis = resetPeriod;
        
        // Create a list which doesn't require too much memory. We don't care about realtime since we shouldn't
        // be creating a destroying throttles very often.
        throttles = new LinkedList();
        
        // Start ourselves
        start();
    }
    
    /**
     * Return the length of time, in milliseconds, remaining until the throttle will told to reset again.
     */
    public long getRemainingTimeThisPeriod()
    {
        return resetPeriodInMillis - (System.currentTimeMillis() - lastSlideTime);
    }
    
    /**
     * Every <code>paneLengthInMillis</code> milliseconds notify all registered throttle controllers.
     */
    public void run()
    {
        ListIterator listIter;
        
        // TODO: This should be self compensating for a drifting 1 second period.
        while( true )
        {
            try
            {
                sleep( resetPeriodInMillis );
            }
            catch ( Exception e )
            {
            }
            
            // Provide exclusive access to the throttle list so that we don't have to worry about
            // it changing part way through this iteration.
            synchronized( throttles )
            {
                listIter = throttles.listIterator();
                //logInfo( "Starting ThrottleController notification" );

                while( listIter.hasNext() )
                {
                    NotifyThrottleController toNotify = (NotifyThrottleController)listIter.next();
                    toNotify.newRatePeriod();
                    //logInfo( "Notify sent to " + toNotify.name() );
                }
                
                //logInfo( "Finished ThrottleController notification" );
                
                // Be sure to reset after all throttles are reset in case queried while notifying all throttles.
                lastSlideTime = System.currentTimeMillis();                
            }
        }
    }
    
    private static void logInfo( String info )
    {
        Debug.msg( info );
    }

    private final List throttles;
    private final long resetPeriodInMillis;
        
    private long lastSlideTime = 0L;
}


/**
 *  Implements a sliding window for the bandwidth throttler.
 *  This class is not synchronized because it should always be called from
 *  synchronized methods inside ThrotlleController.
 */
// TODO: Implement a toString() for SlidingWindow for debugging purposes.
final class SlidingWindow
{
    public SlidingWindow( final int numberOfPanes )
    {
        //assert( timePerPane > 0 );
        //assert( numberOfPanes > 1 );

        // Create a circular list of numberOfPanes WindowPane
        windowPanes = new LinkedList();
        for( int count = 0; count < numberOfPanes; count++ )
        {
            windowPanes.add( new WindowPane() );
        }

        // Set currentPane to the first pane.
        currentPane = (WindowPane)windowPanes.getFirst();
    }

    /**
     * Push one pane out the back and add a new one to the front.
     */
    public void slideWindow( long maxBytes )
    {
        // Follow a simplistic accouting method:
        long transferAdjustment = currentPane.maxBytesTransfered - currentPane.bytesTransfered;

        // If bandwidth was exceeded in this pane, fully remove it from the
        // next pane.
        // If bandwidth was not fully used, make 30% (no reason for this number) of
        // remaining available for this pane.
        //
        // These adjustments will allow us to deal with a bursty transfer, keep data
        // flowing on it, but not steal too much from other transfers. The assumption
        // for this is that bursty traffic will be well distributed over all connections
        // for this to be "perfectly" efficient.
        if( transferAdjustment > 0 )
        {
            transferAdjustment = (long)((float)transferAdjustment * 0.3f);
        }

        // Move the last to the front.
        currentPane = (WindowPane)windowPanes.removeLast();
        windowPanes.addFirst( currentPane );

        // The information in the new front pane is now obsolete.
        currentPane.setNewContents( 0, maxBytes + transferAdjustment );
    }

    /**
     * Indicate that <code>bytesTransfered</code> bytes have just been transfered in
     * the present window pane.
     *
     * @returns Total number of bytes transfered in this period.
     */
    public boolean addBytesTransfered( long bytesTransfered )
    {
        currentPane.bytesTransfered += bytesTransfered;

        return currentPane.bytesTransfered >= currentPane.maxBytesTransfered;

    }

    /**
     * Return the number of bytes transfered this pane.
     * For debug only.
     */
    public long getBytesTransfered()
    {
        return currentPane.bytesTransfered;
    }

    /**
     * Return the max amount of transfer allowed this pane.
     * For debug only.
     */
    public long getMaxTransferAllowed()
    {
        return currentPane.maxBytesTransfered;
    }

    private final LinkedList windowPanes; // Collection is fixed at startup.
    private WindowPane currentPane; // The present pane. All others contain past events.
}

/**
 * Implements a portion of a sliding window. A WindowPane has no concept of how large it is.
 * It is a simple container.
 */
final class WindowPane
{
    public WindowPane()
    {
        setNewContents( 0L, 0L );
    }


    public void setNewContents( final long tx, final long maxTx )
    {
        bytesTransfered = tx;
        maxBytesTransfered = maxTx;
    }

    public long bytesTransfered;
    public long maxBytesTransfered;
}
