
/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2000 William W. Wong
 *  williamw@jps.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package phex;

import java.io.*;
import java.rmi.dgc.*;
import java.rmi.server.*;
import phex.utils.HexDec;


public class GUID implements Serializable
{
    // Constant
    public static final int		sDataLength = 16;

    // Atributes
    private byte[] mBytes;
    private Integer mHashCode = new Integer(0);
    private String mStr = null;



    public GUID()
    {
        VMID vmid = new java.rmi.dgc.VMID();
        mBytes = new byte[sDataLength];
        vmid2GUID(vmid.toString());
        computeHashCode();
    }


    // Create an empty GUID.
    public GUID(boolean dummy)
    {
        mBytes = new byte[sDataLength];
    }


    public GUID(byte[] guidBytes)
    {
        mBytes = guidBytes;
        mStr = null;
        computeHashCode();
    }


    public void setGuid(byte[] guidBytes)
    {
        System.arraycopy(guidBytes, 0, mBytes, 0, sDataLength);
        mStr = null;
        computeHashCode();
    }


    public byte[] getGuid()
    {
        // The caller better not modified it.
        return mBytes;
    }


    public Integer getHashCode()
    {
        return mHashCode;
    }


    public boolean equals(GUID b)
    {
        for (int i = 0; i < sDataLength; i++)
        {
            if (mBytes[i] != b.mBytes[i])
                return false;
        }
        return true;
    }


    private void computeHashCode()
    {
        int		hashedValue;
        int		value;
        int		v1, v2, v3, v4;

        v1 = (((int)mBytes[0]) < 0 ? ((int)mBytes[0]) + 256 : ((int)mBytes[0]));
        v2 = (((int)mBytes[1]) < 0 ? ((int)mBytes[1]) + 256 : ((int)mBytes[1]));
        v3 = (((int)mBytes[2]) < 0 ? ((int)mBytes[2]) + 256 : ((int)mBytes[2]));
        v4 = (((int)mBytes[3]) < 0 ? ((int)mBytes[3]) + 256 : ((int)mBytes[3]));
        hashedValue = (v1 << 24) | (v2 << 16) | (v3 << 8) | (v4);

        for (int i = 4; i < sDataLength; i += 4)
        {
            v1 = (((int)mBytes[i+0]) < 0 ? ((int)mBytes[i+0]) + 256 : ((int)mBytes[i+0]));
            v2 = (((int)mBytes[i+1]) < 0 ? ((int)mBytes[i+1]) + 256 : ((int)mBytes[i+1]));
            v3 = (((int)mBytes[i+2]) < 0 ? ((int)mBytes[i+2]) + 256 : ((int)mBytes[i+2]));
            v4 = (((int)mBytes[i+3]) < 0 ? ((int)mBytes[i+3]) + 256 : ((int)mBytes[i+3]));
            value = (v1 << 24) | (v2 << 16) | (v3 << 8) | (v4);
            hashedValue ^= value;
        }

        mHashCode = new Integer(hashedValue);
    }


    public int getSize()
    {
        return sDataLength;
    }


    public void copy(byte[] guidBytes)
    {
        System.arraycopy(guidBytes, 0, mBytes, 0, sDataLength);
        mStr = null;
        computeHashCode();
    }


    public int serialize(byte[] outbuf, int offset)
    {
        // Copy my content to output buffer.
        System.arraycopy(mBytes, 0, outbuf, offset, sDataLength);

        // return new offset
        return offset + sDataLength;
    }


    public int deserialize(byte[] inbuf, int offset)
    {
        // Copy input buffer to my content.
        System.arraycopy(inbuf, offset, mBytes, 0, sDataLength);
        mStr = null;
        computeHashCode();

        // return new offset
        return offset + sDataLength;
    }


    private void vmid2GUID(String vmid)
    {
        for (int i = 0, j = 0; i < vmid.length(); i++)
        {
            if (i < sDataLength)
            {
                mBytes[j] = (byte)vmid.charAt(i);
            }
            else
            {
                mBytes[j] ^= (byte)vmid.charAt(i);
            }
            j++;
            if (j >= sDataLength)
            {
                j = 0;
            }
        }
        mStr = null;

        // to meet current protocol standard set byte 9 to 0xFF and byte 16 to 0
        // see http://groups.yahoo.com/group/the_gdf/message/1397
        mBytes[8] = (byte) 0xFF;
        mBytes[15] = 0;
    }


    public String toString()
    {
        if (mStr == null)
        {
            mStr = generateString();
        }
        return mStr;
    }


    private String generateString()
    {
        String	guidStr = "";

        guidStr += HexDec.convertBytesToHexString(mBytes, 0, 4) + "-";
        guidStr += HexDec.convertBytesToHexString(mBytes, 4, 2) + "-";
        guidStr += HexDec.convertBytesToHexString(mBytes, 6, 2) + "-";
        guidStr += HexDec.convertBytesToHexString(mBytes, 8, 2) + "-";
        guidStr += HexDec.convertBytesToHexString(mBytes, 10, 6);

        return guidStr;
    }


    public String toHexString()
    {
        return HexDec.convertBytesToHexString(mBytes);
    }


    public void fromHexString(String hexValue)
            throws Exception
    {
        mBytes = HexDec.convertHexStringToBytes(hexValue);
        mStr = null;
        computeHashCode();
    }


    public void readFromOIS(java.io.DataInputStream dis, boolean firstTime)
        throws IOException, ClassNotFoundException
    {
        if (firstTime)
        {
            dis.skipBytes(67);
        } else {
            if (dis.readByte() == 113) {
                ServiceManager.log("found known GUID class: skipping");
                dis.skipBytes(3);
                return;
            }
            dis.skipBytes(3);
        }
        byte[] b={dis.readByte(),dis.readByte(),dis.readByte(),dis.readByte()} ;
        ServiceManager.log("Reading GUID after: "+b[0]+" "+b[1]+" "+b[2]+" "+b[3]);
        int	version = dis.readInt();
        ServiceManager.log("Reading GUID Item: "+version);
        switch (version)
        {
        case 1:
            deserialize1(dis, firstTime);
            break;

        default:
            throw new IOException("Fail to deserialize guid of datafile. Unknown verison.");
        }
        mStr = null;
        computeHashCode();
        ServiceManager.log("Reading GUID: "+toString());
        ServiceManager.log("Reading GUID: "+toHexString());
    }


    private void deserialize1(DataInputStream dis, boolean firstTime)
        throws IOException, ClassNotFoundException
    {
        if (firstTime)
        {
            dis.skipBytes(23);
        } else {
            dis.skipBytes(10);
        }
        for (int i=0; i<sDataLength; i++)
        {
            mBytes[i] = dis.readByte();
        }
    }

    public static GUID createFromHexString( String hexValue )
        throws Exception
    {
        GUID guid = new GUID( true );
        guid.fromHexString( hexValue );
        return guid;
    }

    // Unit test
    public static void main(String[] args)
    {
        GUID	guid = new GUID();

        System.out.println(guid);
    }

}


