
/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *                     Mark Saltzman (marks@marksaltzman.com)
 *  Copyright (C) 2000 William W. Wong
 *  williamw@jps.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package phex;

import java.net.*;
import java.io.*;
import java.util.*;

import phex.config.*;
import phex.host.*;
import phex.interfaces.*;

public class Listener implements Runnable
{
    private InetAddress mMyIP;
    private ServerSocket mListeningSocket;
    private boolean mRequestToDie = false;
    private boolean mRunning = false;
    private boolean mShutdownCompleted = false;
    private IHostChanged mHostChangedListener = null;

    private boolean hasConnectedIncomming;
    private HostAddress localAddress;

    public Listener()
    {
        hasConnectedIncomming = false;
        try
        {
            localAddress = new HostAddress(
                HostAddress.LOCAL_HOST_NAME, HostAddress.DEFAULT_PORT );
            if (ServiceManager.sCfg.mMyIP.length() > 0)
            {
                mMyIP = InetAddress.getByName( ServiceManager.sCfg.mMyIP );
            }
        }
        catch ( UnknownHostException e)
        {
            e.printStackTrace();
        }
    }

    public synchronized void startup()
            throws IOException
    {
        if ( mRunning )
        {
            return;
        }

        // Init the ShareManager.
        ServiceManager.getShareManager();
        ServiceManager.log("Listener starting...");

        // Default to 0 for random port.
        int port = ServiceManager.sCfg.mListeningPort;
        // Create a listening socket at the port.
        int tries = 0;
        boolean error;
        // try to find new port if port not valid
        do
        {
            error = false;

            try
            {
                mListeningSocket = new ServerSocket( port, 50 );
            }
            catch ( BindException exp )
            {
                if ( tries > 10 )
                {
                    throw exp;
                }
                error = true;
                port ++;
                tries ++;
            }
        }
        while ( error == true );

        mRequestToDie = false;
        mRunning = true;

        port = mListeningSocket.getLocalPort();
        String hostName = resolveSelfHostName();
        localAddress.updateAddress( hostName, port );
        if (mHostChangedListener != null)
        {
            mHostChangedListener.updateMyIP();
        }

        new Thread(this, "Listener-" + Integer.toHexString(hashCode())).start();
    }

    public synchronized void shutdown(boolean waitForCompleted)
    {
        // not running, already dead or been requested to die.
        if ( !mRunning || mRequestToDie)
            return;

        ServiceManager.log("Listener shutting down...");

        // Set flag to die.
        mShutdownCompleted = false;
        mRequestToDie = true;

        // Dummy connection to the listener to wake it up to die.
        try
        {
            Socket tmpSock = new Socket( HostAddress.LOCAL_HOST_NAME,
                localAddress.getPort() );
            tmpSock.close();
        }
        catch(IOException e)
        {
            // Don't care.
        }

        if (waitForCompleted)
        {
            // Wait until the thread is dead.
            while (!mShutdownCompleted)
            {
                try
                {
                    wait();
                }
                catch (InterruptedException e)
                {
                    break;
                }
            }
        }
    }


    protected boolean getRequestToDie()
    {
        return mRequestToDie;
    }


    public boolean getRunning()
    {
        return mRunning;
    }

    public void setSelfHostname( String hostname )
    {
        try
        {
            if ( mMyIP == null )
            {
                localAddress.updateAddress( hostname, localAddress.getPort() );
            }
            else
            {
                localAddress.updateAddress( mMyIP.getHostAddress(),
                    localAddress.getPort() );
            }
            if (mHostChangedListener != null)
            {
                mHostChangedListener.updateMyIP();
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }


    public void setMyIP(String ip)
    {
        try
        {
            if (ip.length() > 0)
            {
                mMyIP = InetAddress.getByName(ip);
            }
            else
            {
                mMyIP = null;
            }
            setSelfHostname( localAddress.getHostName() );
        }
        catch (Exception e)
        {
            mMyIP = null;
        }
    }

    public HostAddress getLocalAddress()
    {
        return localAddress;
    }

    public boolean hasConnectedIncoming()
    {
        return hasConnectedIncomming;
    }

    public void setHostChangedListener(IHostChanged listener)
    {
        mHostChangedListener = listener;
    }

    // The listening thread.
    public void run()
    {
        ServiceManager.log("Listener started.");

        Socket incoming;

        while (true)
        {
            try
            {
                // Waiting for incoming connection.
                incoming = mListeningSocket.accept();
                incoming.setSoTimeout(ServiceManager.sCfg.mSocketTimeout);
                hasConnectedIncomming = true;
            }
            catch(IOException e)
            {
                ServiceManager.log(e.toString());
                break;
            }

            // See if I has been asked to die.
            if (getRequestToDie())
            {
                break;
            }

            try
            {
                // Set this will defeat the Nagle Algorithm, making short bursts of
                // transmission faster, but will be worse for the overall network.
                // incoming.setTcpNoDelay(true);

                // Create a Host object for the incoming connection
                // and hand it off to a ReadWorker to handle.
                HostAddress address = new HostAddress(
                    incoming.getInetAddress().getHostAddress(),
                    incoming.getPort() );
                Host host = new Host( address );
                host.setType(Host.sTypeIncoming);
                host.setSock(incoming);
                host.setOs(incoming.getOutputStream());
                host.setIs(incoming.getInputStream());
                host.setStatus(Host.sStatusAccepting, "");
                new ReadWorker(host, 0);
            }
            catch (Exception e)
            {
                //e.printStackTrace();
                ServiceManager.log(e.toString());
            }
        }

        try
        {
            mListeningSocket.close();
        }
        catch(IOException e)
        {
            ServiceManager.log(e.toString());
        }
        shutdownCompleted();
    }

    private String resolveSelfHostName()
    {
        InetAddress addr = mListeningSocket.getInetAddress();
        String hostName = null;
        if ( hostName == null )
        {
            hostName = addr.getHostAddress();
        }
        if ( hostName.equals("0.0.0.0") )
        {
            hostName = localAddress.getHostName();
        }
        return hostName;
    }

    private synchronized void shutdownCompleted()
    {
        mRunning = false;
        ServiceManager.log("Listener stops.");
        mShutdownCompleted = true;
        localAddress.updateAddress(
            HostAddress.LOCAL_HOST_NAME, HostAddress.DEFAULT_PORT );
        if (mHostChangedListener != null)
        {
            mHostChangedListener.updateMyIP();
        }
        notifyAll();
    }
}