
/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *  Copyright (C) 2000 William W. Wong
 *  williamw@jps.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package phex;


import java.io.*;
import java.net.*;
import java.util.*;

import phex.config.*;
import phex.connection.*;
import phex.download.*;
import phex.host.*;
import phex.interfaces.*;
import phex.msg.*;
import phex.share.*;
import phex.query.QueryHistoryMonitor;
import phex.utils.*;
import phex.utils.IPUtils;


public class ReadWorker implements Runnable
{
    // GNUTELLA CONNECT/0.4\n\n
    private static final String	sRequestSignature = "GNUTELLA CONNECT/0.4";
    private static final String	sReplySignature = "GNUTELLA OK";
    private static final String	sPasswordChallege = "GNUTELLA AUTHENTICATION CHALLENGE";
    private static final String	sPasswordReply = "GNUTELLA AUTHENTICATION RESPONSE";
    private static final String	sAuthFailed = "GNUTELLA AUTHENTICATION FAILED";

    private Host			mRemoteHost;
    private byte[]			mHeaderBuf;		// pre-allocated buffer for repeated uses.
    private FlexBuf			mBuf1;			// general buf for read/send
    private FlexBuf			mBuf2;			// general buf for read/send
    private FlexBuf			mBodyBuf;		// Buf for reading msg body
    private int				mSleepBeforeStart = 0;
    private ServiceManager	mManager = ServiceManager.getManager();
    private ConnectionManager	mConnMgr = ServiceManager.getConnectionManager();
    private HostManager		mHostMgr = mManager.getHostManager();
    private SendManager		mSendMgr = mManager.getSendManager();
    private ShareManager	mShareMgr = mManager.getShareManager();
    private QueryHistoryMonitor queryHistory;
    private FileAdministration fileAdministration;
    private DownloadManager	mDownloadMgr = mManager.getDownloadManager();
    private StatisticTracker statTracker = ServiceManager.getStatisticTracker();
    private MsgManager		mMsgMgr = mManager.getMsgManager();
//	private IrcManager		mIrcMgr = mManager.getIrcManager();


    private ReadWorker()
    {
        // disable
    }


    public ReadWorker(Host remoteHost, int sleepBeforeStart)
    {
        mRemoteHost = remoteHost;
        mHeaderBuf = new byte[MsgHeader.sDataLength];
        mBuf1 = new FlexBuf();
        mBuf2 = new FlexBuf();
        mBodyBuf = new FlexBuf();
        mSleepBeforeStart = sleepBeforeStart;
        fileAdministration = mShareMgr.getFileAdministration();
        queryHistory = mManager.getQueryManager().getQueryHistoryMonitor();
        new Thread(this, "ReadWorker-" + Integer.toHexString(hashCode())).start();
    }


    public void run()
    {
        if (!mRemoteHost.acquireByWorker())
        {
            // Already has a worker working on it.  Just exit.
            return;
        }

        if (mSleepBeforeStart > 0)
        {
            try
            {
                Thread.sleep(mSleepBeforeStart);
            }
            catch (Exception e)
            {
            }
        }

        try
        {
            if (mRemoteHost.getType() == Host.sTypeOutgoing)
            {
                // Connect to remote host.
                mRemoteHost.setStatus(Host.sStatusConnecting, "");

                connectToRemoteHost();
                if (mRemoteHost.getStatus() == Host.sStatusTimeout)
                {
                    // Connecting has been taken too long.
                    throw new IOException("Timed out.");
                }

                // Set my local address.
                String localaddr = mRemoteHost.getSock().getLocalAddress().getHostAddress();
                mManager.getListener().setSelfHostname(localaddr);

                // Negotiate handshake
                mRemoteHost.setStatus(Host.sStatusConnecting, "Negotiate handshake.");
                negotiateHandShakeAsClient();

                // Connection to remote gnutella host is completed at this point.
                mRemoteHost.setStatus(Host.sStatusConnected, "");
                mHostMgr.addConnectedHost(mRemoteHost);

                // queue first Init msg to send.
                mSendMgr.queueMsgToSend(mRemoteHost, mMsgMgr.getMyMsgInit(), false);

                processIncomingData();
            }
            else if (mRemoteHost.getType() == Host.sTypeIncoming)
            {
                handleIncomingConnection();
            }
            else
            {
                throw new Exception("Invalid host type");
            }
        }
        catch (IOException e)
        {
            //e.printStackTrace();
            mRemoteHost.setStatus( Host.sStatusError, e.getMessage() );
        }
        catch (Exception e)
        {
            mRemoteHost.setStatus(Host.sStatusError, e.getMessage());
            e.printStackTrace();
        }
        finally
        {
            // Close the connection.  Need to close even for error.
            // Half-closed connection is bad for TCP.
            if (mRemoteHost != null)
            {
                mHostMgr.removeNetworkHost( mRemoteHost );
                mRemoteHost.releaseFromWorker();
                mRemoteHost.disconnect();
            }
        }
    }

    private void handleIncomingConnection()
        throws Exception
    {
        ConnectionRequest req = negotiateHandShakeAsServer();

        // See if the incoming connection is from a gnutella client.
        if (req.getMethod().equals(ConnectionRequest.sGnutellaConnect))
        {
            mHostMgr.addIncomingHost( mRemoteHost );
            // queue first Init msg to send. this fixes the missing file statistic
            mSendMgr.queueMsgToSend(mRemoteHost, mMsgMgr.getMyMsgInit(), false);
            processIncomingData();
        }
        else if (req.getMethod().equals(ConnectionRequest.sGet))
        {
            // Incoming connection is a HTTP GET to upload file.
            mShareMgr.httpRequestHandler(req, mRemoteHost);
        }
        else if (req.getMethod().equals(ConnectionRequest.sGiv))
        {
            // Incoming connection is a GIV, a response to my push request.
            // The remote host is pushing (sending) the file to me now.
            // Transfer mRemoteHost and its connection to the download thread.
            if (mDownloadMgr.notifyReadPushedFile(req, mRemoteHost))
            {
                // mRemoteHost has been transferred.  Relinquish control of it.
                mRemoteHost.releaseFromWorker();
                mRemoteHost = null;
            }
        }
        else
        {
            throw new Exception("Unknown incoming request type.");
        }
    }

    private void connectToRemoteHost()
            throws Exception
    {
        try
        {
            // since there is no way of defining a connection timeout this is
            // done through the a host timer
            mRemoteHost.markConnectionStartTime();
            Socket sock = mConnMgr.connect(mRemoteHost.getHostAddress());

            // Set this will defeat the Nagle Algorithm.  Make short bursts of
            // transmission faster, but will be worse for the overall LAN.
//			sock.setTcpNoDelay(true);

            // I am connected to the remote host at this point.
            mRemoteHost.setSock(sock);
            mRemoteHost.setOs(sock.getOutputStream());
            mRemoteHost.setIs(sock.getInputStream());
        }
        catch (UnknownHostException e)
        {
            throw new IOException("Unknown host.");
        }
    }


    private void negotiateHandShakeAsClient()
            throws Exception
    {
        // Send the first handshake greeting to the remote host.
        boolean	sentPassword = false;
        String greeting = ServiceManager.getNetworkManager().getFullGreeting() + "\n\n";
        byte[] buf = mBuf1.getBuf(1024);
        int len = IOUtil.serializeString(greeting, buf, 0);
        mRemoteHost.getOs().write(buf, 0, len);

        try
        {
            while (true)
            {
                // Read reply from server.
                StringBuffer strBuf = new StringBuffer();
                len = IOUtil.readToCRLF(mRemoteHost.getIs(), buf, 1024, 0);
                IOUtil.deserializeString(buf, 0, len, strBuf);
                String reply = strBuf.toString();
                String password = (String)ServiceManager.sCfg.mNetworkPasswords.get(ServiceManager.sCfg.mCurrentNetwork);

                if ( reply.equals(sReplySignature)
                  // even though this is illegal on an 0.4 connect
                  || reply.startsWith( "GNUTELLA/0.6 200 OK" ) )
                {
                    if (!sentPassword && password != null && password.length() > 0)
                    {
                        // I have password set on this network name but the
                        // remote host doesn't have for any password.
                        // Don't want to join a unsecured network.
                        throw new Exception("Remote host doesn't have password set.");
                    }

                    // Normal Gnutella reply.
                    return;
                }
                else if (reply.startsWith(sPasswordChallege))
                {
                    // Remote host challenges for password.
                    String		keyStr = reply.substring(sPasswordChallege.length() + 1);
                    byte[]		key = HexDec.convertHexStringToBytes(keyStr);

                    if (password == null || password.length() == 0)
                    {
                        throw new Exception("Password requried.");
                    }

                    byte[]		authDigest = Digest.computePasswordDigest(password, key);
                    String		authDigestStr = HexDec.convertBytesToHexString(authDigest);

                    // Send password reply.
                    len = IOUtil.serializeString(sPasswordReply + " " + authDigestStr + "\n\n", buf, 0);
                    mRemoteHost.getOs().write(buf, 0, len);
                    sentPassword = true;

                    // Loopback to read normal reply.
                    continue;
                }
                else if (reply.startsWith(sAuthFailed))
                {
                    throw new Exception("Password not accepted by remote host.");
                }

                // I don't recognize any other reply.
                throw new Exception( "Unrecognized Reply: " + reply + " Host: " +
                    mRemoteHost );
            }
        }
        catch (Exception e)
        {
            if (e instanceof IOException)
            {
                throw new IOException("Disconnected from remote host during initial handshake");
            }
            else
            {
                e.printStackTrace();
                throw e;
            }
        }
    }

    private ConnectionRequest negotiateHandShakeAsServer()
            throws Exception
    {
        if (!ServiceManager.getNetworkManager().getJoined())
        {
            throw new Exception("Network not joined.");
        }
        if ( IPUtils.isHostInUserIgnoreList( mRemoteHost.getHostAddress().getHostName() ) )
        {
            throw new IOException("Host is ignored.");
        }

        ConnectionRequest request = new ConnectionRequest( mRemoteHost );
        request.initialize();

        return request;
    }

    private void processIncomingData()
            throws Exception
    {
        mRemoteHost.setStatus(Host.sStatusConnected, "");

        try
        {
            while (true)
            {
                MsgHeader header = readHeader();
                if (header.getDataLen() > 65536)
                {
                    // Packet looks suspiciously too big.  Disconnect them.
                    mRemoteHost.log("Packet too big.  " + header);
                    throw new IOException("Packet too big.  Disconnecting the remote host.");
                }

                byte[] body = readBody(header);

                mRemoteHost.incReceivedCount();
                statTracker.incStatMsgCount(1);
                statTracker.incStatTakenCount(header.getHopsTaken());

                switch (header.getFunction())
                {
                    case MsgHeader.sInit:
                        handleInit(header, body);
                        break;

                    case MsgHeader.sInitResponse:
                        handleInitResponse(header, body);
                        break;

                    case MsgHeader.sPushRequest:
                        handlePushRequest(header, body);
                        break;

                    case MsgHeader.sQuery:
                        handleQuery(header, body);
                        break;

                    case MsgHeader.sQueryResponse:
                        handleQueryResponse(header, body);
                        break;

                    default:
                        handleUnknown(header, body);
                        break;
                }
            }
        }
        catch ( IOException exp )
        {
            //exp.printStackTrace();
            if (mRemoteHost.getSock() != null)
            {
                mRemoteHost.setStatus(Host.sStatusError, exp.getMessage());
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
            if (mRemoteHost.getSock() != null)
            {
                mRemoteHost.setStatus(Host.sStatusError, e.getMessage());
            }
        }
    }


    private MsgHeader readHeader()
            throws Exception
    {
        InputStream	is = mRemoteHost.getIs();
        if (is == null)
        {
            // IS from getIs() has been cleared when connection closed.
//			mRemoteHost.log("Disconnected from remote host during reading header");
            throw new IOException("Connection closed by remote host");
        }

        int lenRead = 0;
        int len;
        while (lenRead < MsgHeader.sDataLength)
        {
            len = is.read(mHeaderBuf, lenRead, MsgHeader.sDataLength - lenRead);
            if (len == 0 || len == -1)
            {
                mRemoteHost.log("Disconnected from remote host during reading header");
                throw new IOException("Connection closed by remote host");
            }
            lenRead += len;
        }
        statTracker.incBytesCount(lenRead);

        // resolve: reuse msg header from free list.
        MsgHeader header = new MsgHeader(new GUID(true));

        header.deserialize(mHeaderBuf, 0);
        header.setArrivalTime(System.currentTimeMillis());
        header.setFromHost(mRemoteHost);

        // Keep track of the fact that we've done some reading.
        mHostMgr.throttleControl( lenRead );

        return header;
    }


    private byte[] readBody(MsgHeader header)
            throws Exception
    {
        InputStream	is = mRemoteHost.getIs();
        if (is == null)
        {
            // IS from getIs() has been cleared when connection closed.
//			mRemoteHost.log("Disconnected from remote host during reading msg body");
            throw new IOException("Connection closed by remote host");
        }

        int lenToRead = header.getDataLen();
        int lenRead = 0;
        int len;
        int size;
        byte[] body = mBodyBuf.getBuf(lenToRead);

        // cleanup array for easier parsing and testing
        Arrays.fill( body, (byte)0x00 );

        while (lenRead < lenToRead)
        {
            size = lenToRead - lenRead;
            if (size > 1024)
            {
                size = 1024;
            }
            len = is.read(body, lenRead, size);
            if (len == 0 || len == -1)
            {
//				mRemoteHost.log("Disconnected from remote host during reading message body.");
                throw new IOException("Connection closed by remote host");
            }
            lenRead += len;

            statTracker.incBytesCount(len);

            // Keep track of the fact that we've done some reading.
            mHostMgr.throttleControl( lenRead );

        }
        return body;
    }


    private void handleInit( MsgHeader header, byte[] body )
            throws Exception
    {
        MsgInit msg = new MsgInit(header);

        msg.deserialize(body, 0);
//		mRemoteHost.log("Got msg " + msg);

        // See if I have seen this Init before.  Drop msg if duplicate.
        if (mMsgMgr.checkAndAddMsgSeen(msg))
        {
            mRemoteHost.incDropCount();
            statTracker.incStatDropCount(1);
//			mRemoteHost.log("Seen " + msg);
            return;
        }

        // Add the Init msg to the routing table so that I know where
        // to route the InitResponse back.
        mMsgMgr.addToRoutingTable(msg.getHeader().getMsgID(), mRemoteHost);

        // Forward the Init msg to other connected hosts except the one sent it.
        mMsgMgr.forwardMsg(msg, mRemoteHost);

        // Get my host:port for InitResponse.
        Listener listener = mManager.getListener();

        // to reduce the incomming connection attemps of other clients
        // only response to ping a when we have free incomming slots or this is
        // ping has a original TTL ( current TTL + hops ) of 2.
        if ( !mHostMgr.hasIncommingSlotsAvailable() )
        {
            return;
        }
        // Construct InitResponse msg.  Copy the original Init's GUID.
        MsgHeader newHeader = new MsgHeader( header.getMsgID() );
        MsgInitResponse	response = new MsgInitResponse( newHeader );
        newHeader.setTTL(header.getHopsTaken() + 1); // Will take as many hops to get back.
        response.setPort( (short)listener.getLocalAddress().getPort() );
        response.setIP( listener.getLocalAddress().getHostIP() );
        response.setFileCount( fileAdministration.getFileCount() );
        response.setTotalSize( fileAdministration.getTotalFileSizeInKb() );
        mSendMgr.queueMsgToSend( mRemoteHost, response, false );
    }


    private void handleInitResponse(MsgHeader header, byte[] body)
            throws Exception
    {
        MsgInitResponse	msg = new MsgInitResponse(header);

        msg.deserialize(body, 0);
//		mRemoteHost.log("Got msg " + msg);

        HostAddress address = new HostAddress( msg.getIP(), msg.getPort() );
        if ( !IPUtils.isHostInUserInvalidList( address.getHostName() ) )
        {
            mHostMgr.addCaughtHost( address, CaughtHostsContainer.LOW_PRIORITY );
        }

        // Handle init response to my origianl init.
        MsgInit msgInit = mMsgMgr.getMyMsgInit();
        if (msg.getHeader().getMsgID().equals(msgInit.getHeader().getMsgID()))
        {
            // This InitResponse is for me.
//			mRemoteHost.log("Got response to my init.  " + msg);
            statTracker.incStatHosts(1);
            statTracker.incStatFiles(msg.getFileCount());
            statTracker.incStatSize(msg.getTotalSize());
            return;
        }

        // See if the response is to my latency ping to neighbor.
        if (mRemoteHost.checkPingResponse(msg))
        {
            // Yes. Processed.
            return;
        }

        // InitResponse is a response to an Init msg.
        // Did I forward that Init msgID before?
        Host returnHost = mMsgMgr.getRouting(msg.getHeader().getMsgID());
        if (returnHost == null)
        {
//			mRemoteHost.log("Don't route the InitResponse since I didn't forward the Init msg.  " + msg);
            return;
        }

        // Ok, I did forward the Init msg on behalf of returnHost.
        // The InitResponse is for returnHost.  Better route it back.
        if ( mMsgMgr.decTTL(msg))
        {
//			mRemoteHost.log("TTL expired " + msg);
            return;
        }
        mSendMgr.queueMsgToSend(returnHost, msg, false);
    }


    private void handleQuery(MsgHeader header, byte[] body)
            throws Exception
    {
        MsgQuery		msg = new MsgQuery(header);

        msg.deserialize(body, 0);
//		mRemoteHost.log("Got msg " + msg);

        // See if I have seen this Query before.  Drop msg if duplicate.
        if (mMsgMgr.checkAndAddMsgSeen(msg))
        {
            mRemoteHost.incDropCount();
            statTracker.incStatDropCount(1);
//			mRemoteHost.log("Seen " + msg);
            return;
        }

        // Add the Query msg to the routing table so that I know where
        // to route the QueryResponse back.
        mMsgMgr.addToRoutingTable(msg.getHeader().getMsgID(), mRemoteHost);

        // Add to the net search history.
        queryHistory.addSearchQuery( msg );

        // Forward the Query msg to other connected hosts except the one sent it.
        mMsgMgr.forwardMsg(msg, mRemoteHost);

        // Perform search on my list.
        if (msg.getMinSpeed() * StrUtil.s1kB > ServiceManager.sCfg.mUploadMaxBandwidth )
        {
            return;
        }

        // Search the sharefile database and get groups of sharefiles.
        ShareFile[] resultFiles = mShareMgr.searchFile(msg.getSearchString());

        if ( resultFiles.length == 0)
        {
//			mRemoteHost.log("Not found");
            return;
        }

        Listener listener = mManager.getListener();

        // Construct QueryResponse msg.  Copy the original Init's GUID.
        MsgHeader newHeader = new MsgHeader(header.getMsgID());
        // Will take as many hops to get back.
        newHeader.setTTL( header.getHopsTaken() + 1 );

        int resultCount = resultFiles.length;
        if ( resultCount > 255 )
        {
            resultCount = 255;
        }

        ShareFile sfile = null;
        MsgResRecord[] records = new MsgResRecord[ resultCount ];
        MsgResRecord record;
        for (int i = 0; i < resultCount; i++)
        {
            sfile = resultFiles[ i ];
            record = new MsgResRecord( sfile.getFileIndex(),
                (int)sfile.getFileSize(), sfile.getEffectiveName() );
            records[ i ] = record;
        }

        HostAddress hostAddress = listener.getLocalAddress();
        InetAddress address = InetAddress.getByName( hostAddress.getHostName() );
        short port = (short)hostAddress.getPort();

        MsgQueryResponse response = new MsgQueryResponse(
            newHeader, mManager.getClientID(), address, port,
            Math.round( ServiceManager.sCfg.mUploadMaxBandwidth / StrUtil.s1kB ),
            records );

        // Send each batch in sequence.
        mSendMgr.queueMsgToSend(mRemoteHost, response, false);
    }


    private void handleQueryResponse(MsgHeader header, byte[] body)
        throws Exception
    {
        int length = header.getDataLen();
        byte[] realBody = new byte[ length ];
        // create a new body array and copy the data because the original body
        // is always reused and we don't want to parse the body right away.
        System.arraycopy( body, 0, realBody, 0, length );

        MsgQueryResponse msg = new MsgQueryResponse( header, realBody );

//		mRemoteHost.log("Got msg " + msg);

        mMsgMgr.addToPushRoutingTable(msg.getRemoteClientID(), mRemoteHost);
        mMsgMgr.processQueryResponse(mRemoteHost, msg);

        // QueryResponse is a response to a Query msg.
        // Did I forward that Query msgID before?
        Host returnHost = mMsgMgr.getRouting(msg.getHeader().getMsgID());
        if (returnHost == null)
        {
//			mRemoteHost.log("Don't route the QueryResponse since I didn't forward the Query msg.  " + msg);
            return;
        }

        // Ok, I did forward the Query msg on behalf of returnHost.
        // The QueryResponse is for returnHost.  Better route it back.
        if ( mMsgMgr.decTTL(msg) )
        {
//			mRemoteHost.log("TTL expired " + msg);
            return;
        }
        mSendMgr.queueMsgToSend(returnHost, msg, false);
    }


    private void handlePushRequest(MsgHeader header, byte[] body)
            throws Exception
    {
        MsgPushRequest msg = new MsgPushRequest(header);

        msg.deserialize(body, 0);
//		mRemoteHost.log("Got msg " + msg);

        if (mManager.getClientID().equals(msg.getClientID()))
        {
            new PushWorker(msg);
            return;
        }

        Host returnHost = mMsgMgr.getPushRouting(msg.getClientID());
        if (returnHost == null)
        {
//			mRemoteHost.log("Don't route the PushRequest since I didn't forward the QueryResponse msg.  " + msg);
            return;
        }

        // Ok, I did forward the QueryResponse msg on behalf of returnHost.
        // The PushRequest is for the returnHost.  Better route it back.
        if ( mMsgMgr.decTTL(msg))
        {
//			mRemoteHost.log("TTL expired " + msg);
            return;
        }
        mSendMgr.queueMsgToSend(returnHost, msg, false);
    }


    private void handleUnknown(MsgHeader header, byte[] body)
            throws Exception
    {
        mRemoteHost.incDropCount();
        statTracker.incStatDropCount(1);

                // Don't broadcast unknown messages
/*
                MsgUnknown		msg = new MsgUnknown(header);

                msg.deserialize(body, 0);
//		mRemoteHost.log("Got unknown msg " + msg);


        // See if I have seen this msg before.  Drop msg if duplicate.
        if (mMsgMgr.checkAndAddMsgSeen(msg))
        {
            mRemoteHost.incDropCount();
            mHostMgr.incStatDropCount(1);
//			mRemoteHost.log("Seen " + msg);
            return;
        }

        // Forward the msg to other connected hosts except the one sent it.
        mHostMgr.forwardMsg(msg, mRemoteHost); */
    }
}


