
/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *                     Leander Eyer
 *  Copyright (C) 2000 William W. Wong ( williamw@jps.net )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package phex;

import java.net.*;
import java.io.*;
import java.util.*;
import java.text.DateFormat;

import phex.config.*;
import phex.connection.*;
import phex.download.*;
import phex.host.*;
import phex.msg.*;
import phex.share.*;
import phex.utils.*;
import phex.common.TransferRateService;
import phex.query.QueryManager;
import phex.gui.common.LazyEventQueue;

public class ServiceManager
{
    private static final String CONFIG_FILE_NAME;
    private static final String LOG_FILE_NAME;
    private static final String HOSTS_FILE_NAME;
    private static final String AUTOCONNECT_HOSTS_FILE_NAME;
    private static final String DEBUG_FILE_NAME;
    private static final String OLD_DOWNLOAD_FILE_NAME;
    private static final String XML_DOWNLOAD_FILE_NAME;
    private static final String XML_RESEARCH_SERVICE_FILE_NAME;

    static
    {
        checkPhexHomePath();
        String phexHomePath = determinePhexHomePath();
        CONFIG_FILE_NAME = phexHomePath + File.separator + "phex.cfg";
        LOG_FILE_NAME = phexHomePath + File.separator + "phex.log";
        HOSTS_FILE_NAME = phexHomePath + File.separator + "phex.hosts";
        AUTOCONNECT_HOSTS_FILE_NAME = phexHomePath + File.separator + "autoconnecthosts.cfg";
        DEBUG_FILE_NAME = phexHomePath + File.separator + "phex.debug";
        OLD_DOWNLOAD_FILE_NAME = phexHomePath + File.separator + "phex.download";
        XML_DOWNLOAD_FILE_NAME = phexHomePath + File.separator + "phexdownload.xml";
        XML_RESEARCH_SERVICE_FILE_NAME = phexHomePath + File.separator + "phexresearch.xml";
    }

    private static ServiceManager	sTheManager = null;
    private static Object			sInitLocker = new Object();

    public static Cfg				sCfg;

    private String					mLogfilename;
    private String					mDebugfilename;
    private String					mHostsFilename;
    private String mObjectDownloadSaveFilename;
    private String mXMLDownloadSaveFilename;
    private String xmlResearchServiceFilename;
    private PrintWriter				mLogFile;
    private ConnectionManager		mConnectionManager;
    private NetworkManager			mNetworkManager;
    private SendManager				mSendManager;
    private HostManager				mHostManager;
    private MsgManager				mMsgManager;
    private ShareManager			mShareManager;
    private DownloadManager			mDownloadManager;
    private QueryManager queryManager;
    private IndexerManager			mIndexerManager;
    private Listener				mListener;
    private JobsRunner				mJobsRunner;
    private MainFrame				mMainFrame;
    private StatisticTracker statisticTracker;
    private TransferRateService transferRateService;
    private LazyEventQueue lazyEventQueue;

    /**
     * check if the directory for the phex config files exists already and tests if there
     * are old configuration files wich have to be integrated
     */
    private static void checkPhexHomePath ()
    {
        String phexHomePath = determinePhexHomePath();
        //check if the directory for the phex configuration files exist already
        //and generatie it if not
        File directory = new File( phexHomePath );
        if ( directory.isDirectory() == false )
        {
            directory.mkdir();
        }

        //check if there are old phex config files which have to be moved to
        //the new directory
        String [] names = {"phex.cfg", "phex.log", "phex.hosts", "phex.debug", "phex.download", "phexdownload.xml"};
        String home = System.getProperty("user.home");
        for (int i = 0; i < names.length; i++)
        {
            File oldFile = new File( home + File.separator + names [i] );
            if (oldFile.exists() == false)
            {
                continue;
            }
            File newFile = new File( phexHomePath + File.separator + names [i]);
            oldFile.renameTo( newFile );
        }
    }


    /**
     * return a string corresponding to the path of the phex home directory
     * there is no path separator at the end since it directly represents the directory
     */
    private static String determinePhexHomePath()
    {
        StringBuffer path = new StringBuffer(20);
        path.append(System.getProperty("user.home"));
        path.append(File.separator);

        //phex config files are hidden on all UNIX systems. Since there are many UNIX like
        //operation systems with Java Support out there, we can not recognize the OS through it's
        //name. Thus we check if the root of the filesystem starts with "/" since only UNIX uses
        //such filesystem conventions

        // This seems to cause problems for some users so I try it differently
        //if ( File.listRoots () [0].getPath ().startsWith ("/"))
        // hopefully this is valid for all and only unix systems!
        if ( File.separatorChar == '/' )
        {
            path.append ('.');
        }
        path.append ("phex");
        return path.toString();
    }

    public static ServiceManager getManager()
    {

        // Simple and fast check to avoid taking a mutex lock.
        if (sTheManager == null)
        {
            // Mutex to allow only one thread to initialize the manager.
            synchronized (sInitLocker)
            {
                // Check again in case someone else is initializing the manager.
                if (sTheManager == null)
                {
                    sTheManager = new ServiceManager();
                    sTheManager.init();
                }
            }
        }

        return sTheManager;
    }


    private ServiceManager()
    {
        sCfg = new Cfg( CONFIG_FILE_NAME );
        sCfg.load();

        // Set up log file.
        mLogfilename = sCfg.mLogFile;
        if (mLogfilename == null || mLogfilename.length() == 0)
            mLogfilename = LOG_FILE_NAME;
        mLogfilename = envSubstitution(mLogfilename);

        try
        {
            mLogFile = new PrintWriter(new BufferedWriter(new FileWriter(mLogfilename)));
        }
        catch (IOException e)
        {
            mLogFile = new PrintWriter(new OutputStreamWriter(System.out));
            logMsg("Failed to create default log file '" + mLogfilename + ".'  " + e.toString());
            logMsg("Use standard output.");
        }

        // Set up debug file.
        mDebugfilename = sCfg.mDebugFile;
        if (mDebugfilename == null || mDebugfilename.length() == 0)
            mDebugfilename = DEBUG_FILE_NAME;
        mDebugfilename = envSubstitution(mDebugfilename);

        try
        {
            Debug.out = new PrintWriter(new BufferedWriter(new FileWriter(mDebugfilename)));
        }
        catch (IOException e)
        {
            Debug.out = new PrintWriter(new OutputStreamWriter(System.out));
            Debug.out.println("Failed to create default debug file '" + mDebugfilename + ".'  " + e.toString());
            Debug.out.println("Use standard output.");
        }

        // Set up hosts file.
        mHostsFilename = sCfg.mHostFile;
        if (mHostsFilename == null || mHostsFilename.length() == 0)
            mHostsFilename = HOSTS_FILE_NAME;
        mHostsFilename = envSubstitution(mHostsFilename);

        // Set up the filename to save download data.
        mObjectDownloadSaveFilename = sCfg.mDownloadSaveFile;
        if (mObjectDownloadSaveFilename == null || mObjectDownloadSaveFilename.length() == 0 )
        {
            mObjectDownloadSaveFilename = OLD_DOWNLOAD_FILE_NAME;
        }
        mObjectDownloadSaveFilename = envSubstitution(mObjectDownloadSaveFilename);

        mXMLDownloadSaveFilename = sCfg.mXMLDownloadSaveFile;
        if (mXMLDownloadSaveFilename == null || mXMLDownloadSaveFilename.length() == 0 )
        {
            mXMLDownloadSaveFilename = XML_DOWNLOAD_FILE_NAME;
        }
        mXMLDownloadSaveFilename = envSubstitution(mXMLDownloadSaveFilename);

        xmlResearchServiceFilename = sCfg.xmlResearchServiceFilename;
        if (xmlResearchServiceFilename == null || xmlResearchServiceFilename.length() == 0 )
        {
            xmlResearchServiceFilename = XML_RESEARCH_SERVICE_FILE_NAME;
        }
        xmlResearchServiceFilename = envSubstitution( xmlResearchServiceFilename );

    }


    private void init()
    {
        log("\n*** ServiceManager initializing ***");

        statisticTracker = new StatisticTracker();
        transferRateService = new TransferRateService();
        mIndexerManager = new IndexerManager(this);
        mConnectionManager = new ConnectionManager();
        mNetworkManager = new NetworkManager();
        mListener = new Listener();
        mHostManager = new HostManager();
        queryManager = new QueryManager();
        mSendManager = new SendManager();
        mMsgManager = new MsgManager();
        mShareManager = new ShareManager();
        mDownloadManager = new DownloadManager();
        lazyEventQueue = new LazyEventQueue();
        mJobsRunner = new JobsRunner();

        log("*** ServiceManager is ready ***");
    }


    public synchronized void shutdown()
    {
        mLogFile.flush();

        log("manager shutting down...");

        mJobsRunner.shutdown(true);

        mLogFile.close();
    }


    public void startJobsRunner()
    {
        mJobsRunner.startup();
    }


    public static ConnectionManager getConnectionManager()
    {
        return ServiceManager.getManager().mConnectionManager;
    }

    public static LazyEventQueue getLazyEventQueue()
    {
        return ServiceManager.getManager().lazyEventQueue;
    }

    public static NetworkManager getNetworkManager()
    {
        return ServiceManager.getManager().mNetworkManager;
    }


    public static SendManager getSendManager()
    {
        return ServiceManager.getManager().mSendManager;
    }


    public static HostManager getHostManager()
    {
        return ServiceManager.getManager().mHostManager;
    }

    public static QueryManager getQueryManager()
    {
        return ServiceManager.getManager().queryManager;
    }

    public static StatisticTracker getStatisticTracker()
    {
        return ServiceManager.getManager().statisticTracker;
    }

    public static TransferRateService getTransferRateService()
    {
        return ServiceManager.getManager().transferRateService;
    }

    public static MsgManager getMsgManager()
    {
        return ServiceManager.getManager().mMsgManager;
    }


    public static ShareManager getShareManager()
    {
        return ServiceManager.getManager().mShareManager;
    }


    public static DownloadManager getDownloadManager()
    {
        return ServiceManager.getManager().mDownloadManager;
    }


    public static IndexerManager getIndexerManager()
    {
        return ServiceManager.getManager().mIndexerManager;
    }


    public static Listener getListener()
    {
        return ServiceManager.getManager().mListener;
    }

    public GUID getClientID()
    {
        return sCfg.mProgramClientID;
    }


    public static String getHostsFilename()
    {
        if (sCfg.mCurrentNetwork.equals(sCfg.sGeneralNetwork))
        {
            return getManager().mHostsFilename;
        }
        else
        {
            String	filename = "phex_" + sCfg.mCurrentNetwork + ".hosts";
            filename = StrUtil.convertToLocalSystemFilename(filename);
            return System.getProperty("user.home") + File.separator + filename;
        }
    }

    public static String getAutoConnectHostsFilename()
    {
        return AUTOCONNECT_HOSTS_FILE_NAME;
    }


    /**
     * Returns the old download filename based on Object streams.
     * @deprecated only for loading of used 0.4.6 download list
     */
    public static String getObjectDownloadSaveFilename()
    {
        return getManager().mObjectDownloadSaveFilename;
    }

    /**
     * Returns the new download list filename based on XML.
     */
    public static String getXMLDownloadSaveFilename()
    {
        return getManager().mXMLDownloadSaveFilename;
    }

    /**
     * Returns the research config xml filename.
     */
    public static String getXMLResearchServiceFilename()
    {
        return getManager().xmlResearchServiceFilename;
    }

    public static void log(String msg)
    {
        ServiceManager.getManager().logMsg(msg);
    }


    public static void log(Exception e)
    {
        ServiceManager.getManager().logMsg(e);
    }


    public void logMsg(String msg)
    {
        mLogFile.println(DateFormat.getTimeInstance(DateFormat.MEDIUM).format(new Date())+": "+msg);
        mLogFile.flush();
    }


    public void logMsg(Exception e)
    {
        mLogFile.println(e.toString());
        e.printStackTrace(mLogFile);
        mLogFile.flush();
    }


    public static String envSubstitution(String str)
    {
        String	str2 = str.toLowerCase();
        String	strEnv1 = "$user.home";
        String	strEnv2 = "$(user.home)";

        if (str2.startsWith(strEnv1))
        {
            str = System.getProperty("user.home") +
                  str.substring(strEnv1.length());
        }
        else if (str2.startsWith(strEnv2))
        {
            str = System.getProperty("user.home") +
                  str.substring(strEnv2.length());
        }

        return (str);
    }


    void setMainFrame(MainFrame frame)
    {
        mMainFrame = frame;
    }


    public MainFrame getMainFrame()
    {
        return mMainFrame;
    }

    public static void printStack(String str)
    {
        try
        {
            throw new Exception(str);
        }
        catch (Exception e)
        {
            log(e);
        }
    }
}


