
/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *                     Konrad Haenel ( alterego1@gmx.at )
 *  Copyright (C) 2000 William W. Wong ( williamw@jps.net )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package phex.config;


import java.net.*;
import java.io.*;
import java.util.*;
import java.awt.*;
import java.lang.reflect.*;

import phex.*;
import phex.config.*;
import phex.utils.*;

public class Cfg
{
    public static final int DEFAULT_SOCKS5_PORT = 1080;
    public static final int UNLIMITED_BANDWIDTH = Integer.MAX_VALUE;
    public static int MIN_SEARCH_TERM_LENGTH = 2;

    public final static String			sGeneralNetwork = "<General Gnutella Network>";


    public GUID					mProgramClientID = new GUID();
    public String				mMyIP = "";
    public int					mListeningPort = -1;
    public int					mMaxDownload = 4;
    public int					mMaxDownloadPerIP = 1;
    public int					mMaxUpload = 4;
    public int					mMaxUploadPerIP = 1;
    public int					mUploadMaxBandwidth = 102400;
    public int					mNetTTL = 5;
    public int					mNetMaxTTL = 7;
    public int                  mNetMaxHostToCatch = 5000;
    public int					mNetMaxSendQueue = 500;
    public int					mPingFrequency = 15000;
    public int					mPingTimeout = 60000;
    public int					mSearchMaxSearch = 500;
    public int					mSearchMaxConcurrent = 10;
    public String				mSearchFilter = "";
    public int					mNetMaxConnection = 6;
    public int					mNetMaxRate = 50000;

    public int mDownloadMaxBandwidth = 102400;
    public boolean				mDownloadAutoRemoveCompleted = false;
    public String				mDownloadDir = ".";
    public int					mDownloadMaxRetry = 999;
    public int					mDownloadRetryWait = 30*1000;
    public String				mDownloadSaveFile = "";
    public String mXMLDownloadSaveFile = "";
    public String xmlResearchServiceFilename = "";

    public boolean				mAutoConnect = true;
    public int					mNetMinConn = 4;
    public boolean				mAutoCleanup = true;
    public int					mUploadMaxSearch = 100;
    //public boolean mShareEnabled = true;
    public boolean				mShareBrowseDir = true;
    public int					mPushTransferTimeout = 60 * 1000;
    public Vector				mNetIgnoredHosts = new Vector();
    public Vector				mNetInvalidHosts = new Vector();
    public Vector				mFilteredSearchHosts = new Vector();
    public boolean				mApplyFilterdHosts = false;
    public String				mCurrentNetwork = sGeneralNetwork;
    public Vector				mNetNetworkHistory = new Vector();
    public boolean				mAutoJoin = true;
    public Hashtable			mNetworkPasswords = new Hashtable();
    public boolean				mDisconnectApplyPolicy = true;
    public int					mDisconnectDropRatio = 50;
    public int					mDisconnectLatency = 60000;
    public boolean				mIndexFiles = false;
    public boolean				mProxyUse = false;
    public String				mProxyHost = "";
    public int mProxyPort = DEFAULT_SOCKS5_PORT;
    public boolean useProxyAuthentication = false;
    public String mProxyUserName = "";
    public String mProxyPassword = "";
    public Font                 mFontMenu = new Font("Dialog", Font.PLAIN, 12);
    public Font					mFontLabel = new Font("Dialog", Font.PLAIN, 12);
    public Font					mFontTable = new Font("Dialog", Font.PLAIN, 12);
    public String				mFindText = "";
    public boolean				mFindMatchCase = false;
    public boolean				mFindDown = true;
    public boolean				mUIDisplayTooltip = true;
    public String				mLFClassName = javax.swing.UIManager.getSystemLookAndFeelClassName();
    public String				mUserName = "(noname)";
    public String				mUploadDir = "";
    public String				mUploadFileExclusions = "";
    public String				mUploadFileInclusions = "*";
    public boolean				mUploadScanRecursively = true;
    public boolean				mUploadAutoRemoveCompleted = false;
    public String				mHostFile = "";
    public String				mLogFile = "";
    public String				mDebugFile = "";
    public String				mSearchLastSearch = "";
    public int					mSearchMinSpeed = 0;
    public boolean				mPhexPingResponse = true;
    public boolean monitorSearchHistory = true;
    public int searchHistoryLength = 10;
    public long mMinimumFileSize = 0;
    public long mMaximumFileSize = Long.MAX_VALUE;

    /**
     * The total speed in kilo bits per second of the network connection the
     * user has available. This is not the bandwidth the user has available for
     * Phex.
     * The default of 256 matches a DSL/Cable connection.
     */
    public int networkSpeedKbps = 256;

    /**
     * This is the maximal bandwidth in bytes per second Phex is allowed to use
     * in total. This means network, download and upload bandwidth combined.
     * The default of 16384 matches 50% of the bandwidth a 256kbs DSL/Cable connection
     * is able to offer.
     */
    public int maxTotalBandwidth = 16384;

    /**
     * This is introduced to maintain the current version of Phex.
     * After a update of Phex the version in the cfg and the Phex version differs.
     * In this case we know that we need to upgrade the cfg or other stuff to the
     * new Phex version
     */
    public String runningPhexVersion = "";

    /**
     * Defines if the host is behind a firewall. The trying of push transfers
     * and the QueryResponse QHD depends on it.
     */
    public boolean isBehindFirewall = false;

    /**
     * Contains the version number of the last update check.
     */
    public String lastUpdateCheckVersion = "0";
    public long lastUpdateCheckTime = 0;
    public boolean showUpdateNotification = true;

    // mSocketTimeout defaults to 2 minutes
    public int mSocketTimeout = 1000 * 60 * 2;
    // timeout for private ip addresses.
    // 2 sec should be enough for a LAN
    public int privateSocketTimeout = 2000;

    /**
     * Timeout for network host connections.
     */
    public int mNetConnectionTimeout = 30000;

    /**
     * the time after which a automatic candidate search times out
     */
    public int searchRetryTimeout = 30000;

    public Vector				mSearchFilterTokens = new Vector();

    private String				mCfgFilename;
    private Properties			mSetting = new Properties();
    private Hashtable			mOldNames = new Hashtable();

    private Cfg()
    {
        // disable
    }


    public Cfg(String cfgFilename)
    {
        setupOldNames();
        mCfgFilename = cfgFilename;
    }


    public String getConfigFilename()
    {
        return mCfgFilename;
    }


    public void load()
    {
        try
        {
            FileInputStream is = new FileInputStream(mCfgFilename);

            mSetting.load(is);
            is.close();
        }
        catch (Exception e)
        {
            e.printStackTrace();
            System.out.println(e);
        }

        deserializeSimpleFields();
        deserializeComplexFields();

        breakSearchFilter();

        handlePhexVersionAdjustments();

        // ListeningPort doesn't exist.  Randomly generate one on the first time.
        if (mListeningPort == -1)
        {
            Random random = new Random(System.currentTimeMillis());
            mListeningPort = random.nextInt();
            mListeningPort = mListeningPort < 0 ? -mListeningPort : mListeningPort;
            mListeningPort %= 8000;		// resolve: expand to 25000
            mListeningPort += 2000;
        }

        // clean up old names hash to save memory
        mOldNames.clear();
        mOldNames = null;
    }


    public void save()
    {
        mSetting.clear();

        serializeSimpleFields();
        serializeComplexField();
        breakSearchFilter();

        try
        {
            FileOutputStream os = new FileOutputStream( mCfgFilename );
            mSetting.save(os, "PHEX Config Values");
            os.close();
        }
        catch (IOException e)
        {
            e.printStackTrace();
            System.out.println(e);
        }
    }


    private String get(String key)
    {
        String value = (String)mSetting.get(key);

        if (value == null)
        {
            // Load value by old name.
            String	oldName = (String)mOldNames.get(key);
            if (oldName != null)
            {
                value = (String)mSetting.get(oldName);
            }
        }

        if (value != null)
            value = value.trim();
        return value;
    }


    private String get(String key, String defaultVal)
    {
        String	value = get(key);

        if (value == null)
            return defaultVal;
        return value;
    }

/*
    private int getInt(String key, int defaultVal)
    {
        try
        {
            String	value = get(key);
            if (value == null)
                return defaultVal;
            else
                return Integer.parseInt(value);
        }
        catch (Exception e)
        {
            return defaultVal;
        }
    }
*/


    private boolean getBool(String key, boolean defaultVal)
    {
        return get(key, defaultVal ? "true" : "false").equals("true");
    }


    private void set(String key, String value)
    {
        mSetting.put(key, value);
    }

    private void set(String key, long value)
    {
        mSetting.put(key, String.valueOf(value));
    }

    private void set(String key, boolean value)
    {
        mSetting.put(key, value ? "true" : "false");
    }


    private void breakSearchFilter()
    {
        StringTokenizer	tokens = new StringTokenizer(mSearchFilter);

        mSearchFilterTokens.removeAllElements();
        while (tokens.hasMoreTokens())
        {
            mSearchFilterTokens.addElement(tokens.nextToken());
        }
    }


    private void serializeSimpleFields()
    {
        Field[]			fields = this.getClass().getDeclaredFields();

        for (int i = 0; i < fields.length; i++)
        {
            String		name = fields[i].getName();
            int			modifiers = fields[i].getModifiers();
            Class		type = fields[i].getType();

            if (!Modifier.isPublic(modifiers) ||
                Modifier.isTransient(modifiers) ||
                Modifier.isStatic(modifiers))
            {
                continue;
            }

            try
            {
                if (type.getName().equals("int"))
                {
                    set(name, fields[i].getInt(this));
                }
                else if (type.getName().equals("long"))
                {
                    set(name, fields[i].getLong(this));
                }
                else if (type.getName().equals("boolean"))
                {
                    set(name, fields[i].getBoolean(this));
                }
                else if (type.getName().equals("java.lang.String"))
                {
                    set(name, (String)fields[i].get(this));
                }
 /*
                else
                {
                    throw new Exception( "Don't know how to serialize a " + type.getName() + " field!"  );
                }
  */
            }
            catch (Exception e)
            {
                System.out.println(name + " " + e);
            }
        }
    }


    private void serializeComplexField()
    {
        try
        {
            set("mProgramClientID", mProgramClientID.toHexString());

            String	ignoredHosts = "";
            for (int i = 0; i < mNetIgnoredHosts.size(); i++)
            {
                String[]	parts = (String[])mNetIgnoredHosts.elementAt(i);
                ignoredHosts += (parts[0] + "." + parts[1] + "." + parts[2] + "." + parts[3] + " ");
            }
            set("mNetIgnoredHosts", ignoredHosts);

            String	filteredSearchHosts = "";
            for (int i = 0; i < mFilteredSearchHosts.size(); i++)
            {
                String[]	parts = (String[])mFilteredSearchHosts.elementAt(i);
                filteredSearchHosts += (parts[0] + "." + parts[1] + "." + parts[2] + "." + parts[3] + " ");
            }
            set("mFilteredSearchHosts", filteredSearchHosts);

            String	invalidHosts = "";
            for (int i = 0; i < mNetInvalidHosts.size(); i++)
            {
                String[]	parts = (String[])mNetInvalidHosts.elementAt(i);
                invalidHosts += (parts[0] + "." + parts[1] + "." + parts[2] + "." + parts[3] + " ");
            }
            set("mNetInvalidHosts", invalidHosts);

            String	networkHistory = "";
            for (int i = 0; i < mNetNetworkHistory.size(); i++)
            {
                networkHistory += mNetNetworkHistory.elementAt(i) + " ";
            }
            set("mNetNetworkHistory", networkHistory);

            String			networkPasswords = "";
            Enumeration		enum = mNetworkPasswords.keys();
            while (enum.hasMoreElements())
            {
                String		network = (String)enum.nextElement();
                String		pass = (String)mNetworkPasswords.get(network);

                networkPasswords += network + ", " + pass + ", ";
            }
            set("mNetworkPasswords", networkPasswords);

            set("mFontMenu", mFontMenu.getName() + ";" + mFontMenu.getStyle() + ";" + mFontMenu.getSize());
            set("mFontLabel", mFontLabel.getName() + ";" + mFontLabel.getStyle() + ";" + mFontLabel.getSize());
            set("mFontTable", mFontTable.getName() + ";" + mFontTable.getStyle() + ";" + mFontTable.getSize());

            breakSearchFilter();
        }
        catch (Exception e)
        {
            System.out.println(e);
        }
    }


    private void deserializeSimpleFields()
    {
        Field[] fields = this.getClass().getDeclaredFields();

        for (int i = 0; i < fields.length; i++)
        {
            String		name = fields[i].getName();
            int			modifiers = fields[i].getModifiers();
            Class		type = fields[i].getType();
            String		value;

            if (!Modifier.isPublic(modifiers) ||
                Modifier.isTransient(modifiers) ||
                Modifier.isStatic(modifiers))
            {
                continue;
            }

            try
            {
                // Load value by field name.
                value = get(name);
                if (value == null)
                {
                    // Use default.
                    continue;
                }

                if (type.getName().equals("int"))
                {
                    fields[i].setInt(this, Integer.parseInt(value));
                }
                else if (type.getName().equals("long"))
                {
                    fields[i].setLong(this, Long.parseLong(value));
                }
                else if (type.getName().equals("boolean"))
                {
                    fields[i].setBoolean(this, value.equals("true"));
                }
                else if (type.getName().equals("java.lang.String"))
                {
                    fields[i].set(this, value);
                }
            }
            catch (Exception e)
            {
                System.out.println(name + " " + e);
            }
        }
    }


    private void deserializeComplexFields()
    {
        try
        {
            try
            {
                mProgramClientID.fromHexString(get("mProgramClientID"));
            }
            catch (Exception e)
            {
                // ignore.  take the default created value as new client ID.
            }

            String	ignoredHosts = get("mNetIgnoredHosts", "");
            {
                StringTokenizer	tokens = new StringTokenizer(ignoredHosts);
                while (tokens.hasMoreTokens())
                {
                    mNetIgnoredHosts.addElement( IPUtils.splitIP2Parts( tokens.nextToken() ) );
                }
            }

            String	filteredSearchHosts = get("mFilteredSearchHosts", "");
            if (filteredSearchHosts == null)
            {
                mFilteredSearchHosts = getDefaultFilteredHosts();
            }
            else
            {
                StringTokenizer	tokens = new StringTokenizer(filteredSearchHosts);
                while (tokens.hasMoreTokens())
                {
                    mFilteredSearchHosts.addElement( IPUtils.splitIP2Parts(tokens.nextToken()));
                }
            }

            String	invalidHosts = get("mNetInvalidHosts");
            if (invalidHosts == null)
            {
                mNetInvalidHosts = getDefaultInvalidHosts();
            }
            else
            {
                StringTokenizer	tokens = new StringTokenizer(invalidHosts);
                while (tokens.hasMoreTokens())
                {
                    mNetInvalidHosts.addElement(IPUtils.splitIP2Parts(tokens.nextToken()));
                }
            }

            String	networkHistory = get("mNetNetworkHistory", "");
            {
                StringTokenizer	tokens = new StringTokenizer(networkHistory);
                while (tokens.hasMoreTokens())
                {
                    mNetNetworkHistory.addElement(tokens.nextToken());
                }
            }

            mNetworkPasswords.clear();
            String	networkPasswords = get("mNetworkPasswords", "");
            try
            {
                StringTokenizer	tokens = new StringTokenizer(networkPasswords, ",");
                while (tokens.hasMoreTokens())
                {
                    String		network = tokens.nextToken().trim();
                    String		pass = tokens.nextToken().trim();

                    mNetworkPasswords.put(network, pass);
                }
            }
            catch (Exception e)
            {
                // ignore
            }

            String	font;
            font = get("mFontMenu");
            if (font != null)
            {
                try
                {
                    StringTokenizer	tokens = new StringTokenizer(font, ";");
                    mFontMenu = new Font(tokens.nextToken(),
                                         Integer.parseInt(tokens.nextToken()),
                                         Integer.parseInt(tokens.nextToken()));
                }
                catch (Exception e)
                {
                    // ignore
                }
            }
            font = get("mFontLabel");
            if (font != null)
            {
                try
                {
                    StringTokenizer	tokens = new StringTokenizer(font, ";");
                    mFontLabel = new Font(tokens.nextToken(),
                                         Integer.parseInt(tokens.nextToken()),
                                         Integer.parseInt(tokens.nextToken()));
                }
                catch (Exception e)
                {
                    // ignore
                }
            }
            font = get("mFontTable");
            if (font != null)
            {
                try
                {
                    StringTokenizer	tokens = new StringTokenizer(font, ";");
                    mFontTable = new Font(tokens.nextToken(),
                                         Integer.parseInt(tokens.nextToken()),
                                         Integer.parseInt(tokens.nextToken()));
                }
                catch (Exception e)
                {
                    // ignore
                }
            }

        }
        catch (Exception e)
        {
//			System.out.println(e);
        }
    }


    private void setupOldNames()
    {
        mOldNames.put("mProgramClientID", "Program.CientID");
        mOldNames.put("mNetIgnoredHosts", "Net.IgnoredHosts");
        mOldNames.put("mNetFilteredSearchHosts", "Net.FilteredSearchHosts");
        mOldNames.put("mNetInvalidHosts", "Net.InvalidHosts");
        mOldNames.put("mNetNetworkHistory", "Net.NetworkHistory");

        mOldNames.put("mMyIP", "Net.MyIP");
        mOldNames.put("mListeningPort", "Net.ListeningPort");
        mOldNames.put("mMaxDownload", "Download.MaxDownload");
        mOldNames.put("mMaxDownloadPerIP", "Download.MaxDownloadPerIP");
        mOldNames.put("mDownloadMaxBandwidth", "Download.MaxBandwidth");
        mOldNames.put("mMaxUpload", "Upload.MaxUpload");
        mOldNames.put("mMaxUploadPerIP", "Upload.MaxUploadPerIP");
        mOldNames.put("mUploadMaxBandwidth", "Upload.MaxBandwidth");
        mOldNames.put("mNetTTL", "Net.TTL");
        mOldNames.put("mNetMaxTTL", "Net.MaxTTL");
        mOldNames.put("mNetMaxSendQueue", "Net.MaxSendQueue");
        mOldNames.put("mDisconnectDropRatio", "Disconnect.DropRatio");
        mOldNames.put("mDisconnectLatency", "Disconnect.Latency");
        mOldNames.put("mPingFrequency", "Net.PingFrequency");
        mOldNames.put("mPingTimeout", "Net.PingTimeout");
        mOldNames.put("mBehindFirewall", "Net.BehindFirewall");
        mOldNames.put("isBehindFirewall", "mBehindFirewall" );
        mOldNames.put("mSearchMaxSearch", "Search.MaxSearch");
        mOldNames.put("mSearchFilter", "Search.Filter");
        mOldNames.put("mSearchEnableMonitor", "Search.EnableMonitor");
        mOldNames.put("mNetMaxConnection", "Net.MaxConnection");
        mOldNames.put("mNetMaxRate", "Net.MaxRate");
        mOldNames.put("mNetConnectionTimeout", "Net.ConnectionTimeout");
        mOldNames.put("mDownloadAutoRemoveCompleted", "Download.AutoRemoveCompleted");
        mOldNames.put("mUploadAutoRemoveCompleted", "Upload.AutoRemoveCompleted");
        mOldNames.put("mDownloadDir", "Download.Dir");
        mOldNames.put("mAutoSearchCandidate", "Download.AutoSearchCandidate");
        mOldNames.put("mAutoConnect", "Net.AutoConnect");
        mOldNames.put("mNetMinConn", "Net.MinConn");
        mOldNames.put("mAutoCleanup", "Net.AutoCleanup");
        mOldNames.put("mUploadMaxSearch", "Upload.MaxSearch");
        mOldNames.put("mShareEnabled", "Share.Enabled");
        mOldNames.put("mShareBrowseDir", "Share.BrowseDir");
        mOldNames.put("mPushTransferTimeout", "Download.PushTransferTimeout");
        mOldNames.put("mApplyFilterdHosts", "Net.ApplyFilterdHosts");
        mOldNames.put("mCurrentNetwork", "Net.CurrentNetwork");
        mOldNames.put("mAutoJoin", "Net.AutoJoin");
        mOldNames.put("mSaveMyPassword", "Host.SaveMyPassword");
        mOldNames.put("mIndexFiles", "Upload.IndexFiles");
        mOldNames.put("mProxyUse", "Proxy.Use");
        mOldNames.put("mProxyHost", "Proxy.Host");
        mOldNames.put("mProxyPort", "Proxy.Port");
        mOldNames.put("mProxyUserName", "Proxy.UserName");
        mOldNames.put("mProxyPassword", "Proxy.Password");
        mOldNames.put("mLFClassName", "LF.ClassName");
        mOldNames.put("mUserName", "User.Name");
        mOldNames.put("mUploadDir", "Upload.Dir");
        mOldNames.put("mUploadFileExclusions", "Upload.FileExclusions");
        mOldNames.put("mUploadFileInclusions", "Upload.FileInclusions");
        mOldNames.put("mUploadScanRecursively", "Upload.ScanRecursively");
        mOldNames.put("mUploadAutoRemoveCompleted", "Upload.AutoRemoveCompleted");
        mOldNames.put("mHostFile", "HostFile");
        mOldNames.put("mLogFile", "LogFile");
        mOldNames.put("mDebugFile", "DebugFile");
        mOldNames.put("mDownloadSaveFile", "DownloadSaveFile");
        mOldNames.put("mSearchLastSearch", "Search.LastSearch");
        mOldNames.put("mSearchMinSpeed", "Search.MinSpeed");
        mOldNames.put("mFontMenu", "Font.Menu");
        mOldNames.put("mFontLabel", "Font.Label");
        mOldNames.put("mFontTable", "Font.Table");

    }

    public static Vector getDefaultInvalidHosts()
    {
        Vector	hosts = new Vector();

        hosts.addElement(IPUtils.splitIP2Parts("0.0.0.0"));
        hosts.addElement(IPUtils.splitIP2Parts("255.*.*.*"));
        hosts.addElement(IPUtils.splitIP2Parts("10.0.0.*"));
        hosts.addElement(IPUtils.splitIP2Parts("172.16.*.*"));
        hosts.addElement(IPUtils.splitIP2Parts("192.168.*.*"));

        return hosts;
    }


    public static Vector getDefaultFilteredHosts()
    {
        Vector	hosts = new Vector();

        hosts.addElement(IPUtils.splitIP2Parts("0.0.0.0"));
        hosts.addElement(IPUtils.splitIP2Parts("255.*.*.*"));
        hosts.addElement(IPUtils.splitIP2Parts("10.0.0.*"));
        hosts.addElement(IPUtils.splitIP2Parts("172.16.*.*"));
        hosts.addElement(IPUtils.splitIP2Parts("192.168.*.*"));

        return hosts;
    }


    public void handlePhexVersionAdjustments()
    {
        // either first time running Phex or update from Phex <= 0.5.6 to Phex 0.6
        if ( runningPhexVersion == null || runningPhexVersion.length() == 0 )
        {
            // raise mNetMaxHostToCatch to a higher value for
            // old Phex installations.
            if ( mNetMaxHostToCatch < 5000 )
            {
                mNetMaxHostToCatch = 5000;
            }
            // we are at 0.6 level now
            runningPhexVersion = "0.6";
        }
        // we have a phex version now check version updates
        // update from 0.6/0.6.1 to 0.6.2
        if ( runningPhexVersion.equals( "0.6" ) || runningPhexVersion.equals( "0.6.1" ))
        {
            updatesFor0_6_2();
        }

        runningPhexVersion = Res.getStr( "Program.Version" );
        save();
    }

    private void updatesFor0_6_2()
    {
        useProxyAuthentication = mProxyUserName.length() > 0;

        // adjust bandwidth to new settings.
        // find highest bandwidth setting...
        int maxSpeed = Math.max( mNetMaxRate,
            Math.max( mUploadMaxBandwidth, mDownloadMaxBandwidth ) );
        // calc required Kbps ( converting bytes to KB and then to Kbps
        int kbps = maxSpeed / 1024 * 8;
        networkSpeedKbps = kbps;
        maxTotalBandwidth = maxSpeed;
    }
}
