
/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package phex.connection;

import java.net.*;
import java.io.*;
import java.util.*;
import phex.FlexBuf;
import phex.ServiceManager;
import phex.HttpRequest;
import phex.host.Host;
import phex.common.HandshakeHeaders;
import phex.utils.IOUtil;
import phex.utils.VersionUtils;
import phex.utils.Digest;
import phex.utils.HexDec;

public class ConnectionRequest extends HttpRequest
{
    private static final String PROTOCOL_04 = "0.4";
    private static final String	GNUTELLA_OK_04 = "GNUTELLA OK";
    private static final String	GNUTELLA_OK_06 = "GNUTELLA/0.6 200 OK";

    public static final String sGnutellaConnect = "GNUTELLA CONNECT";
    public static final String sGet = "GET";
    public static final String sGiv = "GIV";
    private static final String sPasswordChallege = "GNUTELLA AUTHENTICATION CHALLENGE";
    private static final String sPasswordReply = "GNUTELLA AUTHENTICATION RESPONSE";
    private static final String sAuthFailed = "GNUTELLA AUTHENTICATION FAILED";


    private Host remoteHost;
    private FlexBuf flexBuffer;

    public ConnectionRequest( Host host )
            throws Exception
    {
        super();
        remoteHost = host;
        flexBuffer = new FlexBuf();
    }

    public void initialize()
        throws Exception
    {
        // GNUTELLA CONNECT/0.x\n\n
        // GET /get/1/foo.txt HTTP/1.1\r\n
        // GIV <file-ref-num>:<ClientID GUID in hexdec>/<filename>\n\n
        InputStream is = remoteHost.getIs();
        if (is == null)
        {
            // IS from getIs() has been cleared when connection closed.
            throw new IOException("Disconnected from remote host during initial handshake");
        }

        String requestLine = readLineFromHost();
        String greeting = ServiceManager.getNetworkManager().getGreeting();
        if ( requestLine.startsWith( greeting + "/" ) )
        {
            handleGnutellaConnect( requestLine );
        }
        // used from PushWorker
        else if (requestLine.startsWith(sGet + " "))
        {
            // requestLine = GET /get/1/foo doo.txt HTTP/1.1
            setMethod(sGet);

            int getIdx = requestLine.indexOf( "/get/" );
            setUri( requestLine.substring( getIdx, requestLine.length() ) );
            int versionIdx = requestLine.indexOf( "HTTP/" ) + 5;
            setVersion( requestLine.substring( versionIdx, requestLine.length() ) );
            // TODO maybe read headers here instead in shareMgr.httpRequestHandler
            //readHeaders( is );
            return;
        }
        // used when requesting push transfer
        else if (requestLine.startsWith(sGiv + " "))
        {
            // GIV <file-ref-num>:<ClientID GUID in hexdec>/<filename>\n\n
            setMethod(sGiv);
            String modLine = requestLine.substring(4); // skip GIV
            StringTokenizer	tokens = new StringTokenizer(modLine, ":");
            putHeader("FileRef", tokens.nextToken());
            StringTokenizer	tokens2 = new StringTokenizer(tokens.nextToken(), "/");
            putHeader("ClientID", tokens2.nextToken());
            putHeader("Filename", tokens2.nextToken());

            return;
        }
        else
        {
            throw new Exception("Unknown request: " + requestLine );
        }
    }

    private void handleGnutellaConnect( String requestLine )
        throws Exception
    {
        if ( !ServiceManager.getHostManager().hasIncommingSlotsAvailable() )
        {
            // Exceed total connection limit (both incoming and outgoing).
            // Refuse additional incoming connection.
            throw new IOException( "No free slots." );
        }

        // "GNUTELLA CONNECT/0.x";
        // "GNUTELLA PCONNECT name/0.x";
        setMethod( sGnutellaConnect );
        int idx = requestLine.lastIndexOf( '/' ) + 1;
        String version = requestLine.substring( idx, requestLine.length() );
        setVersion( version );
        if ( version.equals( PROTOCOL_04 ) )
        {
            checkPassword();
            sendStringToHost( GNUTELLA_OK_04 + "\n\n" );
        }
        else if ( is06orHigher( version ) )
        {
            // read connect headers
            readHeaders( remoteHost.getIs() );
            // accept connection
            sendStringToHost( GNUTELLA_OK_06 + "\r\n" );
            // currently no headers
            sendHeadersToHost( HandshakeHeaders.OPEN_HEADERS );

            String response = readLineFromHost();
            if ( !response.equals( GNUTELLA_OK_06 ) )
            {
                throw new IOException( "Bad protocol response" );
            }

            // read vendor headers
            readHeaders( remoteHost.getIs() );
        }
    }

    private void sendHeadersToHost( Properties headers )
        throws Exception
    {
        Enumeration enum = headers.propertyNames();
        while ( enum.hasMoreElements() )
        {
            String key = (String)enum.nextElement();
            String value = headers.getProperty( key );
            sendStringToHost( key + ": " + value + "\r\n" );
        }
        sendStringToHost( "\r\n" );
    }

    private String readLineFromHost( )
        throws Exception
    {
        byte[] buf = flexBuffer.getBuf( 1024 );
        int lenRead = IOUtil.readToCRLF( remoteHost.getIs(), buf, 1024, 0 );
        StringBuffer strBuf = new StringBuffer();
        IOUtil.deserializeString(buf, 0, lenRead, strBuf);
        String line = strBuf.toString();
        return line;
    }

    private void sendStringToHost( String str )
        throws Exception
    {
        byte[] buf = flexBuffer.getBuf( str.length() );
        int len;
        // Send reply signature back to remote host
        len = IOUtil.serializeString( str, buf, 0 );
        remoteHost.getOs().write( buf, 0, len );
    }

    private boolean is06orHigher( String version )
    {
        int diff = VersionUtils.compare( version, PROTOCOL_04 );
        return ( diff >= 0 );
    }

    private void checkPassword()
            throws Exception
    {
        String currPassword = (String)ServiceManager.sCfg.mNetworkPasswords.get(
            ServiceManager.sCfg.mCurrentNetwork);

        if (currPassword != null && currPassword.length() > 0)
        {
            StringBuffer	strBuf = new StringBuffer();
            byte[] buf = flexBuffer.getBuf(1024);
            byte[]			key = Digest.generateChallengeKey();
            String			keyStr = HexDec.convertBytesToHexString(key);
            byte[]			myDigest = Digest.computePasswordDigest(currPassword, key);
            String			myDigestStr = HexDec.convertBytesToHexString(myDigest);

            int len = IOUtil.serializeString(sPasswordChallege + " " + keyStr + "\n\n", buf, 0);
            remoteHost.getOs().write(buf, 0, len);

            len = IOUtil.readToCRLF( remoteHost.getIs(), buf, 1024, 0 );
            IOUtil.deserializeString(buf, 0, len, strBuf);
            String			reply = strBuf.toString();

            if (reply.startsWith(sPasswordReply))
            {
                String		clientDigestStr = reply.substring(sPasswordReply.length() + 1);
                clientDigestStr.toUpperCase();

                if (!myDigestStr.equals(clientDigestStr))
                {
                    len = IOUtil.serializeString(sAuthFailed + "\n\n", buf, 0);
                    remoteHost.getOs().write(buf, 0, len);

                    // Wait a bit before closing the connection.
                    // Somehow the remote gnutella won't read the last
                    // buffer if closing the connection too soon.
                    Thread.sleep(2000);

                    throw new Exception("Client password not matching.");
                }
            }
        }
    }
}