
/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2000 William W. Wong
 *  williamw@jps.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package phex.dialogues;


import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

import phex.*;
import phex.gui.common.TextFieldFactory;

/**
 *	The DlgInput1Str class is a generic modal dialog box for accepting one 
 *	input from the user.  The usage is to first create a DlgInut1Str object,
 *	passing in all the parameters to the constructor.  Next, call 
 *	setVisible(true) to display the dialog box.  When the user hits OK or
 *	Cancel, the dialog box will be closed.  Call getCanceled() to check
 *	whether the user canceled the dialog box.  Call getValue1() to get
 *	the user input.  Call dispose() to free the dialg box resource.
 */
public class DlgInput3Str extends JDialog implements KeyListener
{
	private BaseFrame			mFrame;
	private JTextField			mTextValue1;
	private JTextField			mTextValue2;
	private JTextField			mTextValue3;
	private String				mValue1;
	private String				mValue2;
	private String				mValue3;
	private boolean				mEmpty1;
	private boolean				mEmpty2;
	private boolean				mEmpty3;
	private boolean				mEditable1;
	private boolean				mEditable2;
	private boolean				mEditable3;
	private boolean				mCanceled = true;	// default to true.


	/**
	 *	Constructor to build a dialog box.
	 *
	 *	@param frame	the owner frame.
	 *	@param title	the title of the dialog box
	 *	@param label1	the label of the input field
	 *	@param value1	the default value (old value) of the input field.
	 *	@param editable	field is editable.
	 *	@param empty1	input can be empty.
	 *	@param label2	the label of the input field
	 *	@param value2	the default value (old value) of the input field.
	 *	@param editable	field is editable.
	 *	@param empty2	input can be empty.
	 *	@param label3	the label of the input field
	 *	@param value3	the default value (old value) of the input field.
	 *	@param editable	field is editable.
	 *	@param empty3	input can be empty.
	 */
	DlgInput3Str(BaseFrame frame, String title, String desc,
				 String label1, String value1, boolean editable1, boolean empty1,
				 String label2, String value2, boolean editable2, boolean empty2,
				 String label3, String value3, boolean editable3, boolean empty3)
	{
		// Call base class to set up modal dialog box.
		super(frame, true);
		mFrame = frame;
		mValue1 = value1;
		mValue2 = value2;
		mValue3 = value3;
		mEmpty1 = empty1;
		mEmpty2 = empty2;
		mEmpty3 = empty3;
		mEditable1 = editable1;
		mEditable2 = editable2;
		mEditable3 = editable3;
		setTitle(title);

		setDefaultCloseOperation(JDialog.HIDE_ON_CLOSE);

		// Set up panel and borders.
		JPanel		outPanel = new JPanel(new BorderLayout());
		outPanel.setBorder(
			BorderFactory.createCompoundBorder(
				BorderFactory.createCompoundBorder(
					BorderFactory.createEmptyBorder(20, 20, 20, 20),
					BorderFactory.createEtchedBorder()),
				BorderFactory.createEmptyBorder(16, 20, 20, 20)));

		JPanel		descPanel = new JPanel(new BorderLayout());
		descPanel.setBorder(BorderFactory.createEmptyBorder(10, 20, 0, 0));
		descPanel.add(BorderLayout.CENTER, new JLabel(desc));

		// Label panel
		JPanel		labelPanel = new JPanel(new GridLayout(3, 1, 0, 8));
		labelPanel.add(new JLabel(label1, JLabel.RIGHT));
		labelPanel.add(new JLabel(label2, JLabel.RIGHT));
		labelPanel.add(new JLabel(label3, JLabel.RIGHT));

		// Input panel
		JPanel		inputPanel = new JPanel(new GridLayout(3, 1, 0, 8));
		mTextValue1 = TextFieldFactory.newTextField(mValue1, 20);
		mTextValue2 = TextFieldFactory.newTextField(mValue2, 20);
		mTextValue3 = TextFieldFactory.newTextField(mValue3, 20);
		mTextValue1.setEditable(editable1);
		mTextValue2.setEditable(editable2);
		mTextValue3.setEditable(editable3);
		mTextValue1.addFocusListener(new FocusListenerHandler(mTextValue1, !editable1));
		mTextValue2.addFocusListener(new FocusListenerHandler(mTextValue2, !editable2));
		mTextValue3.addFocusListener(new FocusListenerHandler(mTextValue3, !editable3));
		inputPanel.add(mTextValue1);
		inputPanel.add(mTextValue2);
		inputPanel.add(mTextValue3);

		JPanel		topPanel = new JPanel();
		topPanel.add(BorderLayout.WEST, labelPanel);
		topPanel.add(BorderLayout.CENTER, inputPanel);

		outPanel.add(BorderLayout.CENTER, topPanel);

		getContentPane().add(BorderLayout.NORTH, descPanel);
		getContentPane().add(BorderLayout.CENTER, outPanel);

		// Listener for the OK and Cancel buttons.
		ActionListener	actionListener = new ActionListener()
		{ 
			public void actionPerformed(ActionEvent event)
			{ 
				if (!event.getActionCommand().equals("Cancel"))
				{
					if (!doOk())
						return;
					mCanceled = false;
				}
				setVisible(false);
			}
		};

		// Set up the OK and Cancel buttons.
		JPanel		buttonPanel = new JPanel();
		JButton		ok = new JButton("OK");
		JButton		cancel = new JButton("Cancel");
		ok.setMnemonic('O');
		ok.addActionListener(actionListener);
		cancel.addActionListener(actionListener);
		cancel.setMnemonic('C');
		buttonPanel.add(ok);
		buttonPanel.add(cancel);
		getContentPane().add(BorderLayout.SOUTH, buttonPanel);

		addKeyListener(this);

		pack();

		BaseFrame.centerWindow(this, new Point(0, 0));
	}


	/**
	 *	Returns the cancel flag.
	 *
	 *	@return		true = user canceled, false = user hits OK.
	 */
	boolean getCanceled()
	{
		return mCanceled;
	}


	/**
	 *	Returns the input value.
	 *
	 *	@return		the input value.
	 */
	String getValue1()
	{
		return mValue1;
	}


	/**
	 *	Returns the input value.
	 *
	 *	@return		the input value.
	 */
	String getValue2()
	{
		return mValue2;
	}


	/**
	 *	Returns the input value.
	 *
	 *	@return		the input value.
	 */
	String getValue3()
	{
		return mValue3;
	}



	private boolean doOk()
	{
		mValue1 = mTextValue1.getText().trim();
		mValue2 = mTextValue2.getText().trim();
		mValue3 = mTextValue3.getText().trim();

		if (!mEmpty1 && mValue1.length() == 0)
		{
			JOptionPane.showMessageDialog(mFrame,
										  "Please enter a value.", "Input Needed",
										  JOptionPane.ERROR_MESSAGE);
			mTextValue1.requestFocus();
			return false;
		}
		if (!mEmpty2 && mValue2.length() == 0)
		{
			JOptionPane.showMessageDialog(mFrame,
										  "Please enter a value.", "Input Needed",
										  JOptionPane.ERROR_MESSAGE);
			mTextValue2.requestFocus();
			return false;
		}
		if (!mEmpty3 && mValue3.length() == 0)
		{
			JOptionPane.showMessageDialog(mFrame,
										  "Please enter a value.", "Input Needed",
										  JOptionPane.ERROR_MESSAGE);
			mTextValue3.requestFocus();
			return false;
		}

		return true;
	}


	// KeyListener implementation
	public void keyPressed(KeyEvent e)
	{
	}


	public void keyReleased(KeyEvent e)
	{
		if (e.getKeyCode() == KeyEvent.VK_ESCAPE)
		{
			mCanceled = true;
			setVisible(false);
		}
		else if (e.getKeyCode() == KeyEvent.VK_ENTER)
		{
			if (!doOk())
				return;
			mCanceled = false;
			setVisible(false);
		}
	}


	public void keyTyped(KeyEvent e)
	{
	}


	// FocusListener
	private class FocusListenerHandler implements FocusListener
	{
		private Component	mComponent;
		private boolean		mSkip;


		FocusListenerHandler(Component component, boolean skip)
		{
			mComponent = component;
			mSkip = skip;
		}


		public void focusGained(FocusEvent e)
		{
			if (mSkip)
			{
				mComponent.transferFocus();
			}
		}

		
		public void focusLost(FocusEvent e)
		{
			
		}

	}

}


