
/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Konrad Haenel ( www.konrad-haenel.de )
 *                     Mark Saltzman ( www.marksaltzman.com )
 *                     Gregor Koukkoullis ( phex@kouk.de )
 *  Copyright (C) 2000 William W. Wong ( williamw@jps.net )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.download;

import java.io.*;
import java.net.*;
import java.util.*;
import javax.swing.SwingUtilities;

import phex.*;
import phex.common.FileHandlingException;
import phex.common.TransferDataProvider;
import phex.config.*;
import phex.download.*;
import phex.interfaces.*;
import phex.msg.*;
import phex.utils.*;
import phex.query.Search;
import phex.query.ResearchSetting;
import phex.query.BackgroundSearchContainer;
import phex.event.DownloadCandidatesChangeListener;
import phex.xml.XMLDownloadFile;
import phex.xml.XMLRemoteFile;

public class DownloadFile implements Serializable, TransferDataProvider
{
    public static final int	sQueued = 1;
    public static final int	sDownloading = 2;
    public static final int	sNoCandidates = 3;
    public static final int	sCompleted = 4;
    public static final int	sError = 5;
    /**
     * The HOST_BUSY status is handled like an error status for displaying currently
     */
    public static final int	HOST_BUSY = 6;
    public static final int	sStopping = 7;
    public static final int	sStopped = 8;
    public static final int	sRemoved = 9;
    public static final int	sConnecting = 10;
    public static final int	sRequestPushTransfer = 11;
    public static final int	sErrorMaxRetry = 12;
    public static final int	sRetryWait = 13;

    private String			mFilename;  // Fields to identify the file
    private long			mFileSize;  // Fields to identify the file
    private String			mShortname; // Filename without the path info for constructing local filename to save.
    private ArrayList mRemoteCandidates;
    private int				mCurrCandidate;
    private int				mStatus = sQueued;
    private String			mStatusText = "";
    private long			mStatusTime = 0;
    private String			mLog = "";
    private int				mStartSize = 0;
    private long			mTransferredSize = 0;
    private DownloadWorker	mDownloadWorker = null;
    private int				mFileOffset = 0;
    private String			mRemoteAppName = null;
    private DownloadManager	mDownloadMgr;
    private int				mRetryLast = 0;
    private long			mNextRetryTime = 0;
    private long lastSearchTime;
    private Integer currentProgress;
    private Integer retryCount;
    private int transferRate;
    private long transferRateTimestamp;
    private long startTime;
    private long stopTime;
    private int transferRateBytes;

    /**
     * Contains the filename in the local download directory.
     */
    private String localFilename;

    /**
     * Settings for the research.
     */
    private ResearchSetting researchSetting;

    /**
     * All listeners interested in events.
     */
    private ArrayList listenerList = new ArrayList( 2 );


    public DownloadFile()
    {
        this( 0, null, null );
    }

    public DownloadFile( XMLDownloadFile xfile )
    {
        this( xfile.getFilesize(), xfile.getFilename(), xfile.getFilename() );
        researchSetting.setSearchTerm( xfile.getSearchterm() );
        if ( xfile.getLocalFilename() != null )
        {
            localFilename = xfile.getLocalFilename();
        }
        // set default status
        mStatus = sNoCandidates;
        if ( xfile.getStatus() != null )
        {// update status
            if ( xfile.getStatus().equalsIgnoreCase(getStatusName(sCompleted)))
            {
                mStatus = sCompleted;
            }
            if ( xfile.getStatus().equalsIgnoreCase(getStatusName(sStopped)))
            {
                mStatus = sStopped;
            }
        }
        Iterator iterator = xfile.createXMLRemoteFileIterator();
        while ( iterator.hasNext() )
        {
            XMLRemoteFile xRemoteFile = (XMLRemoteFile) iterator.next();
            try
            {
                RemoteFile remoteFile = new RemoteFile( xRemoteFile );
                addRemoteCandidate( remoteFile );
            }
            catch (Exception exp )
            {
                //exp.printStackTrace();
            }
        }
    }

    /**
     * Default constructor for initialization should be called by every other
     * constructor.
     */
    public DownloadFile(long fileSize, String filename, String shortname)
    {
        mDownloadMgr = ServiceManager.getDownloadManager();
        ServiceManager.getTransferRateService().registerTransferDataProvider( this );
        mShortname = shortname;
        mFilename = filename;
        localFilename = shortname;
        mFileSize = fileSize;

        mCurrCandidate = -1;
        mStatus = sNoCandidates;
        lastSearchTime = 0;
        transferRate = 0;
        transferRateTimestamp = 0;
        transferRateBytes = 0;
        startTime = 0;
        stopTime = 0;
        mRemoteCandidates = new ArrayList();
        currentProgress = new Integer(0);
        retryCount = new Integer(0);
        researchSetting = new ResearchSetting( this );
        researchSetting.setSearchTerm( StrUtil.createNaturalSearchTerm( mShortname ) );
    }

    public DownloadFile( RemoteFile rfile )
    {
        this( rfile.getFileSize(), rfile.getFilename(), rfile.getFilename() );

        mRemoteCandidates.add( rfile );
        mStatus = sQueued;
        mCurrCandidate = 0;

        // Some servents just return the path along with the filename.
        // Path info is platform-dependent.  Get rid of it.
        int	i = mFilename.lastIndexOf("/");
        if (i != -1)
        {
            mShortname = mFilename.substring(i + 1);
        }
        else
        {
            i = mFilename.lastIndexOf("\\");
            if (i != -1)
            {
                mShortname = mFilename.substring(i + 1);
            }
        }

        researchSetting.setSearchTerm( StrUtil.createNaturalSearchTerm( mShortname ) );
        localFilename = mShortname;
    }

    public DownloadFile( RemoteFile rfile, String searchTerm,
        String aLocalFilename )
    {
        this( rfile );
        researchSetting.setSearchTerm( searchTerm );
        localFilename = aLocalFilename;
    }


    public String getFilename()
    {
        return (mFilename);
    }


    public String getShortname()
    {
        return (mShortname);
    }

    /**
     * Return the size of data that is attempting to be transfered. This is
     * NOT necessarily the full size of the file as could be the case during
     * a download resumption.
     */
    public long getTransferDataSize()
    {
        return mFileSize - mStartSize;
    }

    /**
     * This is way importent for researching and adding candidates!!
     */
    public long getTotalFileSize()
    {
        return mFileSize;
    }

    /**
     * Returns the progress in percent. If mStatus == sCompleted will always be 100%.
     */
    // TODO: This is exactly the same implementation as UploadFile. Should they be combined?
    public Integer getProgress()
    {
        int percentage;

        if( mStatus == sCompleted )
        {
            percentage = 100;
        }
        else
        {
            long toTransfer = getTransferDataSize();
            percentage = (int)(getTransferredDataSize() * 100L / (toTransfer == 0L ? 1L : toTransfer));
        }

        if ( currentProgress.intValue() != percentage )
        {
            // only create new object if necessary
            currentProgress = new Integer( percentage );
        }

        return currentProgress;
    }


    public int getCurrentCandidate()
    {
        return mCurrCandidate;
    }

    /**
    * Sets the candidate pointer to the next candidate in the list or
    * to 0 if the end of the list has been reached. Returns true if the
    * end of the list has been reached.
    **/
    public boolean advanceCurrCandidate()
    {
        mCurrCandidate++;
        if (mCurrCandidate >= mRemoteCandidates.size())
        {
            mCurrCandidate = 0;
            // Reach end of candidate list and warp around.
            return true;
        }

        // Not reach end of candidate list yet.
        return false;
    }

    /**
    * Returns the current candidate. If no candidates exist it returns
    * null.
    **/
    public RemoteFile getCurrentRemoteFile()
    {
        synchronized( mRemoteCandidates )
        {
            int curr = mCurrCandidate;
            if ( curr < mRemoteCandidates.size() &&
                 curr >= 0)
            {
                return (RemoteFile)mRemoteCandidates.get( curr );
            }
            return null;
        }
    }

    public int getRemoteCandidatesCount()
    {
        return mRemoteCandidates.size();
    }

    public RemoteFile getRemoteCandidateAt( int index )
    {
        synchronized( mRemoteCandidates )
        {
            if ( index < 0 || index >= mRemoteCandidates.size() )
            {
                return null;
            }
            return (RemoteFile) mRemoteCandidates.get( index );
        }
    }

    /**
     * Adds a candidate to the candidates list. Returns true if the remote
     * candidate was added to the list. False if the file size dosn't match or
     * the candidate is already in the list.
     */
    public boolean addRemoteCandidate( RemoteFile rFile )
    {
        // check file size... only allow files with right size
        if ( rFile.getFileSize() != mFileSize )
        {
            return false;
        }

        // Check for already existing candidate.
        if ( mRemoteCandidates.contains( rFile ) )
        {
            return false;
        }

        // Add as new candidate.
        int pos = mRemoteCandidates.size();
        mRemoteCandidates.add( pos, rFile );
        fireDownloadCandidateAdded( pos );
        return true;
    }

    /**
    * Sets the candidate pointer. If the pointer is out of range
    * it stays unchanged.
    **/
    public void setCandidateCurrent(int index)
    {
        if (index < 0 || index >= mRemoteCandidates.size() || index == mCurrCandidate)
        {
            return;
        }
        mCurrCandidate = index;
        fireDownloadCandidateChanged( index );
    }

    public void removeAllRemoteCandidates()
    {
        synchronized( mRemoteCandidates )
        {
            for ( int i = mRemoteCandidates.size() - 1; i >= 0; i-- )
            {
                removeRemoteCandidate( i );
            }
        }
    }

    /**
    * Removes the candidate at the given position. If the position
    * is out of range the candidates ArrayList is not changed.
    **/
    public void removeRemoteCandidate(int index)
    {
        synchronized( mRemoteCandidates )
        {
            if ( index < 0 || index >= mRemoteCandidates.size() )
            {
                return;
            }
            appendLog( "Removing Candidate: " + getRemoteCandidateAt( index ).getRemoteHost() );
            mRemoteCandidates.remove( index );
            fireDownloadCandidateRemoved( index );
            if (mCurrCandidate >= mRemoteCandidates.size())
            {
                if ( mRemoteCandidates.size() == 0 )
                {
                    mCurrCandidate = -1;
                    mStatus = sNoCandidates;
                }
                else
                {
                    mCurrCandidate = mRemoteCandidates.size() - 1;
                    fireDownloadCandidateChanged( mCurrCandidate );
                }
            }
        }
    }

    /**
     * Removes the current candidate.
     **/
    public void removeCurrentCandidate()
    {
        if ( mCurrCandidate >= 0 )
        {
            removeRemoteCandidate( mCurrCandidate );
        }
    }

    public String getSearchTerm()
    {
        return researchSetting.getSearchTerm();
    }


    public void setSearchTerm(String term)
    {
        researchSetting.setSearchTerm( term );
    }

    /**
     * Stops the current background search for the search term
     * used by this DownloadFile.
     **/
    public void stopSearchForCandidates()
    {
        researchSetting.stopSearch();
    }

    /**
     * This method is used when a user triggers a search from the user interface
     * it should not be used for any automatic searching! Automatic searching
     * is done vie the ResearchService class.
     */
    public void startSearchForCandidates()
    {
        researchSetting.stopSearch();
        // user triggered search with default timeout
        researchSetting.startSearch( Search.DEFAULT_SEARCH_TIMEOUT );
    }

    public boolean isSearchRunning()
    {
        return researchSetting.isSearchRunning();
    }

    public ResearchSetting getResearchSetting()
    {
        return researchSetting;
    }

    public String getRangeHeader(int startPos)
    {
        if (startPos == 0)
            return "";

//		return "Range: bytes=" + startPos + "-" (mFileSize - 1) + "\r\n";
        return "Range: bytes=" + startPos + "-\r\n";
    }

    public int getStatus()
    {
        return mStatus;
    }

    public short getDataTransferStatus()
    {
        switch ( mStatus )
        {
            case sDownloading:
                return TransferDataProvider.TRANSFER_RUNNING;
            case sError:
                return TransferDataProvider.TRANSFER_ERROR;
            case sCompleted:
                return TransferDataProvider.TRANSFER_COMPLETED;
            default:
                return TransferDataProvider.TRANSFER_NOT_RUNNING;
        }
    }

    public String getStatusName()
    {
        return getStatusName(mStatus);
    }


    public String getStatusName(int aStatus)
    {
        switch (aStatus)
        {
            case sQueued:
                return "queued...";

            case sDownloading:
                return "downloading...";

            case sNoCandidates:
                return "no candidates...";

            case sCompleted:
                return "COMPLETED";

            case sError:
                return  mStatusText;

            case HOST_BUSY:
                return "Host is busy.";

            case sStopping:
                return "stopping...";

            case sStopped:
                return "STOPPED";

            case sConnecting:
                return "connecting...";

            case sRequestPushTransfer:
                return "requesting push transfer...";

            case sErrorMaxRetry:
                return "Error: retried too many times.";

            case sRetryWait:
            {
                int sec = (int)((mNextRetryTime - System.currentTimeMillis()) / 1000);
                return "waiting for " + sec + " seconds...";
            }
        }
        return "Error: unknown status ( " + aStatus + " )";
    }


    public void setStatus(int status)
    {
        setStatus(status, "");
    }


    public void setStatus(int status, String text)
    {
        mStatus = status;
        mStatusText = text;
        mStatusTime = System.currentTimeMillis();

        mDownloadMgr.fireDownloadFileChanged( this );
    }

    public boolean isErrorStatusExpired()
    {
        if ( mStatus == sError || mStatus == HOST_BUSY )
        {
            if (System.currentTimeMillis() - mStatusTime > 5000)
            {
                return true;
            }
        }
        return false;
    }

    public String getRemoteAppName()
    {
        return mRemoteAppName;
    }


    public void setRemoteAppName(String appName)
    {
        mRemoteAppName = appName;
        mDownloadMgr.fireDownloadFileChanged( this );
    }


    public void setLog(String log)
    {
        mLog = log;
    }


    /**
    * Appends a new entry to the downloads log. If the log
    * exceeds 4089 characters, it is cut to 2048 characters.
    **/
    public void appendLog(String log)
    {
        if (mLog != null)
            mLog += log + "\n";
        else
            mLog = log + "\n";

        if (mLog.length() > 4096)
        {
            mLog = mLog.substring(2048);
        }
    }


    /**
    * Returns the current downloads log.
    **/
    public String getLog()
    {
        return mLog;
    }

    /**
     * Call this method to indicate where the download should start in the file
     */
    public void setStartSize(int size)
    {
        mStartSize = size;

        // We need to reset the transfer count in case this is a download resumption.
        mTransferredSize = 0;

    }

    // FIXME: Should setStartingTime and setStoppingTime be made part of TransferDataProvider?
    //        The same question should be answered for UploadFile.
    /**
     * Indicate that the download is just starting.
     */
    public void setStartingTime( long startingTime )
    {
        transferRateTimestamp = startingTime;

        startTime = startingTime;
        stopTime = 0;
    }

    /**
     * Indicate that the download is no longer running.
     */
    public void setStoppingTime( long stoppingTime )
    {
        // Ignore nested calls.
        if( stopTime == 0 )
        {
            stopTime = stoppingTime;
        }
    }

    public long getTransferredDataSize()
    {
        return mTransferredSize;
    }

    /**
     * Return the length of time, in seconds, that the download has been progressing.
     */
    public long getTransferTimeInSeconds()
    {
        // There are two cases; a download is still in progress, or it's been stopped.
        // If stopTime is non zero, then it's been stopped.
        if( stopTime != 0 )
        {
            return (stopTime - startTime) / 1000;
        }
        else
        {
            // Get current elapsed time and convert millis into seconds
            return (System.currentTimeMillis() - startTime) / 1000;
        }
    }

    public void setTransferredDataSize( long transferredSize )
    {
        long diff = transferredSize - mTransferredSize;
        transferRateBytes += diff;

        mTransferredSize = transferredSize;
        mDownloadMgr.fireDownloadFileChanged( this );
    }

    /**
     * To be able to provide a constantly valid data transfer rate the time
     * to calculate the data rate from needs to be updated.
     * If you want your timestamp to be updated regularly you need to register
     * your TransferDataProvider with the TransferRateService.
     */
    public void setTransferRateTimestamp( long timestamp )
    {
        transferRateTimestamp = timestamp;
        transferRateBytes = 0;
    }

    public int getDataTransferRate()
    {
        if ( transferRateTimestamp != 0 )
        {
            double sec = (System.currentTimeMillis() - transferRateTimestamp) / 1000;
            // don't drop transfer rate to 0 if we just have a new timestamp and
            // no bytes transfered
            if ( transferRateBytes > 0 || sec > 1 )
            {
                transferRate = (int) ( transferRateBytes / sec );
            }
        }
        return transferRate;
    }

    public Integer getRetryCountObject()
    {
        return retryCount;
    }

    public int getRetryCount()
    {
        return retryCount.intValue();
    }

    public void setRetryCount( int count )
    {
        if ( count != retryCount.intValue() )
        {
            retryCount = new Integer( count );
            mDownloadMgr.fireDownloadFileChanged( this );
        }
    }

    public int getRetryLast()
    {
        return mRetryLast;
    }


    public void setRetryLast(int count)
    {
        mRetryLast = count;
    }


    public long getNextRetryTime()
    {
        return mNextRetryTime;
    }


    public void setNextRetryTime(long nextRetryTime)
    {
        mNextRetryTime = nextRetryTime;
    }


    public DownloadWorker getDownloadWorker()
    {
        return mDownloadWorker;
    }

    public void setDownloadWorker(DownloadWorker worker)
    {
        mDownloadWorker = worker;
    }

    public boolean isDownloadInProgress()
    {
        return mDownloadWorker != null;
    }

    public boolean isDownloadCompleted()
    {
        return mStatus == sCompleted;
    }

    public boolean isDownloadStopped()
    {
        return mStatus == sStopped;
    }

    public String getLocalFilename()
    {
        return localFilename;
    }

    /**
     * Sets the new local filename of a file. You can specify if you like
     * to rename the old file or not. The method can throw a FileHandlingException
     * if you like to have it renamed. Otherwise it can be ignored.
     */
    public void setLocalFilename( String aLocalFilename, boolean rename )
        throws FileHandlingException
    {
        if ( rename && !aLocalFilename.equals( localFilename ))
        {
            // first check if download mgr has any running download with this
            // filename
            if ( mDownloadMgr.isNewLocalFilenameUsed( this, aLocalFilename ) )
            {
                // cant rename to file that already exists
                throw new FileHandlingException(
                    FileHandlingException.FILE_ALREADY_EXISTS );
            }
            DownloadFile.renameLocalFile( localFilename, aLocalFilename );
        }
        localFilename = aLocalFilename;
    }

    /**
     * We need to rename the local file and the download file accordingly...
     * TODO move to some global file handling class...
     */
    private static void renameLocalFile( String oldFilename, String newFilename )
        throws FileHandlingException
    {
        File oldLocalFile = new File( DownloadFile.getFullLocalFilename(
            oldFilename ) );
        File newLocalFile = new File( DownloadFile.getFullLocalFilename(
            newFilename ) );

        File oldDownloadFile = new File( DownloadFile.getDownloadFileName(
            oldFilename ) );
        File newDownloadFile = new File( DownloadFile.getDownloadFileName(
            newFilename ) );

        if ( newLocalFile.exists() || newDownloadFile.exists() )
        {
            // cant rename to file that already exists
            throw new FileHandlingException(
                FileHandlingException.FILE_ALREADY_EXISTS );
        }

        if ( oldLocalFile.exists() )
        {
            if ( !oldLocalFile.renameTo( newLocalFile ) )
            {
                // I don't know why it fails... :-(
                throw new FileHandlingException(
                    FileHandlingException.RENAME_FAILED );
            }
        }

        if ( oldDownloadFile.exists() )
        {
            if ( !oldDownloadFile.renameTo( newDownloadFile ) )
            {
                // I don't know why it fails... :-(
                throw new FileHandlingException(
                    FileHandlingException.RENAME_FAILED );
            }
        }
    }

    public String getFullLocalFilename()
    {
        return DownloadFile.getFullLocalFilename( localFilename );
    }

    private static String getFullLocalFilename( String filename )
    {
        String downloadDir = ServiceManager.sCfg.mDownloadDir;
        if ( downloadDir.endsWith( File.separator ) )
        {
            downloadDir = downloadDir.substring( 0, downloadDir.length() - 1 );
        }
        return (downloadDir + File.separator +
            StrUtil.convertToLocalSystemFilename( filename ) );
    }

    private static String getDownloadFileName( String filename )
    {
        return getFullLocalFilename( filename ) + ".dl";
    }

    public String getDownloadName()
    {
        // resolve: have shorter suffix because the Mac has a 31 character limit.
        // TODO no 31 char limit anymore
        return getFullLocalFilename() + ".dl";
//		return getLocalFilename() + ".download-" + mFileSize + "-" + mFileOffset;
    }


    public String toString()
    {
        return mFilename + "(" + mFileSize + ")";
    }

    public XMLDownloadFile createXMLDownloadFile()
    {
        XMLDownloadFile xfile = new XMLDownloadFile();
        xfile.setFilename( mFilename );
        xfile.setFilesize( mFileSize );
        xfile.setSearchterm( researchSetting.getSearchTerm() );
        xfile.setLocalFilename( localFilename );
        xfile.setStatus( getStatusName() );
        Iterator iterator = mRemoteCandidates.iterator();
        while ( iterator.hasNext() )
        {
            RemoteFile remoteFile = (RemoteFile) iterator.next();
            XMLRemoteFile xRemoteFile = remoteFile.createXMLRemoteFile();
            xfile.addXMLRemoteFile( xRemoteFile );
        }
        return xfile;
    }





    public void readFromOIS(java.io.DataInputStream dis, boolean firstTime)
            throws IOException, ClassNotFoundException
    {
        mDownloadMgr = ServiceManager.getDownloadManager();
        if (firstTime)
        {
            dis.skipBytes(536); //28*20
        }
        else
        {
            dis.skipBytes(5);
        }
        byte[] b={dis.readByte(),dis.readByte(),dis.readByte(),dis.readByte()} ;
        ServiceManager.log("Reading DownloadFile after: "+b[0]+" "+b[1]+" "+b[2]+" "+b[3]);
        if (b[2]== 122)
        {
            ServiceManager.log("found long entry exception");
            dis.skipBytes(3);
        }
        int	version = dis.readInt();
        ServiceManager.log("Reading DownloadFile Version: "+version);
        switch (version)
        {
            case 1:
                deserialize1(dis, firstTime);
                break;

            default:
                throw new IOException("Fail to deserialize datafile.  Unknown verison.");
        }
    }


    private void deserialize1(DataInputStream dis, boolean firstTime)
            throws IOException, ClassNotFoundException
    {
        mFilename = dis.readUTF();
        ServiceManager.log("Reading DownloadFile Item: "+mFilename);
        mFileSize = dis.readLong();
        mShortname = dis.readUTF();
        researchSetting.setSearchTerm( dis.readUTF() );
        mCurrCandidate = dis.readInt();
        if (firstTime)
        {
            dis.skipBytes(39); //28*20 279
        }
        else
        {
            dis.skipBytes(2);
        }
        byte[] b={dis.readByte(),dis.readByte(),dis.readByte(),dis.readByte()} ;
        ServiceManager.log("Reading DownloadFile after: "+b[0]+" "+b[1]+" "+b[2]+" "+b[3]);
        int	size = dis.readInt();
        ServiceManager.log("Reading DownloadFile RemoteFile Items: "+size);
        dis.skipBytes(5);
        for(int j=0;j<size;j++)
        {
            RemoteFile rFile= new RemoteFile(new GUID(), 0, mFilename,
                               (int)mFileSize, "", 0);;
            rFile.readFromOIS(dis, firstTime);
            firstTime= false;
            mRemoteCandidates.add(rFile);
        }
        dis.skipBytes(4);
        mStatus = dis.readInt();
        ServiceManager.log("Reading DownloadFile Status: "+mStatus);
        if (mStatus != sCompleted)
        {
            mStatus = sQueued;
        }
    }

    ///////////////////// START event handling methods ////////////////////////
    public void addDownloadCandidatesChangeListener(
        DownloadCandidatesChangeListener listener )
    {
        listenerList.add( listener );
    }

    public void removeDownloadCandidatesChangeListener(
        DownloadCandidatesChangeListener listener )
    {
        listenerList.remove( listener );
    }

    private void fireDownloadCandidateChanged( final int position )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                DownloadCandidatesChangeListener listener;
                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (DownloadCandidatesChangeListener)listeners[ i ];
                    listener.downloadCandidateChanged( position );
                }
            }
        });
    }

    private void fireDownloadCandidateAdded( final int position )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                DownloadCandidatesChangeListener listener;
                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (DownloadCandidatesChangeListener)listeners[ i ];
                    listener.downloadCandidateAdded( position );
                }
            }
        });
    }

    private void fireDownloadCandidateRemoved( final int position )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                DownloadCandidatesChangeListener listener;
                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (DownloadCandidatesChangeListener)listeners[ i ];
                    listener.downloadCandidateRemoved( position );
                }
            }
        });
    }

    public void fireDownloadCandidateChanged( RemoteFile candidate )
    {
        int position = mRemoteCandidates.indexOf( candidate );
        if ( position >= 0 )
        {
            fireDownloadCandidateChanged( position );
        }
    }

    /**
     * Returns the length of time, in seconds, that this transfer has
     * been TRANSFER_RUNNING so that one may calculate average
     * transfer rates.
 */


    ///////////////////// END event handling methods ////////////////////////
}
