/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2000 William W. Wong
 *  williamw@jps.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package phex.gui.common;


import java.awt.event.*;
import javax.swing.*;
import javax.swing.text.*;
import java.awt.Container;
import java.awt.Component;
import java.awt.AWTEvent;

/**
 * This includes a bugfix for a X focus problem.
 * Some references:
 * 4290675   Focus Management Enhancements
 * 4135827   Cursor not appearing in JTextField <br>
 * 4186928   focus events for Lightweights in Windows are not generated with requestFocus
 * 4284547   textField.requestFocus on window activation can leave 2 fields flashing
 * 4381959   Focus not handled correctly after two consecutive dialogs
 * 4351446   requestFocus() does not work on Windows NT
 * <p>
 * Just forget about a perfect fix. If focus manager doesn't work force override.
 * Else use focusmanager. If we can't remove focus we assume the focusmanager doesn't
 * work. This way phex will still be useable.
 */

public class TextFieldFix extends JTextField implements KeyListener, MouseListener
{
    private boolean mSkipKey = false;

    private static TextFieldFix lastFocusedComponent = null;

    private static TextFieldFix nowFocusedComponent = null;

    private static boolean focusInTextField = false;

    // some problems are unix or rather X specific.
    private static boolean mUnix = true;

    public TextFieldFix()
    {
        this( null, 0 );
    }

    public TextFieldFix( int columns )
    {
        this( null, columns );
    }

    public TextFieldFix( String text, int columns )
    {
        super(text, columns);
        addKeyListener(this);
        addMouseListener(this);
    }

    /**
     *  An initialization block that determines the operating system and sets the
     *  unix variable for unknown systems.
     */
    static {
        String osName = System.getProperty("os.name");
        if ("Mac OS".equals(osName)) {
            mUnix= false;
        } else if (osName.startsWith("Windows")) {
            mUnix= false;
        } else {
            // We have to use the "safe but not so nice way"
            mUnix= true;
        }
    }


    public static void setUnix(boolean unix)
    {
        mUnix = unix;
    }


    public boolean isManagingFocus()
    {
        return mUnix;
    }

    private void removeEveryFocus(Component c)
    {
        if (c instanceof TextFieldFix) {
            ((TextFieldFix)c).getCaret().setVisible(false);
            ((TextFieldFix)c).getCaret().setSelectionVisible(false);
        }
        if (c instanceof Container) {
            int n = ((Container)c).getComponentCount();
            for (int i = 0; i < n; i++) {
                removeEveryFocus(((Container)c).getComponent(i));
            }
        }
    }

    /**
     * This is the heart to the workaround. We set and reset focus very agressive.
     */
    private void unfocus(JTextField tf)
    {
        //if (Debug.DbgGui(4) && Debug.DbgAlex(4))
        //    Debug.msg("TextFieldFix refocus focus switch needed. Perform refocus.");

        if (tf.hasFocus())
            processEvent(new FocusEvent(tf,FocusEvent.FOCUS_LOST));
        Component focusOwner = SwingUtilities.findFocusOwner(getRootPane());
        if (focusOwner != null)
        {
            //if (Debug.DbgGui(4) && Debug.DbgAlex(4))
            //    Debug.msg("TextFieldFix refocus focusOwner still set.");
            JTextField dummy = new JTextField();
            getParent().add(dummy);
            dummy.grabFocus();
            getParent().remove(dummy);
        }
        focusOwner = SwingUtilities.findFocusOwner(getRootPane());
        if (focusOwner != null)
        {
            //if (Debug.DbgGui(4) && Debug.DbgAlex(4))
            //    Debug.msg("TextFieldFix refocus can't unfocus. Force visuals");
            Container c= getRootPane();
            removeEveryFocus(c);
        }
    }

    /**
     * This is the heart to the workaround. We set and reset focus very agressive.
     */
    private void focus(JTextField tf)
    {
        tf.grabFocus();
        processFocusEvent(new FocusEvent(tf,FocusEvent.FOCUS_GAINED));
        focusInTextField = true;
        tf.getCaret().setVisible(true);
        tf.getCaret().setSelectionVisible(true);
    }

    private void refocus(JTextField tf)
    {
        unfocus(tf);
        focus(tf);
    }

    // KeyListener interface
    public void keyTyped(KeyEvent e)
    {
        // If normal key was typed
        boolean altKey = (e.getKeyCode() == KeyEvent.VK_ALT) || mSkipKey;
        boolean tabKey = (e.getKeyCode() == KeyEvent.VK_TAB || e.getKeyChar() == '\t');
        if (mUnix && !altKey &&
            (!focusInTextField || ((Component)e.getSource() != nowFocusedComponent)))
        {
            // catch faulty focus
            JTextField tf = (JTextField) e.getSource();
            if (tabKey)
                unfocus(tf);
            focus(tf);
        }
    }


    public void keyReleased(KeyEvent e)
    {
    }


    public void keyPressed(KeyEvent e)
    {
        // If Alt pressed, set a flag for processInputMethodEvent() to skip.
        // Avoids bug in java 1.2.x
        mSkipKey = (e.isAltDown() && e.getKeyCode() != KeyEvent.VK_ALT);
        // If tab wa pressed
        boolean tabKey = (e.getKeyCode() == KeyEvent.VK_TAB || e.getKeyChar() == '\t');
        if (mUnix && tabKey)
        {
            // we do focus management by ourself now
            //if (Debug.DbgGui(4) && Debug.DbgAlex(4))
            //{
            //    Debug.msg("TextFieldFix keyPressed tab: "+e.toString().substring(0,130));
            //}

        if((e.getModifiers() & ActionEvent.SHIFT_MASK) == ActionEvent.SHIFT_MASK) {
          FocusManager.getCurrentManager().focusPreviousComponent(this);
        } else { FocusManager.getCurrentManager().focusNextComponent(this); }

            e.consume();

        }
    }

    // MouseListener interface
    public void mouseClicked(MouseEvent e){
        if (mUnix)
        {
            JTextField tf = (JTextField) e.getSource();
            Component focusOwner = SwingUtilities.findFocusOwner(getRootPane());
            refocus(tf);
        }
    }

    public void mousePressed(MouseEvent e){}

    public void mouseReleased(MouseEvent e){}

    public void mouseEntered(MouseEvent e){}

    public void mouseExited(MouseEvent e){}

    protected void processInputMethodEvent(InputMethodEvent e)

    {
        if (!mSkipKey)
        {
            super.processInputMethodEvent(e);
        }
        else
        {
            //if (Debug.DbgGui(4) && Debug.DbgAlex(4))
            //{
            //    Debug.msg("TextFieldFix processInputMethodEvent event skiped: "+e.toString().substring(0,130));
            //}
        }
    }

    /**
     * Processes events on this container. If the event is a ContainerEvent,
     * it invokes the processContainerEvent method, else it invokes its
     * superclass's processEvent.
     * @param e the event
     */
    protected void processEvent(AWTEvent e) {
        super.processEvent(e);
        // can't build on the event system here
        if (mUnix && e.getID() == FocusEvent.FOCUS_LOST)
        {
            if (e.getSource() instanceof TextFieldFix)
                lastFocusedComponent = (TextFieldFix)e.getSource();
            focusInTextField = false;
            getCaret().setVisible(false);
            getCaret().setSelectionVisible(false);
        }
        if (mUnix)
        {
            if (e.getID() == FocusEvent.FOCUS_GAINED)
            {
                Component focusOwner = SwingUtilities.findFocusOwner(getRootPane());
                if ((focusOwner != null) && (focusOwner != this))
                {
                    unfocus(this);
                }
                focusOwner = SwingUtilities.findFocusOwner(getRootPane());

                if (e.getSource() instanceof TextFieldFix)
                    nowFocusedComponent = (TextFieldFix)e.getSource();
                focusInTextField = true;
                getCaret().setVisible(true);
                getCaret().setSelectionVisible(true);
            }
            // remove faulty alt focus. Helps but didn't solve the problem.
            if (e instanceof KeyEvent)
            {
                KeyEvent ke = (KeyEvent)e;
                boolean altKey = (ke.getKeyCode() == KeyEvent.VK_ALT) || mSkipKey;
                // If Alt was pressed, remove focus. Workaround for lost focus events bug.
                // Usually happened when selecting phex with alt tab.
                if (altKey)
                {
                    if (hasFocus())
                        processEvent(new FocusEvent(this,FocusEvent.FOCUS_LOST));
                    if (nowFocusedComponent != null && (nowFocusedComponent).hasFocus())
                        processEvent(new FocusEvent(nowFocusedComponent,FocusEvent.FOCUS_LOST));
                }
            }
        }
    }
}
