/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2002 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.gui.dialogs.options;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import phex.ServiceManager;
import phex.config.*;
import phex.gui.common.*;
import phex.utils.*;

public class BandwidthPane extends OptionsSettingsPane
{
    private int UNLIMITED_VALUE = 101;

    /**
     * Modem = 56K Modem
     * ISDN = 64K ISDN
     * DualISDN = 128K ISDN
     * DSLCable1 = 256Kbps DSL / Cable
     * DSLCable2 = 512Kbps DSL / Cable
     * DSLCable3 = 512Kbps DSL / Cable
     * T1 = T1 (1.5 Mbps)
     * 10LAN = 10Mbps LAN
     * T3 = T3 (44 Mbps)
     * 100LAN = 100Mbps LAN
     */
    private SpeedDefinition[] speedDefinitions =
        {
            new SpeedDefinition( "Modem", 56 ),
            new SpeedDefinition( "ISDN", 64 ),
            new SpeedDefinition( "DualISDN", 128 ),
            new SpeedDefinition( "DSLCable1", 256 ),
            new SpeedDefinition( "DSLCable2", 512 ),
            new SpeedDefinition( "DSLCable3", 768 ),
            new SpeedDefinition( "T1", 1544 ),
            new SpeedDefinition( "10LAN", 10000 ),
            new SpeedDefinition( "T3", 44736 ),
            new SpeedDefinition( "100LAN", 100000 )
        };

    private Dictionary sliderLabels;

    private JComboBox connectionSpeedCbx;
    private JLabel totalBandwidthLabel;
    private JSlider totalBandwidthSldr;
    private JLabel netBandwidthLabel;
    private JSlider netBandwidthSldr;
    private JLabel downloadBandwidthLabel;
    private JSlider downloadBandwidthSldr;
    private JLabel uploadBandwidthLabel;
    private JSlider uploadBandwidthSldr;

    private double maxConnectionBandwidth;

    /**
     * Total bandwidth setting in bytes per second.
     * When bandwidth is set to unlimited the value is set to maxTotalBandwidth
     * for calculation reasons. To find out if set to unlimited check the
     * coresponding slider value for UNLIMITED_VALUE.
     */
    private double currentTotalBandwidth;

    /**
     * Network bandwidth setting in bytes per second.
     * When bandwidth is set to unlimited the value is set to maxTotalBandwidth
     * for calculation reasons. To find out if set to unlimited check the
     * coresponding slider value for UNLIMITED_VALUE.
     */
    private double currentNetBandwidth;

    /**
     * Download bandwidth setting in bytes per second.
     * When bandwidth is set to unlimited the value is set to maxTotalBandwidth
     * for calculation reasons. To find out if set to unlimited check the
     * coresponding slider value for UNLIMITED_VALUE.
     */
    private double currentDownloadBandwidth;

    /**
     * Upload bandwidth setting in bytes per second.
     * When bandwidth is set to unlimited the value is set to maxTotalBandwidth
     * for calculation reasons. To find out if set to unlimited check the
     * coresponding slider value for UNLIMITED_VALUE.
     */
    private double currentUploadBandwidth;

    public BandwidthPane()
    {
        super( "Bandwidth" );
        sliderLabels = new Hashtable( 5 );
        sliderLabels.put( new Integer( 0 ), new JLabel( "0%", JLabel.CENTER ) );
        sliderLabels.put( new Integer( 25 ), new JLabel( "25%", JLabel.CENTER ) );
        sliderLabels.put( new Integer( 50 ), new JLabel( "50%", JLabel.CENTER ) );
        sliderLabels.put( new Integer( 75 ), new JLabel( "75%", JLabel.CENTER ) );
        sliderLabels.put( new Integer( 100 ), new JLabel( "100%", JLabel.CENTER ) );
    }

    /**
     * Called when preparing this settings pane for display the first time. Can
     * be overriden to implement the look of the settings pane.
     */
    protected void prepareComponent()
    {
        GridBagConstraints constraints;
        setLayout( new GridBagLayout() );

        JPanel connectionPanel = new JPanel( new GridBagLayout() );
        connectionPanel.setBorder( BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(),
            Localizer.getString( "NetworkSpeedSettings" ) ) );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.BOTH;
            constraints.weightx = 1;
        add( connectionPanel, constraints );

        JLabel label = new JLabel( Localizer.getString( "SelectConnection" ) + ": ");
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.gridwidth = 2;
            constraints.anchor = GridBagConstraints.NORTHWEST;
        connectionPanel.add( label, constraints );

        connectionSpeedCbx = new JComboBox( speedDefinitions );
        connectionSpeedCbx.addItemListener( new SpeedItemListener() );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 1;
            constraints.gridwidth = 2;
            constraints.anchor = GridBagConstraints.NORTHEAST;
        connectionPanel.add( connectionSpeedCbx, constraints );

        label = new JLabel( Localizer.getString( "PhexBandwidth" ) + ": ");
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 2;
            constraints.anchor = GridBagConstraints.NORTHWEST;
        connectionPanel.add( label, constraints );

        totalBandwidthLabel = new JLabel(  );
            constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 2;
            constraints.anchor = GridBagConstraints.NORTHEAST;
            constraints.weightx = 1;
        connectionPanel.add( totalBandwidthLabel, constraints );

        totalBandwidthSldr = new JSlider( JSlider.HORIZONTAL, 0, 101, 0 );
        totalBandwidthSldr.addChangeListener( new SliderChangeListener() );
        totalBandwidthSldr.setLabelTable( sliderLabels );
        totalBandwidthSldr.setPaintLabels( true );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 3;
            constraints.gridwidth = 2;
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.weightx = 1;
        connectionPanel.add( totalBandwidthSldr, constraints );

        JPanel phexPanel = new JPanel( new GridBagLayout() );
        phexPanel.setBorder( BorderFactory.createTitledBorder(
            BorderFactory.createEtchedBorder(),
            Localizer.getString( "PhexBandwidthSettings" ) ) );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 1;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.BOTH;
            constraints.weightx = 1;
            constraints.weighty = 1;
        add( phexPanel, constraints );


        label = new JLabel( Localizer.getString( "MaxNetworkBandwidth" ) + ": ");
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
        phexPanel.add( label, constraints );

        netBandwidthLabel = new JLabel( );
            constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHEAST;
            constraints.weightx = 1;
        phexPanel.add( netBandwidthLabel, constraints );

        netBandwidthSldr = new JSlider( JSlider.HORIZONTAL, 0, 101, 0 );
        netBandwidthSldr.addChangeListener( new SliderChangeListener() );
        netBandwidthSldr.setLabelTable( sliderLabels );
        netBandwidthSldr.setPaintLabels( true );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 1;
            constraints.gridwidth = 2;
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.weightx = 1;
        phexPanel.add( netBandwidthSldr, constraints );

        label = new JLabel( Localizer.getString( "MaxDownloadBandwidth" ) + ": ");
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 2;
            constraints.anchor = GridBagConstraints.NORTHWEST;
        phexPanel.add( label, constraints );

        downloadBandwidthLabel = new JLabel( );
            constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 2;
            constraints.anchor = GridBagConstraints.NORTHEAST;
            constraints.weightx = 1;
        phexPanel.add( downloadBandwidthLabel, constraints );

        downloadBandwidthSldr = new JSlider( JSlider.HORIZONTAL, 0, 101, 0 );
        downloadBandwidthSldr.addChangeListener( new SliderChangeListener() );
        downloadBandwidthSldr.setPaintLabels( true );
        downloadBandwidthSldr.setLabelTable( sliderLabels );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 3;
            constraints.gridwidth = 2;
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.weightx = 1;
        phexPanel.add( downloadBandwidthSldr, constraints );

        label = new JLabel( Localizer.getString( "MaxUploadBandwidth" ) + ": ");
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 4;
            constraints.anchor = GridBagConstraints.NORTHWEST;
        phexPanel.add( label, constraints );

        uploadBandwidthLabel = new JLabel( );
            constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 4;
            constraints.anchor = GridBagConstraints.NORTHEAST;
            constraints.weightx = 1;
        phexPanel.add( uploadBandwidthLabel, constraints );

        uploadBandwidthSldr = new JSlider( JSlider.HORIZONTAL, 0, 101, 0 );
        uploadBandwidthSldr.addChangeListener( new SliderChangeListener() );
        uploadBandwidthSldr.setPaintLabels( true );
        uploadBandwidthSldr.setLabelTable( sliderLabels );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 5;
            constraints.gridwidth = 2;
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.weightx = 1;
            constraints.weighty = 1;
        phexPanel.add( uploadBandwidthSldr, constraints );

        initConfigValues();
    }

    /**
     * Override this method if you like to verify inputs before storing them.
     * A input dictionary is given to the pane. It can be used to store values
     * like error flags or prepared values for saving. The dictionary is given
     * to every settings pane checkInput(), displayErrorMessage() and
     * saveAndApplyChanges() method.
     * When the input has been flaged as invalid with the method setInputValid()
     * the method displayErrorMessage() is called directly after return of
     * checkInput() and the focus is given to settings pane.
     * After checking all settings pane without any error the method
     * saveAndApplyChanges() is called for all settings panes to save the
     * changes.
     */
    public void checkInput( HashMap inputDic )
    {// wrong input not possible...
    }

    /**
     * When isInputValid() returns a false this method is called.
     * The input dictionary should contain the settings pane specific information
     * of the error.
     * The settings pane should override this method to display a error
     * message. Before calling the method the focus is given to the
     * settings pane.
     */
    public void displayErrorMessage( HashMap inputDic )
    {// no error possible...
    }

    /**
     * Override this method if you like to apply and save changes made on
     * settings pane. To trigger saving of the configuration if any value was
     * changed call triggerConfigSave().
     */
    public void saveAndApplyChanges( HashMap inputDic )
    {
        int value;

        SpeedDefinition def = (SpeedDefinition) connectionSpeedCbx.getSelectedItem();
        value = def.getSpeedInKbps();
        if ( ServiceManager.sCfg.networkSpeedKbps != value )
        {
            ServiceManager.sCfg.networkSpeedKbps = value;
            OptionsSettingsPane.triggerConfigSave( inputDic );
        }

        if ( totalBandwidthSldr.getValue() == UNLIMITED_VALUE )
        {
            value = Cfg.UNLIMITED_BANDWIDTH;
        }
        else
        {
            value = (int)Math.round( currentTotalBandwidth );
        }
        if ( ServiceManager.sCfg.maxTotalBandwidth != value )
        {
            ServiceManager.sCfg.maxTotalBandwidth = value;
            OptionsSettingsPane.triggerConfigSave( inputDic );
        }

        if ( netBandwidthSldr.getValue() == UNLIMITED_VALUE )
        {
            value = Cfg.UNLIMITED_BANDWIDTH;
        }
        else
        {
            value = (int)Math.round( currentNetBandwidth );
        }
        if ( ServiceManager.sCfg.mNetMaxRate != value )
        {
            ServiceManager.sCfg.mNetMaxRate = value;
            OptionsSettingsPane.triggerConfigSave( inputDic );
        }

        if ( downloadBandwidthSldr.getValue() == UNLIMITED_VALUE )
        {
            value = Cfg.UNLIMITED_BANDWIDTH;
        }
        else
        {
            value = (int)Math.round( currentDownloadBandwidth );
        }
        if ( ServiceManager.sCfg.mDownloadMaxBandwidth != value )
        {
            ServiceManager.sCfg.mDownloadMaxBandwidth = value;
            OptionsSettingsPane.triggerConfigSave( inputDic );
        }

        if ( uploadBandwidthSldr.getValue() == UNLIMITED_VALUE )
        {
            value = Cfg.UNLIMITED_BANDWIDTH;
        }
        else
        {
            value = (int)Math.round( currentUploadBandwidth );
        }
        if ( ServiceManager.sCfg.mUploadMaxBandwidth != value )
        {
            ServiceManager.sCfg.mUploadMaxBandwidth = value;
            OptionsSettingsPane.triggerConfigSave( inputDic );
        }
    }

    private void initConfigValues()
    {
        int netSpeed = ServiceManager.sCfg.networkSpeedKbps;
        SpeedDefinition currentDef;
        int speedDiff;
        for ( int i = 0; i < speedDefinitions.length; i++ )
        {
            currentDef = speedDefinitions[ i ];
            speedDiff = currentDef.getSpeedInKbps() - netSpeed;
            if ( speedDiff >= 0 )
            {
                connectionSpeedCbx.setSelectedIndex( i );
                if ( i == 0 )
                {
                    // if the index stays 0 then the selection didn't change
                    // we need to update manually
                    updateMaxConnectionBandwidth( currentDef );
                }
                break;
            }
        }

        double bandwidth = (double)ServiceManager.sCfg.maxTotalBandwidth;
        int perc = (int)Math.round( bandwidth / maxConnectionBandwidth * 100.0 );
        totalBandwidthSldr.setValue( perc );

        bandwidth = (double)ServiceManager.sCfg.mNetMaxRate;
        perc = (int)Math.round( bandwidth / currentTotalBandwidth * 100.0 );
        netBandwidthSldr.setValue( perc );


        bandwidth = (double)ServiceManager.sCfg.mDownloadMaxBandwidth;
        perc = (int)Math.round( bandwidth / currentTotalBandwidth * 100.0 );
        downloadBandwidthSldr.setValue( perc );

        bandwidth = (double)ServiceManager.sCfg.mUploadMaxBandwidth;
        perc = (int)Math.round( bandwidth / currentTotalBandwidth * 100.0 );
        uploadBandwidthSldr.setValue( perc );
    }

    private void calculateBandwidth()
    {
        String labelText;
        double perc;

        int value = totalBandwidthSldr.getValue();
        if ( value == UNLIMITED_VALUE )
        {
            labelText = Localizer.getString( "Unlimited" );
            currentTotalBandwidth = maxConnectionBandwidth;
        }
        else
        {
            perc = (double)value / 100;
            currentTotalBandwidth = maxConnectionBandwidth * perc;
            labelText = StrUtil.formatSizeBytes( currentTotalBandwidth ) +
                Localizer.getString( "PerSec" );
        }
        totalBandwidthLabel.setText( labelText );

        value = netBandwidthSldr.getValue();
        if ( value == UNLIMITED_VALUE )
        {
            labelText = Localizer.getString( "Unlimited" );
            currentNetBandwidth = maxConnectionBandwidth;
        }
        else
        {
            perc = (double)value / 100;
            currentNetBandwidth = currentTotalBandwidth * perc;
            labelText = StrUtil.formatSizeBytes( currentNetBandwidth ) +
                Localizer.getString( "PerSec" );
        }
        netBandwidthLabel.setText( labelText );

        value = downloadBandwidthSldr.getValue();
        if ( value == UNLIMITED_VALUE )
        {
            labelText = Localizer.getString( "Unlimited" );
            currentDownloadBandwidth = maxConnectionBandwidth;
        }
        else
        {
            perc = (double)value / 100;
            currentDownloadBandwidth = currentTotalBandwidth * perc;
            labelText = StrUtil.formatSizeBytes( currentDownloadBandwidth ) +
                Localizer.getString( "PerSec" );
        }
        downloadBandwidthLabel.setText( labelText );

        value = uploadBandwidthSldr.getValue();
        if ( value == UNLIMITED_VALUE )
        {
            labelText = Localizer.getString( "Unlimited" );
            currentUploadBandwidth = maxConnectionBandwidth;
        }
        else
        {
            perc = (double)value / 100;
            currentUploadBandwidth = currentTotalBandwidth * perc;
            labelText = StrUtil.formatSizeBytes( currentUploadBandwidth ) +
                Localizer.getString( "PerSec" );
        }
        uploadBandwidthLabel.setText( labelText );
    }

    private void updateMaxConnectionBandwidth( SpeedDefinition speedDefinition )
    {
        maxConnectionBandwidth = speedDefinition.getSpeedInKB() * StrUtil.s1kB;
        calculateBandwidth();
    }

    private class SliderChangeListener implements ChangeListener
    {
        public void stateChanged( ChangeEvent e )
        {
            calculateBandwidth();
        }
    }

    private class SpeedItemListener implements ItemListener
    {
        public void itemStateChanged( ItemEvent e )
        {
            if ( e.getStateChange() == ItemEvent.SELECTED )
            {
                SpeedDefinition speedDefinition = ( SpeedDefinition ) e.getItem();
                updateMaxConnectionBandwidth( speedDefinition );
            }
        }
    }

    private static class SpeedDefinition
    {
        private String representation;

        /**
         * The speed of the connection in kilo bits per second.
         */
        private int speedInKbps;

        /**
         * @param aRepresentation the not localized string representation
         */
        public SpeedDefinition( String aRepresentation, int aSpeedInKbps )
        {
            representation = Localizer.getString( aRepresentation );
            speedInKbps = aSpeedInKbps;
        }

        /**
         * Returns the speed of the connection in kilo bytes per second.
         */
        public double getSpeedInKB()
        {
            return speedInKbps / 8;
        }

        public int getSpeedInKbps()
        {
            return speedInKbps;
        }

        public String toString()
        {
            return representation;
        }
    }
}