/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.gui.dialogs.options;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;
import java.util.*;
import phex.ServiceManager;
import phex.utils.Localizer;

public class OptionsDialog extends JDialog
{
    /**
     * Represents the right hand side pane that shows the options corresponding
     * to the selected entry on the tree.
     */
    private JPanel optionViewPane;

    private JTree optionTree;

    /**
     * This array should contain all setting panes that need to be validated in
     * the order of validation.
     */
    private OptionsSettingsPane[] settingsPanes = null;

    /**
     * The dictionary is filled with values while validating and applying the
     * changes done to the OptionsSettingsPanes.
     */
    private HashMap inputDictionary;

    public OptionsDialog()
    {
        super( ServiceManager.getManager().getMainFrame(),
            Localizer.getString( "PhexOptions" ), false );
        inputDictionary = new HashMap();
        settingsPanes = new OptionsSettingsPane[ 9 ];
        prepareComponent();
    }

    public void setOptionView( OptionsSettingsPane pane )
    {
        pane.prepareForDisplay();
        optionViewPane.removeAll();
        optionViewPane.setLayout( new GridBagLayout() );
            GridBagConstraints constraints = new GridBagConstraints();
            constraints.fill = GridBagConstraints.BOTH;
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.weightx = 1;
            constraints.weighty = 1;
            constraints.anchor = GridBagConstraints.NORTHWEST;
        optionViewPane.add( pane, constraints );
        optionViewPane.revalidate();
        optionViewPane.repaint();


        // sometime the size of a option pane can freak out...
        // here we adjust the size of the dialog to have the pane and tree
        // layed out right...
        Dimension prefSize = getPreferredSize();
        Dimension currSize = getSize();
        if ( prefSize.width  > currSize.width
          || prefSize.height > currSize.height )
        {
            currSize.setSize(
                Math.max( prefSize.width, currSize.width ),
                Math.max( prefSize.height, currSize.height ) );
            setSize( currSize );
            doLayout();
        }
    }

    private void prepareComponent()
    {
        addWindowListener(new WindowAdapter()
            {
                public void windowClosing( WindowEvent evt )
                {
                    closeDialog( );
                }
            }
        );

        Container contentPane = getContentPane();
        contentPane.setLayout( new GridBagLayout() );
        GridBagConstraints constraints;
        SpecialLAFPanel pane = new SpecialLAFPanel();
        pane.setLayout( new GridBagLayout() );
            constraints = new GridBagConstraints();
            constraints.fill = GridBagConstraints.BOTH;
            constraints.weightx = 1;
            constraints.weighty = 1;
            constraints.insets = new Insets( 0, 0, 0, 0 );
        contentPane.add( pane, constraints );

        JPanel treePane = new JPanel();
        treePane.setLayout( new GridBagLayout() );
            constraints = new GridBagConstraints();
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.VERTICAL;
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.insets = new Insets( 2, 2, 5, 0 );
            constraints.weighty = 1;
            constraints.weightx = 0;
        pane.add( treePane, constraints );

        optionTree = new JTree( createOptionTreeModel() );
        optionTree.setCellRenderer( new OptionsTreeCellRenderer() );
        // no root
        optionTree.setRootVisible( false );
        // this is no nice but fast way to expand the tree
        optionTree.expandRow( 2 );
        optionTree.expandRow( 1 );
        optionTree.expandRow( 0 );
        // single selection mode
        optionTree.getSelectionModel().setSelectionMode(
            TreeSelectionModel.SINGLE_TREE_SELECTION );
        // show lines between leafs
        optionTree.putClientProperty("JTree.lineStyle", "Angled");
        // selection listener
        optionTree.addTreeSelectionListener( new OptionSelectionListener() );
            constraints = new GridBagConstraints();
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.fill = GridBagConstraints.BOTH;
            constraints.weightx = 1;
            constraints.weighty = 1;
            constraints.insets = new Insets( 0, 0, 0, 0 );
        JScrollPane scrollPane = new JScrollPane( optionTree );
        treePane.add( optionTree, constraints );

        optionViewPane = new JPanel();
        optionViewPane.setLayout( new GridBagLayout() );
            constraints = new GridBagConstraints();
            constraints.fill = GridBagConstraints.BOTH;
            constraints.gridx = 1;
            constraints.gridy = 0;
            constraints.weightx = 1;
            constraints.weighty = 1;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.insets = new Insets( 0, 5, 3, 2 );
        pane.add( optionViewPane, constraints );

        JPanel btnPane = new JPanel();
        btnPane.setLayout( new GridBagLayout() );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 1;
            constraints.gridwidth = 2;
            constraints.weightx = 1;
            constraints.insets = new Insets( 2, 0, 2, 0 );
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.anchor = GridBagConstraints.SOUTHEAST;
        pane.add( btnPane, constraints );

        JSeparator sep = new JSeparator();
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 1;
            constraints.gridwidth = 3;
            constraints.weighty = 1;
            constraints.insets = new Insets( 6, 0, 6, 0 );
            constraints.anchor = GridBagConstraints.SOUTH;
            constraints.fill = GridBagConstraints.HORIZONTAL;
        btnPane.add( sep, constraints );

        JButton okBtn = new JButton( Localizer.getString( "OK" ) );
        okBtn.addActionListener( new ActionListener()
            {
                public void actionPerformed( ActionEvent e )
                {
                    if ( isAllInputValid() )
                    {
                        saveAndApplyAllChanges();
                        closeDialog();
                    }
                }
            } );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 2;
            constraints.weightx = 1;
            constraints.insets = new Insets( 3, 3, 3, 3 );
            constraints.anchor = GridBagConstraints.NORTHEAST;
        btnPane.add( okBtn, constraints );

        JButton applyBtn = new JButton( Localizer.getString( "Apply" ) );
        applyBtn.addActionListener( new ActionListener()
            {
                public void actionPerformed( ActionEvent e )
                {
                    if ( isAllInputValid() )
                    {
                        saveAndApplyAllChanges();
                    }
                }
            } );
            constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 2;
            constraints.insets = new Insets( 3, 3, 3, 3 );
            constraints.anchor = GridBagConstraints.NORTHEAST;
        btnPane.add( applyBtn, constraints );

        JButton cancelBtn = new JButton( Localizer.getString( "Cancel" ) );
        cancelBtn.addActionListener( new ActionListener()
            {
                public void actionPerformed( ActionEvent e )
                {
                    closeDialog();
                }
            } );
            constraints = new GridBagConstraints();
            constraints.gridx = 2;
            constraints.gridy = 2;
            constraints.insets = new Insets( 3, 3, 3, 3 );
            constraints.anchor = GridBagConstraints.NORTHEAST;
        btnPane.add( cancelBtn, constraints );

        optionTree.setSelectionRow( 0 );
        pack();
        pane.setMinimumSize( pane.getPreferredSize() );
        setDefaultCloseOperation( JDialog.DISPOSE_ON_CLOSE );
        setLocationRelativeTo( ServiceManager.getManager().getMainFrame() );
    }

    private boolean isAllInputValid()
    {
        DefaultMutableTreeNode root = (DefaultMutableTreeNode)optionTree.getModel().getRoot();
        for ( int i = 0; i < settingsPanes.length; i++ )
        {
            // don't verify if not displayed...
            if ( !settingsPanes[i].isSettingsPaneDisplayed() )
            {
                continue;
            }

            settingsPanes[i].checkInput( inputDictionary );
            if ( settingsPanes[i].isInputValid( inputDictionary ) )
            {
                continue;
            }

            // in error case....

            // find tree node with pane...
            // TODO this is not nice but the easiest solution that I found and works
            // enumerate depth first through whole tree till found...
            Enumeration enum = root.depthFirstEnumeration();
            while ( enum.hasMoreElements() )
            {
                DefaultMutableTreeNode node = (DefaultMutableTreeNode) enum.nextElement();
                Object obj = node.getUserObject();
                if ( obj instanceof OptionsSettingsPane )
                {
                    if ( obj == settingsPanes[i] )
                    {
                        optionTree.setSelectionPath( new TreePath( node.getPath() ) );
                        break;
                    }
                }
            }
            setOptionView( settingsPanes[i] );
            settingsPanes[i].displayErrorMessage( inputDictionary );

            return false;
        }
        return true;
    }

    private void saveAndApplyAllChanges()
    {
        DefaultMutableTreeNode root = (DefaultMutableTreeNode)optionTree.getModel().getRoot();
        for ( int i = 0; i < settingsPanes.length; i++ )
        {
            // don't apply if not displayed...
            if ( !settingsPanes[i].isSettingsPaneDisplayed() )
            {
                continue;
            }

            settingsPanes[i].saveAndApplyChanges( inputDictionary );
        }
        if ( OptionsSettingsPane.isSaveConfigTriggered( inputDictionary ) )
        {
            ServiceManager.sCfg.save();
        }
        if ( OptionsSettingsPane.isSharedFilesRescanTriggered( inputDictionary ) )
        {
            ServiceManager.getShareManager().getFileAdministration().rescan();
        }
    }

    private void closeDialog( )
    {
        setVisible(false);
        dispose();
    }

    /**
     * + General Settings
     *   + Network
     *   + Bandwidth
     *   + Filters
     * + Download/Upload Settings
     *   + Download
     *   + Upload
     *   + Directories
     * + User Interface
     */
    private TreeModel createOptionTreeModel()
    {
        int pos = 0;
        DefaultMutableTreeNode root = new DefaultMutableTreeNode();
        DefaultMutableTreeNode parent;

        settingsPanes[ pos ] = new MainTextPane();
        parent = new DefaultMutableTreeNode( settingsPanes[ pos ] );
        root.add( parent );
        settingsPanes[ ++pos ] = new NetworkPane();
        parent.add( new DefaultMutableTreeNode( settingsPanes[ pos ] ) );
        settingsPanes[ ++pos ] = new BandwidthPane();
        parent.add( new DefaultMutableTreeNode( settingsPanes[ pos ] ) );
        /*settingsPanes[ ++pos ] = new FiltersPane();
        parent.add( new DefaultMutableTreeNode( settingsPanes[ pos ] ) );*/

        settingsPanes[ ++pos ] = new GeneralTextPane(
            "DownloadSharingSettings", "DownloadSharingSettingsText",
            "DownloadSharingSettings"  );
        parent = new DefaultMutableTreeNode( settingsPanes[ pos ] );
        root.add(parent);
        settingsPanes[ ++pos ] = new DownloadPane();
        parent.add( new DefaultMutableTreeNode( settingsPanes[ pos ] ) );
        settingsPanes[ ++pos ] = new SharingPane();
        parent.add( new DefaultMutableTreeNode( settingsPanes[ pos ] ) );
        settingsPanes[ ++pos ] = new DirectoriesPane();
        parent.add( new DefaultMutableTreeNode( settingsPanes[ pos ] ) );

        settingsPanes[ ++pos ] = new GeneralTextPane(
            "UserInterface", "UserInterfaceText",
            "UserInterface"  );
        parent = new DefaultMutableTreeNode( settingsPanes[ pos ] );
        root.add(parent);
        settingsPanes[ ++pos ] = new GeneralUIPane();
        parent.add(new DefaultMutableTreeNode( settingsPanes[ pos ] ));
        return new DefaultTreeModel(root);
    }

    class OptionSelectionListener implements TreeSelectionListener
    {
        public void valueChanged( TreeSelectionEvent e )
        {
            TreePath path = e.getPath();
            DefaultMutableTreeNode selectionNode = (DefaultMutableTreeNode)path.getLastPathComponent();
            Object obj = selectionNode.getUserObject();
            if ( obj instanceof OptionsSettingsPane )
            {
                setOptionView( (OptionsSettingsPane) obj );
            }
        }
    }

    /**
     * This class is used to have a more powerfull updateUI method to update
     * none displayed component like the settings panes.
     */
    class SpecialLAFPanel extends JPanel
    {
        private boolean inited = true;

        public SpecialLAFPanel()
        {
            super();
            inited = true;
        }

        public void updateUI()
        {
            if ( inited && settingsPanes != null )
            {
                // go through settings panes and update ui
                for ( int i = 0; i < settingsPanes.length; i++ )
                {
                    if ( settingsPanes[i] != null )
                    {
                        // don't do anything if not displayed...
                        if ( !settingsPanes[i].isSettingsPaneDisplayed() )
                        {
                            continue;
                        }
                        SwingUtilities.updateComponentTreeUI( settingsPanes[i] );
                    }
                }
            }
            super.updateUI();
        }
    }
}