/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.gui.models;

import java.util.*;
import javax.swing.table.*;
import javax.swing.event.*;
import phex.ServiceManager;
import phex.event.DownloadFilesChangeListener;
import phex.common.TransferDataProvider;
import phex.download.DownloadManager;
import phex.download.DownloadFile;
import phex.download.RemoteFile;
import phex.gui.common.LazyEventQueue;
import phex.utils.StrUtil;
import phex.gui.renderer.*;

/**
 * Describes the download table.
 */
public class DownloadTableModel extends AbstractTableModel
{
    private DownloadManager downloadMgr;
 
    private final static int colNumSharingHost = 0;
    private final static int colNumClient = 1;
    private final static int colNumFile = 2;
    private final static int colNumRetries = 3;
    private final static int colNumPercent = 4;
    private final static int colNumSize = 5;
    private final static int colNumRate = 6;
    private final static int colNumStatus = 7;
    
    private final static String[] tableColumns =
    {
        "Sharing Host",
        "Server",
        "File",
        "Retries",
        "%",
        "Size",
        "Rate",
        "Status",
    };

    public DownloadTableModel()
    {
        downloadMgr = ServiceManager.getDownloadManager();
        downloadMgr.addDownloadFilesChangeListener( new DownloadFilesListener() );
    }

    public String getColumnName(int col)
    {
        return tableColumns[ col ];
    }

    public int getColumnCount()
    {
        return tableColumns.length;
    }

    public int getRowCount()
    {
        return downloadMgr.getDownloadCount();
    }

    public Object getValueAt(int row, int col)
    {
        DownloadFile download = downloadMgr.getDownloadFileAt( row );
        if ( download == null )
        {
            fireTableRowsDeleted( row, row );
            return "";
        }
        RemoteFile rfile = download.getCurrentRemoteFile();

        switch (col)
        {
            case colNumSharingHost:
                    if ( rfile == null )
                    {
                        return "unknown";
                    }
                    else
                    {
                        return rfile.getRemoteHost();
                    }
                    
            case colNumClient:
                if (download.getRemoteAppName() == null)
                    return "unknown";
                else
                    return download.getRemoteAppName();
                
            case colNumFile:
                    return download.getShortname();
                    
            case colNumRetries:
                    return download.getRetryCountObject();
                    
            case colNumPercent:
                // handled by ProgressCellRenderer
                return download.getProgress();
                
            case colNumSize:
                // handled by TransferSizeCellRenderer
                return download;
            
            case colNumRate:
                // handled by TransferRateCellRenderer
                return download;
            
            case colNumStatus:
                return download.getStatusName();
        }
        return "";
    }


    public void setValueAt(Object value, int row, int col)
    {
        // Do nothing
    }

    public boolean isCellEditable(int row, int col)
    {
        return false;
    }

    public Class getColumnClass(int col)
    {
        switch ( col )
        {
            case colNumRetries:
                return Integer.class;
                
            case colNumPercent:
                return ProgressCellRenderer.class;
                
            case colNumSize:
                return TransferSizeCellRenderer.class;
                
            case colNumRate:
                return TransferRateCellRenderer.class;
        }
        
        return String.class;
    }

    private class DownloadFilesListener
        implements DownloadFilesChangeListener
    {
        private LazyEventQueue lazyEventQueue;

        public DownloadFilesListener()
        {
            lazyEventQueue = ServiceManager.getLazyEventQueue();
        }

        public void downloadFileChanged( final int position )
        {
            lazyEventQueue.addTableModelEvent(
                new TableModelEvent( DownloadTableModel.this, position, position,
                    TableModelEvent.ALL_COLUMNS, TableModelEvent.UPDATE ) );
        }

        public void downloadFileAdded( final int position )
        {
            fireTableChanged( new TableModelEvent(DownloadTableModel.this,
                position, position, TableModelEvent.ALL_COLUMNS,
                TableModelEvent.INSERT ) );
        }

        public void downloadFileRemoved( final int position )
        {
            fireTableChanged( new TableModelEvent(DownloadTableModel.this,
                position, position, TableModelEvent.ALL_COLUMNS,
                TableModelEvent.DELETE ) );
        }
    }
}