/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.gui.models;

import java.util.*;
import javax.swing.*;
import javax.swing.table.*;

import phex.ServiceManager;
import phex.host.Host;
import phex.host.HostManager;
import phex.event.NetworkHostsChangeListener;
import phex.utils.StrUtil;

public class NetworkTableModel extends AbstractTableModel
{
    private HostManager hostMgr;
    
    private static final int numColRemoteHost = 0;
    private static final int numColType = 1;
    private static final int numColReceivedDropped = 2;
    private static final int numColSentQueued = 3;
    private static final int numColPingLatency = 4;
    private static final int numColShared = 5;
    private static final int numColUptime = 6;
    private static final int numColStatus = 7;
    
    private static String[] tableColumns =
    {
        "Remote Host",
        "Type",
        "Received (Dropped)",
        "Sent (Queued)",
        "Ping Latency (ms)",
        "Shared",
        "Uptime",
        "Status"
    };
    
    public NetworkTableModel()
    {
        hostMgr = ServiceManager.getHostManager();
        hostMgr.addNetworkHostsChangeListener( new NetworkHostsListener() );
    }
    
    public String getColumnName(int col)
    {
        return tableColumns[ col ];
    }
    
    public int getColumnCount()
    {
        return tableColumns.length;
    }
    
    public int getRowCount()
    {
        return hostMgr.getNetworkHostCount();
    }
    
    public Object getValueAt(int row, int col)
    {
        Host host = hostMgr.getNetworkHostAt( row );
        if ( host == null )
        {
            fireTableRowsDeleted( row, row );
            return "";
        }
        
        switch (col)
        {
            case numColRemoteHost:
                return host.getHostAddress().getHostName() + ":" +
                host.getHostAddress().getPort();
                
            case numColType:
                return host.getTypeName();
                
            case numColReceivedDropped:
                return String.valueOf(host.getReceivedCount() + " (" + String.valueOf(host.getDropCount()) + ")");
                
            case numColSentQueued:
                return String.valueOf(host.getSentCount()) + " (" + String.valueOf(host.getSendQueueLength()) + ")";
                
            case numColPingLatency:
                if (host.getLatency() == 99999)
                    return "Timeout";
                else
                    return String.valueOf(host.getLatency());
                
            case numColShared:
                if (host.getFileCount() == 0)
                {
                    return "";
                }
                else
                {
                    return host.getFileCount() + "/" + StrUtil.formatSizeBytes( ((long)host.getTotalSize()) * 1024L );
                }
                
            case numColUptime:
                long upSeconds = host.getConnectionUpTime( System.currentTimeMillis() ) / 1000;
                return StrUtil.formatSignificantElapsedTime( upSeconds );
                
            case numColStatus:
                return host.getStatusName();
        }
        return "";
    }
    
    
    public void setValueAt(Object value, int row, int col)
    {
        // Do nothing
    }
    
    
    public boolean isCellEditable(int row, int col)
    {
        return false;
    }
    
    // TODO improve it according to the real type.
    public Class getColumnClass(int col)
    {
        return String.class;
    }
    
    private class NetworkHostsListener
    implements NetworkHostsChangeListener
    {
        public void networkHostChanged( final int position )
        {
            fireTableRowsUpdated( position, position );
        }
        
        public void networkHostAdded( final int position )
        {
            fireTableRowsInserted( position, position );
        }
        
        public void networkHostRemoved( final int position )
        {
            fireTableRowsDeleted( position, position );
        }
    }
}