/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.gui.models;

import javax.swing.event.*;
import javax.swing.table.AbstractTableModel;
import phex.ServiceManager;
import phex.event.SharedFilesChangeListener;
import phex.share.ShareFile;
import phex.share.FileAdministration;
import phex.gui.common.LazyEventQueue;
import phex.utils.Localizer;

public class SharedFilesTableModel extends AbstractTableModel
{
    private static final int colNumFile = 0;
    private static final int colNumDirectory = 1;
    private static final int colNumSize = 2;
    private static final int colNumSearchCount = 3;
    private static final int colNumUploadCount = 4;
    
    private static String[] tableColumns =
    {
        Localizer.getString( "File" ),
        Localizer.getString( "Directory" ),
        Localizer.getString( "Size" ),
        Localizer.getString( "SearchCount" ),
        Localizer.getString( "UploadCount" )
    };

    private FileAdministration sharedFileAdmin;

    public SharedFilesTableModel()
    {
        sharedFileAdmin = ServiceManager.getShareManager().getFileAdministration();
        sharedFileAdmin.addSharedFilesChangeListener( new SharedFilesListener() );
    }

    public String getColumnName(int col)
    {
        return tableColumns[ col ];
    }

    public int getColumnCount()
    {
        return tableColumns.length;
    }

    public int getRowCount()
    {
        return sharedFileAdmin.getFileCount();
    }

    public Object getValueAt(int row, int col)
    {
        ShareFile shareFile = sharedFileAdmin.getFileAt( row );
        if ( shareFile == null )
        {
            fireTableRowsDeleted( row, row );
            return "";
        }
        switch ( col )
        {
            case colNumFile:
                return shareFile.getEffectiveName();
                
            case colNumDirectory:
                return shareFile.getFile().getParent();
                
            case colNumSize:
                return shareFile.getFileSizeObject();
                
            case colNumSearchCount:
                return new Integer( shareFile.getSearchCount() );
                
            case colNumUploadCount:
                return new Integer( shareFile.getUploadCount() );
        }

        return "";
    }

    public Class getColumnClass(int col)
    {
        switch (col)
        {
            case colNumSize:
                return Long.class;
                
            case colNumSearchCount:
                return Integer.class;
                
            case colNumUploadCount:
                return Integer.class;
        }
        return String.class;
    }

    private class SharedFilesListener implements SharedFilesChangeListener
    {
        private LazyEventQueue lazyEventQueue;

        public SharedFilesListener()
        {
            lazyEventQueue = ServiceManager.getLazyEventQueue();
        }

        /**
         * Called if a shared file changed.
         */
        public void sharedFileChanged( int position )
        {
            fireTableCellUpdated( position, position );
        }

        /**
         * Called if a shared file was added.
         */
        public void sharedFileAdded( int position )
        {
            lazyEventQueue.addTableModelEvent(
                new TableModelEvent( SharedFilesTableModel.this,
                    position, position, TableModelEvent.ALL_COLUMNS,
                    TableModelEvent.INSERT ) );
        }

        /**
         * Called if all shared files changed.
         */
        public void allSharedFilesChanged( )
        {
            fireTableDataChanged();
        }
    }
}