/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Konrad Haenel ( alterego1@gmx.at )
 *                     Gregor Koukkoullis ( phex@kouk.de )
 *  Copyright (C) 2000 William W. Wong ( williamw@jps.net )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.gui.tabs;

import java.awt.*;
import java.awt.event.*;
import java.awt.datatransfer.StringSelection;
import java.util.Vector;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;

import phex.ServiceManager;
import phex.MainFrame;
import phex.query.QueryHistoryMonitor;
import phex.msg.MsgManager;
import phex.interfaces.IMonitorSearch;
import phex.download.RemoteFile;
import phex.download.DownloadManager;
import phex.gui.common.GUIUtils;
import phex.gui.common.IntegerTextField;
import phex.gui.models.QueryHistoryMonitorTableModel;
import phex.event.QueryHistoryChangeListener;
import phex.utils.TableSorter;
import phex.utils.URLUtil;
import phex.utils.IPUtils;
import phex.gui.common.TextFieldFactory;

public class MonitorTab extends JPanel implements IMonitorSearch
{
    private MainFrame mainFrame;
    private MsgManager mMsgManager;
    private DownloadManager mDownloadMgr;
    private QueryHistoryMonitor queryHistory;
    private JTable mMonitorTable;
    private QueryHistoryMonitorTableModel queryHistoryModel;
    private JTable mPassiveSearchTable;
    private AbstractTableModel mPassiveSearchModel;
    private TableSorter mPassiveSearchSorter;
    private JTextField mPassiveSearch;
    private JButton mPassiveButton;
    private JLabel mPassiveStatus;
    private JPopupMenu mSearchResultPopupMenu;

    private JCheckBox enableMonitorCheckbox;
    private JTextField numberOfMonitorRows;

    public MonitorTab( MainFrame frame )
    {
        mainFrame = frame;
        mMsgManager = ServiceManager.getMsgManager();
        mDownloadMgr = ServiceManager.getDownloadManager();
        queryHistory = ServiceManager.getQueryManager().getQueryHistoryMonitor();

        mMsgManager.setMonitorSearchListener(this);

        mSearchResultPopupMenu = new JPopupMenu();
        mainFrame.populatePopupMenu(mSearchResultPopupMenu, "SearchResultTable.PopupMenu");
    }

    public void initComponent()
    {
        setLayout(new BorderLayout());
        GridBagConstraints constraints;
        JLabel label;

        JPanel historyHeader = new JPanel( new GridBagLayout() );

        label = new JLabel( "Search Monitor" );
        constraints = new GridBagConstraints();
        constraints.anchor = GridBagConstraints.NORTHWEST;
        historyHeader.add( label, constraints );

        enableMonitorCheckbox = new JCheckBox( "enable" );
        enableMonitorCheckbox.setSelected( queryHistory.isHistoryMonitored() );
        enableMonitorCheckbox.addChangeListener( new EnableMonitorChangeListener() );
        constraints = new GridBagConstraints();
        constraints.weightx = 1;
        constraints.anchor = GridBagConstraints.NORTHEAST;
        historyHeader.add( enableMonitorCheckbox, constraints );

        queryHistoryModel = new QueryHistoryMonitorTableModel();
        mMonitorTable = new JTable( queryHistoryModel );

        ServiceManager.getQueryManager().
            getQueryHistoryMonitor().setQueryHistoryChangeListener(
            new QueryHistoryChangeListener()
            {
                public void queryHistoryChanged( )
                {
                    GUIUtils.fireTableChanged( mMonitorTable, queryHistoryModel );
                }
            });
        mMonitorTable.getSelectionModel().addListSelectionListener(new SelectionHandler());


        JPanel historyFooter = new JPanel( new GridBagLayout() );
        label = new JLabel( "Show " );
        constraints = new GridBagConstraints();
        constraints.weightx = 1;
        constraints.anchor = GridBagConstraints.SOUTHEAST;
        historyFooter.add( label, constraints );

        numberOfMonitorRows = new IntegerTextField(
            String.valueOf( queryHistory.getMaxHistorySize() ), 3, 3 );
        numberOfMonitorRows.getDocument().addDocumentListener(
            new MonitorRowsDocumentListener() );
        constraints = new GridBagConstraints();
        constraints.weighty = 1;
        constraints.anchor = GridBagConstraints.SOUTHEAST;
        historyFooter.add( numberOfMonitorRows, constraints );

        label = new JLabel( " Rows" );
        constraints = new GridBagConstraints();
        constraints.anchor = GridBagConstraints.SOUTHEAST;
        historyFooter.add( label, constraints );


        JPanel monitorTablePanel = new JPanel(new BorderLayout());
        monitorTablePanel.add(BorderLayout.NORTH, historyHeader );
        monitorTablePanel.add(BorderLayout.CENTER, new JScrollPane(mMonitorTable));
        monitorTablePanel.add(BorderLayout.SOUTH, historyFooter );
        monitorTablePanel.setBorder(
            BorderFactory.createCompoundBorder(
                BorderFactory.createCompoundBorder(
                    BorderFactory.createEmptyBorder(8, 8, 8, 8),
                    BorderFactory.createEtchedBorder()),
                BorderFactory.createEmptyBorder(8, 8, 8, 8)));

        JPanel	passiveSearchPanel = new JPanel(new BorderLayout());
        mPassiveSearchModel = new PassiveSearchTableModel();
        mPassiveSearchSorter = new TableSorter(mPassiveSearchModel);
        mPassiveSearchTable = new JTable(mPassiveSearchSorter);
        mPassiveSearchTable.getSelectionModel().addListSelectionListener(new SelectionHandler());
        mPassiveSearchTable.addMouseListener(new MouseHandler());
        mPassiveSearchSorter.addMouseListenerToHeaderInTable(mPassiveSearchTable);

        JPanel	passiveButtonPanel = new JPanel(new BorderLayout());
        passiveButtonPanel.setBorder(BorderFactory.createEmptyBorder(8, 0, 0, 0));
        mPassiveSearch = TextFieldFactory.newTextField("", 20);
        GUIUtils.setToolTipText(mPassiveSearch, "Specify keywords to match, separated by space.");
        mPassiveButton = new JButton("Start Passive Search");
        GUIUtils.setToolTipText(mPassiveButton, "Passive search monitors the search results routed through here.  This might take a while.");
        mPassiveStatus = new JLabel("Passive Search Result");

        mPassiveButton.addActionListener(new PassiveSearchHandler());
        passiveButtonPanel.add(BorderLayout.WEST, new JLabel("Monitor Filter:"));
        passiveButtonPanel.add(BorderLayout.CENTER, mPassiveSearch);
        passiveButtonPanel.add(BorderLayout.EAST, mPassiveButton);
        passiveButtonPanel.add(BorderLayout.SOUTH, mPassiveStatus);

        JPanel passiveDownloadButtonsPanel = new JPanel(new FlowLayout());
        JButton passiveDownloadButton = new JButton("Download");
        JButton passiveClearButton = new JButton("Clear");
        passiveDownloadButton.addActionListener(new PassiveDownloadHandler());
        mainFrame.addRefreshComponent("ActionTransferDownload", passiveDownloadButton);
        passiveClearButton.addActionListener(new PassiveClearHandler());
        passiveDownloadButtonsPanel.add(passiveDownloadButton);
        passiveDownloadButtonsPanel.add(passiveClearButton);

        passiveSearchPanel.add(BorderLayout.NORTH, passiveButtonPanel);
        passiveSearchPanel.add(BorderLayout.CENTER, new JScrollPane(mPassiveSearchTable));
        passiveSearchPanel.add(BorderLayout.SOUTH, passiveDownloadButtonsPanel);
        passiveSearchPanel.setBorder(
            BorderFactory.createCompoundBorder(
                BorderFactory.createCompoundBorder(
                    BorderFactory.createEmptyBorder(8, 8, 8, 8),
                    BorderFactory.createEtchedBorder()),
                BorderFactory.createEmptyBorder(8, 8, 8, 8)));

        monitorTablePanel.setPreferredSize(new Dimension(780, 200));
        passiveSearchPanel.setPreferredSize(new Dimension(780, 230));
        JSplitPane splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
        splitPane.setDividerSize(6);
        splitPane.setOneTouchExpandable(true);
        splitPane.setTopComponent(monitorTablePanel);
        splitPane.setBottomComponent(passiveSearchPanel);

        add(BorderLayout.CENTER, splitPane );
    }

    public RemoteFile getSelectedRemoteFile()
    {
        if (mPassiveSearchTable.getSelectedRowCount() < 1)
        {
            return null;
        }

        int[] rows = mPassiveSearchTable.getSelectedRows();
        int row = mPassiveSearchSorter.indexes[rows[0]];
        Vector results = mMsgManager.getPassiveResults();
        RemoteFile rfile = (RemoteFile)results.elementAt(row);
        return rfile;
    }

    public void passiveSearch()
    {
        if (mMsgManager.getPassiveSearching() == null)
        {
            String	searchStr = mPassiveSearch.getText().trim();

            mMsgManager.setPassiveSearching(searchStr);

            mPassiveButton.setText("Stop Passive Search");
            mPassiveStatus.setText("Passive Search Result (Monitoring...)");
        }
        else
        {
            mMsgManager.setPassiveSearching(null);
            mPassiveButton.setText("Start Passive Search");
            mPassiveStatus.setText("Passive Search Result");
        }
        mainFrame.refreshAllActions();
    }

    public void passiveDownload()
    {
        int rowCount = mPassiveSearchTable.getSelectedRowCount();

        if (rowCount <= 0)
        {
            return;
        }

        int[] rows = mPassiveSearchTable.getSelectedRows();
        Vector results = mMsgManager.getPassiveResults();

        for (int i = 0; i < rowCount; i++)
        {
            int			row = rows[i];
            row = mPassiveSearchSorter.indexes[row];
            RemoteFile	rfile = (RemoteFile)results.elementAt(row);
            RemoteFile	dfile = new RemoteFile(rfile);
            mDownloadMgr.addFileToDownload(dfile);
        }

        mainFrame.selectTab( 3 );
        mainFrame.refreshAllActions();
    }

    public void passiveCopyURL()
    {
        int				rowCount = mPassiveSearchTable.getSelectedRowCount();

        if (rowCount != 1)
        {
            // Cannot copy multiple URLs to the clipboard
            return;
        }

        int[]			rows = mPassiveSearchTable.getSelectedRows();
        Vector			results = mMsgManager.getPassiveResults();
        int				row = mPassiveSearchSorter.indexes[rows[0]];
        RemoteFile		rfile = (RemoteFile)results.elementAt(row);

        String url = URLUtil.encodeURL (rfile.getURL());
        Toolkit.getDefaultToolkit().getSystemClipboard().setContents
                (new StringSelection(url), null);
    }

    public void filterPassiveDownloadHost()
    {
        int				rowCount = mPassiveSearchTable.getSelectedRowCount();

        if (rowCount <= 0)
            return;

        int[]			rows = mPassiveSearchTable.getSelectedRows();
        Vector			results = mMsgManager.getPassiveResults();

        for (int i = 0; i < rowCount; i++)
        {
            int			row = rows[i];
            row = mPassiveSearchSorter.indexes[row];
            RemoteFile	rfile = (RemoteFile)results.elementAt(row);
            ServiceManager.sCfg.mFilteredSearchHosts.addElement(IPUtils.splitIP2Parts(rfile.getRemoteHost()));
        }
        ServiceManager.sCfg.save();
    }

    public boolean isPassiveResultSelected()
    {
        return (mPassiveSearchTable.getSelectedRowCount() > 0);
    }

    private void passiveClearResult()
    {
        mMsgManager.resetPassiveResults();
        mainFrame.refreshAllActions();
    }

    private class PassiveClearHandler implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            passiveClearResult();
        }
    }


    private class PassiveDownloadHandler implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            passiveDownload();
        }
    }

    // IMonitorSearch
    public void passiveResultArrived()
    {
        GUIUtils.fireTableChanged(mPassiveSearchTable, mPassiveSearchModel);
    }

    private class MouseHandler extends MouseAdapter implements MouseListener
    {
        public void mouseClicked(MouseEvent e)
        {
            if (e.getClickCount() == 2)
            {
                if (e.getSource() == mPassiveSearchTable)
                {
                    passiveDownload();
                }
            }
        }


        public void mouseReleased(MouseEvent e)
        {
            if (e.isPopupTrigger())
            {
                popupMenu((Component)e.getSource(), e.getX(), e.getY());
            }
        }


        public void mousePressed(MouseEvent e)
        {
            if (e.isPopupTrigger())
            {
                popupMenu((Component)e.getSource(), e.getX(), e.getY());
            }
        }


        private void popupMenu(Component source, int x, int y)
        {
            if ( source == mPassiveSearchTable)
            {
                mSearchResultPopupMenu.show(source, x, y);
            }
        }
    }

    private class PassiveSearchHandler implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            passiveSearch();
        }
    }

    private class PassiveSearchTableModel extends AbstractTableModel
    {
        public int getColumnCount()
        {
            return 5;
        }


        public int getRowCount()
        {
            Vector			results = mMsgManager.getPassiveResults();
            return results.size();
        }


        public String getColumnName(int col)
        {
            switch (col)
            {
                case 0:
                    return "Sharing Host";
                case 1:
                    return "File";
                case 2:
                    return "Extension";
                case 3:
                    return "Size";
                case 4:
                    return "Host Speed";
            }
            return "";
        }


        public Object getValueAt(int row, int col)
        {
            Vector			results = mMsgManager.getPassiveResults();

            if (row >= results.size())
                return "";

            RemoteFile		rfile = (RemoteFile)results.elementAt(row);

            switch (col)
            {
                case 0:
                    return rfile.getRemoteHost();
                case 1:
                    return rfile.getFilename();
                case 2:
                    return rfile.getFileExt();
                case 3:
                    return rfile.getFileSizeLong();
                case 4:
                    return rfile.getSpeedInt();
            }
            return "";
        }


        public void setValueAt(Object value, int row, int col)
        {
            // Do nothing
        }


        public boolean isCellEditable(int row, int col)
        {
            return false;
        }


        public Class getColumnClass(int col)
        {
            switch (col)
            {
                case 0:
                    return String.class;
                case 1:
                    return String.class;
                case 2:
                    return String.class;
                case 3:
                    return Long.class;
                case 4:
                    return Integer.class;
            }
            return String.class;
        }

    }

    private class SelectionHandler implements ListSelectionListener
    {
        public void valueChanged(ListSelectionEvent e)
        {
            mainFrame.refreshAllActions();
        }
    }

    private class EnableMonitorChangeListener implements ChangeListener
    {
        public void stateChanged( ChangeEvent e )
        {
            queryHistory.setHistoryMonitored(
                enableMonitorCheckbox.isSelected() );
        }
    }

    private class MonitorRowsDocumentListener implements DocumentListener
    {
        public void insertUpdate(DocumentEvent documentevent)
        {
            updateMonitorRows();
        }

        public void removeUpdate(DocumentEvent documentevent)
        {
            updateMonitorRows();
        }

        public void changedUpdate(DocumentEvent documentevent)
        {
        }

        private void updateMonitorRows()
        {
            String rowsStr = numberOfMonitorRows.getText();
            try
            {
                int rows = Integer.parseInt( rowsStr );
                queryHistory.setMaxHistroySize( rows );
            }
            catch ( NumberFormatException sandra )
            {// ignore
            }
        }
    }
}