/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.gui.tabs;

import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import javax.swing.table.*;
import javax.swing.event.*;

import phex.MainFrame;
import phex.ServiceManager;
import phex.msg.MsgManager;
import phex.host.HostManager;
import phex.gui.common.TextFieldFactory;
import phex.interfaces.IHostChanged;
import phex.host.Host;
import phex.host.HostAddress;
import phex.host.CaughtHostsContainer;
import phex.gui.common.GUIUtils;
import phex.gui.models.CaughtHostsListModel;
import phex.gui.models.AutoConnectHostsListModel;
import phex.gui.models.NetworkTableModel;
import phex.utils.*;
import phex.gui.renderer.*;

/**
 * The NetworkTab Panel.
 */
public class NetworkTab extends JPanel implements IHostChanged
{
    private MainFrame mainFrame;
    private HostManager mHostMgr;
    private StatisticTracker statTracker;
    private MsgManager mMsgManager;
    private JTable mNetTable;
    private AbstractTableModel mNetModel;
    private TableSorter mNetSorter;
    private JLabel mConnectionLabel;
    private JTextField mConnectHostText;
    private JLabel mMyIP;
    private JCheckBox mAutoCleanupCheck;
    private JCheckBox mAutoConnectCheck;
    private JList mCatcherList;
    private JTextArea mStats;
    private JTextField newAutoConnectHostTF;
    private JButton addAutoConnectHostBtn;
    private JList autoConnectList;
    private JButton removeAutoConnectHostBtn;
    private JPopupMenu mHostPopupMenu;
    private JPopupMenu mCatcherPopupMenu;

    public NetworkTab( MainFrame frame )
    {
        mainFrame = frame;
        mHostMgr = ServiceManager.getHostManager();
        mMsgManager = ServiceManager.getMsgManager();
        statTracker = ServiceManager.getStatisticTracker();

        mHostPopupMenu = new JPopupMenu();
        mainFrame.populatePopupMenu(mHostPopupMenu, "HostTable.PopupMenu");

        mCatcherPopupMenu = new JPopupMenu();
        mainFrame.populatePopupMenu(mCatcherPopupMenu, "CatcherTable.PopupMenu");

    }

    public void initComponent()
    {
        GridBagConstraints constraints;
        setLayout(new BorderLayout());
        JPanel	hostTablePanel = new JPanel(new BorderLayout());
        mNetModel = new NetworkTableModel();
        mNetSorter = new TableSorter(mNetModel);
        mNetTable = new JTable(mNetSorter);
        mNetTable.getSelectionModel().addListSelectionListener(new SelectionHandler());
        mNetTable.addMouseListener(new MouseHandler());
        mNetSorter.addMouseListenerToHeaderInTable(mNetTable);
        mConnectionLabel = new JLabel("Connections:");
        hostTablePanel.add(BorderLayout.NORTH, mConnectionLabel);
        hostTablePanel.add(BorderLayout.CENTER, new JScrollPane(mNetTable));

        JPanel	hostButtonsPanel = new JPanel(new FlowLayout());
        mConnectHostText = TextFieldFactory.newTextField("", 16);
        mConnectHostText.addKeyListener(new ConnectHostKeyHandler());
        mainFrame.addRefreshComponent("ActionNetworkLeave", mConnectHostText);
        JLabel label = new JLabel("My Address:");
        mainFrame.addRefreshComponent("ActionNetworkLeave", label);
        mMyIP = new JLabel(" ");
        mainFrame.addRefreshComponent("ActionNetworkLeave", mMyIP);
        JButton	connectHostButton = new JButton("Connect");
        connectHostButton.setDefaultCapable(false);
        mainFrame.addRefreshComponent("ActionHostConnect", connectHostButton);
        mAutoCleanupCheck = new JCheckBox("Auto Cleanup", ServiceManager.sCfg.mAutoCleanup);
        mainFrame.addRefreshComponent("ActionNetworkLeave", mAutoCleanupCheck);
        hostButtonsPanel.add(label);
        hostButtonsPanel.add(mMyIP);
        hostButtonsPanel.add(new JLabel("        "));
        hostButtonsPanel.add(mConnectHostText);
        hostButtonsPanel.add(connectHostButton);
        hostButtonsPanel.add(mAutoCleanupCheck);
        connectHostButton.addActionListener(new ConnectHostHandler());
        mAutoCleanupCheck.addActionListener(new AutoCleanupHandler());

        JPanel	upperPanel = new JPanel(new BorderLayout());
        upperPanel.add(BorderLayout.CENTER, hostTablePanel);
        upperPanel.add(BorderLayout.SOUTH, hostButtonsPanel);
        upperPanel.setBorder(
            BorderFactory.createCompoundBorder(
                BorderFactory.createCompoundBorder(
                    BorderFactory.createEmptyBorder(8, 4, 4, 4),
                    BorderFactory.createEtchedBorder()),
                BorderFactory.createEmptyBorder(8, 8, 8, 8)));

        JPanel	lowerPanel = new JPanel(new GridLayout(1, 3, 10, 0));

        lowerPanel.setBorder(
            BorderFactory.createCompoundBorder(
                BorderFactory.createCompoundBorder(
                    BorderFactory.createEmptyBorder(8, 4, 4, 4),
                    BorderFactory.createEtchedBorder()),
                BorderFactory.createEmptyBorder(8, 8, 8, 8)));

        JPanel	statPanel = new JPanel(new BorderLayout());
        mStats = new JTextArea();
        mStats.setEditable(false);
        mStats.setBorder(BorderFactory.createEtchedBorder());

        JButton	updateStat = new JButton("Reset");
        updateStat.setDefaultCapable(false);
        updateStat.addActionListener(new UpdateStatHandler());
        mainFrame.addRefreshComponent("ActionNetworkLeave", updateStat);
        JPanel	updateButtonPanel = new JPanel(new FlowLayout());
        updateButtonPanel.add(updateStat);
        statPanel.add(BorderLayout.NORTH, new JLabel("Statistics"));
        statPanel.add(BorderLayout.CENTER, new JScrollPane( mStats ) );
        statPanel.add(BorderLayout.SOUTH, updateButtonPanel);

        JPanel catcherPanel = new JPanel(new BorderLayout());
        mCatcherList = new JList( new CaughtHostsListModel() );
        mCatcherList.setVisibleRowCount( 4 );
        mCatcherList.getSelectionModel().addListSelectionListener(new SelectionHandler());
        mCatcherList.addMouseListener(new MouseHandler());
        
        mAutoConnectCheck = new JCheckBox("Auto Connect", ServiceManager.sCfg.mAutoConnect);
        mAutoConnectCheck.addActionListener(new AutoConnectHandler());
        
        JButton	connectButton = new JButton("Connect");
        connectButton.setDefaultCapable(false);
        
        mainFrame.addRefreshComponent("ActionHostConnectFromCatcher", connectButton);
        
        JButton	resetButton = new JButton("Reset");
        resetButton.setDefaultCapable(false);
        
        JPanel	connectButtonPanel = new JPanel(new FlowLayout());
        connectButtonPanel.add(mAutoConnectCheck);
        connectButtonPanel.add(connectButton);
        connectButtonPanel.add(resetButton);
        
        catcherPanel.add(BorderLayout.NORTH, new JLabel("Host Catcher"));
        catcherPanel.add(BorderLayout.CENTER, new JScrollPane( mCatcherList ));
        catcherPanel.add(BorderLayout.SOUTH, connectButtonPanel);
        
        connectButton.addActionListener(new ConnectCatcherHandler());
        resetButton.addActionListener(new ResetCatcherHandler());

        JPanel autoConnectMainPanel = new JPanel( new BorderLayout() );

        JLabel autoConnectLabel = new JLabel( Localizer.getString( "AutoConnectHosts" ) );
        autoConnectMainPanel.add( autoConnectLabel, BorderLayout.NORTH );

        JPanel autoConnectPanel = new JPanel( new GridBagLayout() );
        newAutoConnectHostTF = new JTextField( 15 );
        newAutoConnectHostTF.setMinimumSize( newAutoConnectHostTF.getPreferredSize()  );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.WEST;
            constraints.insets = new Insets( 0, 0, 0, 5 );
            autoConnectPanel.add( newAutoConnectHostTF, constraints );
            
        addAutoConnectHostBtn = new JButton( Localizer.getString( "Add" ) );
        addAutoConnectHostBtn.addActionListener( new ActionListener()
            {
                public void actionPerformed( ActionEvent e )
                {
                    performAddAutoConnectHostAction();
                }
            });
            constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.WEST;
            autoConnectPanel.add( addAutoConnectHostBtn, constraints );            
        removeAutoConnectHostBtn = new JButton( Localizer.getString( "Remove" ) );
        removeAutoConnectHostBtn.addActionListener( new ActionListener()
            {
                public void actionPerformed( ActionEvent e )
                {
                    performRemoveAutoConnectHostAction();
                }
            });
            constraints = new GridBagConstraints();
            constraints.gridx = 2;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.WEST;
            constraints.insets = new Insets( 0, 5, 0, 0 );
            autoConnectPanel.add( removeAutoConnectHostBtn, constraints );

        autoConnectList = new JList( new AutoConnectHostsListModel() );
        autoConnectList.setVisibleRowCount( 3 );
        autoConnectList.addListSelectionListener( new ListSelectionListener()
            {
                public void valueChanged( ListSelectionEvent e )
                {
                    if ( e.getValueIsAdjusting() )
                    {
                        autoConnectListValueChanged( );
                    }
                }
            } );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 1;
            constraints.gridwidth = 3;
            constraints.insets = new Insets( 5, 0, 0, 0 );
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.anchor = GridBagConstraints.NORTH;
            autoConnectPanel.add( new JScrollPane( autoConnectList ), constraints );

        autoConnectMainPanel.add( autoConnectPanel, BorderLayout.CENTER );

            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 2;
            constraints.gridwidth = 3;
            constraints.weightx = 1;
            constraints.weighty = 1;
            constraints.fill = GridBagConstraints.BOTH;
            constraints.anchor = GridBagConstraints.NORTH;
            autoConnectPanel.add( catcherPanel, constraints );

        lowerPanel.add( statPanel );
        lowerPanel.add( autoConnectMainPanel );

        Dimension nullDim = new Dimension( 0, 0 );
        upperPanel.setMinimumSize( nullDim );
        lowerPanel.setMinimumSize( nullDim );
        upperPanel.setPreferredSize(new Dimension(780, 280));
        lowerPanel.setPreferredSize(new Dimension(780, 150));
        JSplitPane	splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
        splitPane.setOneTouchExpandable(true);
        splitPane.setTopComponent(upperPanel);
        splitPane.setBottomComponent(lowerPanel);

        add(BorderLayout.CENTER, splitPane );
        
        // Set up cell renderer to provide the correct colour based on connection.
        // TODO decouple this cell renderer stuff
        CellRenderer cellRenderer = new CellRenderer();
        mNetTable.getColumn( mNetModel.getColumnName( 0 ) ).setCellRenderer( cellRenderer );
        mNetTable.getColumn( mNetModel.getColumnName( 1 ) ).setCellRenderer( cellRenderer );
        mNetTable.getColumn( mNetModel.getColumnName( 2 ) ).setCellRenderer( cellRenderer );
        mNetTable.getColumn( mNetModel.getColumnName( 3 ) ).setCellRenderer( cellRenderer );
        mNetTable.getColumn( mNetModel.getColumnName( 4 ) ).setCellRenderer( cellRenderer );
        mNetTable.getColumn( mNetModel.getColumnName( 5 ) ).setCellRenderer( cellRenderer );
        mNetTable.getColumn( mNetModel.getColumnName( 6 ) ).setCellRenderer( cellRenderer );

        // Set up cell renderers. Just use our standard default renderer to class relationships.
        DefaultPhexCellRenderers.setDefaultPhexCellRenderers( mNetTable );
        
    }

    //// IHostChanged Interface
    // * TODO find a better solution for the IP update.
    public void updateMyIP()
    {
        HostAddress address = ServiceManager.getListener().getLocalAddress();
        mMyIP.setText( address.getHostName() + ":" + address.getPort() );
    }
    ////

    public void removeHost()
    {
        int				rowCount = mNetTable.getSelectedRowCount();

        if (rowCount <= 0 || rowCount > mHostMgr.getNetworkHostCount() )
            return;

        int[]			rows = mNetTable.getSelectedRows();
        int[]			orderedRows = new int[rows.length];

        for (int i = 0; i < rowCount; i++)
        {
            int			row = rows[i];
            if (row >= mHostMgr.getNetworkHostCount() )
                return;

            orderedRows[i] = mNetSorter.indexes[row];
        }

        // Bubble sort to assure ascending order.
        for (int i = orderedRows.length - 1; i > 0; i--)
        {
            for (int j = 0; j < i; j++)
            {
                if (orderedRows[j] > orderedRows[j+1])
                {
                    int	tmp = orderedRows[j];
                    orderedRows[j] = orderedRows[j+1];
                    orderedRows[j+1] = tmp;
                }
            }
        }

        // Delete from the highest index to the lowest index.
        for (int i = orderedRows.length - 1; i >= 0; i--)
        {
            Host host = mHostMgr.getNetworkHostAt( orderedRows[i] );
            mHostMgr.removeNetworkHost( host );
        }

        GUIUtils.fireTableChanged(mNetTable, mNetModel);

        mainFrame.refreshAllActions();
    }

    public boolean isConnectSelected()
    {
        String	hostAddr = mConnectHostText.getText().trim();

        if (hostAddr.length() > 0)
        {
            return true;
        }
        return isConnectTableSelected();
    }

    public boolean isConnectTableSelected()
    {
        int		rowCount = mNetTable.getSelectedRowCount();

        return (rowCount > 0 && mHostMgr.getNetworkHostCount() > 0);
    }

    public void disconnectHost()
    {
        int[] rows = mNetTable.getSelectedRows();
        int rowCount = rows.length;
        if ( rowCount == 0 )
        {
            return;
        }

        int row;
        Host[] hosts = new Host[ rowCount ];
        for (int i = 0; i < rowCount; i++)
        {
            row = rows[ i ];
            row = mNetSorter.indexes[row];
            hosts[ i ] = mHostMgr.getNetworkHostAt( row );
        }
        mHostMgr.removeNetworkHosts( hosts );
        GUIUtils.fireTableChanged(mNetTable, mNetModel);
        mainFrame.refreshAllActions();
    }

    public boolean isDisconnectSelected()
    {
        int		rowCount = mNetTable.getSelectedRowCount();

        return (rowCount > 0 && mHostMgr.getNetworkHostCount() > 0);
    }

    public void ignoreHost()
    {
        int[] rows = mNetTable.getSelectedRows();
        int	rowCount = rows.length;
        if ( rowCount == 0 )
        {
            return;
        }

        int row;
        Host[] hosts = new Host[ rowCount ];
        for (int i = 0; i < rowCount; i++)
        {
            row = rows[i];
            row = mNetSorter.indexes[row];
            hosts[ i ] = mHostMgr.getNetworkHostAt( row );
            if ( hosts[ i ] != null )
            {
                String ip = hosts[ i ].getHostAddress().getHostName();
                ServiceManager.sCfg.mNetIgnoredHosts.addElement( IPUtils.splitIP2Parts(ip) );
            }
        }
        mHostMgr.removeNetworkHosts( hosts );
        ServiceManager.sCfg.save();
        GUIUtils.fireTableChanged(mNetTable, mNetModel);
        mainFrame.refreshAllActions();
    }

    public void invalidHost()
    {
        int[] rows = mCatcherList.getSelectedIndices();
        if ( rows.length <= 0 )
        {
            return;
        }
        String[] hosts = mHostMgr.getCaughtHostsAt( rows );
        int hostCount = hosts.length;
        for (int i = 0; i < hostCount; i++)
        {
            String hostaddr = hosts[ i ];
            ServiceManager.sCfg.mNetInvalidHosts.addElement(
                IPUtils.splitIP2Parts( hostaddr ) );
        }
        ServiceManager.sCfg.save();
        mainFrame.refreshAllActions();
    }

    public void refresh()
    {
        GUIUtils.fireTableChanged(mNetTable, mNetModel);

        StringBuffer buf = new StringBuffer();
        buf.append( statTracker.getStatHosts()).append(" hosts\n");
        int statFiles = statTracker.getStatFiles();
        if ( statFiles == Integer.MAX_VALUE )
        {
            buf.append( "> " );
        }
        buf.append( statTracker.getStatFiles()).append(" files\n");
        buf.append( statTracker.getStatSizeStr()).append("\n");
        buf.append( statTracker.getStatMsgCount()).append(" messages\n");
        buf.append( statTracker.getRate()).append(" bytes/second\n");
        buf.append( statTracker.getStatDropCount()).append(" dropped messages\n");


        float avrf = ((float)statTracker.getStatTakenCount())/(statTracker.getStatMsgCount()+1);
        // TODO it would be nicer to do this kind of formating with a DecimalFormatter
        // somewhere centralized... I'm seeing many of this kind of unnice formating...
        // but performance must be considered.
        String avrs = String.valueOf(avrf) + "00000";
        buf.append( avrs.substring(0,5) ).append( " average hops\n" );
        
        buf.append( statTracker.getStatDownloadCount() ).append(" downloads\n");
        buf.append( statTracker.getStatUploadCount()  ).append(" uploads\n");
        
        buf.append( StrUtil.formatSignificantElapsedTime( statTracker.getPhexUptimeInSeconds() ) ).append( " Phex uptime\n" );

        mStats.setText(buf.toString());
    }

    public boolean isConnectFromHostCatcherSelected()
    {
        boolean isSelectionEmpty = mCatcherList.isSelectionEmpty();
        return !isSelectionEmpty;
    }

    public void updateStat()
    {
        mMsgManager.resetMyMsgInit();
        mMsgManager.sendMsgToHosts(mMsgManager.getMyMsgInit());
        statTracker.resetStat();
    }

    public void connectHost()
    {
        String hostAddr = mConnectHostText.getText().trim();

        if (hostAddr.length() > 0)
        {
            StringTokenizer	tokens = new StringTokenizer(hostAddr, ";");
            String firstHost = tokens.nextToken();
            boolean moreThanOne = false;

            // Add new host and connect.
            mHostMgr.createOutgoingConnectionToHost( firstHost );
            GUIUtils.fireTableChanged(mNetTable, mNetModel);

            while (tokens.hasMoreTokens())
            {
                String hostString = tokens.nextToken();
                int idx = hostString.indexOf( ':' );
                String hostIP = hostString.substring( 0, idx );
                String portStr = hostString.substring( idx + 1, hostString.length() );
                int port = Integer.parseInt( portStr );
                HostAddress address = new HostAddress( hostIP, port );
                mHostMgr.addCaughtHost( address, CaughtHostsContainer.HIGH_PRIORITY );
                moreThanOne = true;
            }

            if (moreThanOne)
            {
                mConnectHostText.setText("");
            }
        }
        // Since I never was able to find out if this code is usefull I will
        // comment it out so I don't need to rework it to function better - Gregor
        /*else
        {
            // Connect to list of existing hosts.
            int rowCount = mNetTable.getSelectedRowCount();

            if (rowCount <= 0)
                return;

            int[]	rows = mNetTable.getSelectedRows();

            for (int i = 0; i < rowCount; i++)
            {
                int			row = rows[i];

                if (row >= mHostMgr.getNetworkHostCount() )
                {
                    continue;
                }
                row = mNetSorter.indexes[row];
                mHostMgr.connectHost(row);
            }
            GUIUtils.fireTableChanged(mNetTable, mNetModel);
        }*/
    }

    /*public void selectCatcherTable(int index)
    {
        Vector	hosts = mHostMgr.getHostsCaught();

        mCatcherTable.clearSelection();
        if (index < hosts.size())
        {
            mCatcherTable.addRowSelectionInterval(index, index);
            mCatcherTable.scrollRectToVisible(mCatcherTable.getCellRect(index, 0, true));
        }
    }*/

    public void connectFromHostCatcher()
    {
        int[] indices = mCatcherList.getSelectedIndices();
        if ( indices.length == 0 )
        {
            return;
        }
        String[] addressArr = mHostMgr.getCaughtHostsAt( indices );
        int arrLength = addressArr.length;
        for ( int i = 0; i < arrLength; i++ )
        {
            mHostMgr.createOutgoingConnectionToHost( addressArr[i] );
            mHostMgr.removeCaughtHost( addressArr[i] );
        }
        GUIUtils.fireTableChanged(mNetTable, mNetModel);
    }

    private void toggleAutoConnect()
    {
        ServiceManager.sCfg.mAutoConnect = mAutoConnectCheck.isSelected();
        ServiceManager.sCfg.save();
    }

    private void toggleAutoCleanup()
    {
        ServiceManager.sCfg.mAutoCleanup = mAutoCleanupCheck.isSelected();
        ServiceManager.sCfg.save();
    }

    private void performAddAutoConnectHostAction()
    {
        String host = newAutoConnectHostTF.getText();
        host = host.trim();
        if ( host.length() > 0 )
        {
            mHostMgr.getCaughtHostsContainer().addAutoConnectHost( host );
        }
        newAutoConnectHostTF.setText("");
    }

    private void performRemoveAutoConnectHostAction()
    {
        String host = newAutoConnectHostTF.getText();
        host = host.trim();
        if ( host.length() > 0 )
        {
            mHostMgr.getCaughtHostsContainer().removeAutoConnectHost( host );
        }
        newAutoConnectHostTF.setText("");
    }

    private void autoConnectListValueChanged( )
    {
        String val = (String) autoConnectList.getSelectedValue();
        if ( val != null )
        {
            newAutoConnectHostTF.setText( val );
        }
    }

    /////////////////////// inner classes //////////////////////////////

    private class ConnectCatcherHandler implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            connectFromHostCatcher();
        }
    }

    private class ResetCatcherHandler implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            mHostMgr.resetCaughtHosts();
        }
    }

    private class AutoConnectHandler implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            toggleAutoConnect();
        }
    }

    private class SelectionHandler implements ListSelectionListener
    {
        public void valueChanged(ListSelectionEvent e)
        {
            mainFrame.refreshAllActions();
        }
    }

    private class ConnectHostHandler implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            connectHost();
        }
    }

    private class AutoCleanupHandler implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            toggleAutoCleanup();
        }
    }

    private class ConnectHostKeyHandler implements KeyListener
    {
        public void keyTyped(KeyEvent event)
        {
        }

        public void keyReleased(KeyEvent event)
        {
            if (event.getKeyCode() == KeyEvent.VK_ENTER)
            {
                if (mConnectHostText.getText().trim().length() > 0)
                {
                    connectHost();
                }
            }
            mainFrame.refreshAction("ActionHostConnect");
        }

        public void keyPressed(KeyEvent event)
        {
        }
    }

    private class UpdateStatHandler implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            updateStat();
        }
    }

    private class MouseHandler extends MouseAdapter implements MouseListener
    {
        public void mouseReleased(MouseEvent e)
        {
            if (e.isPopupTrigger())
            {
                popupMenu((Component)e.getSource(), e.getX(), e.getY());
            }
        }

        public void mousePressed(MouseEvent e)
        {
            if (e.isPopupTrigger())
            {
                popupMenu((Component)e.getSource(), e.getX(), e.getY());
            }
        }

        private void popupMenu(Component source, int x, int y)
        {
            if (source == mNetTable)
            {
                mHostPopupMenu.show(source, x, y);
            }
            else if ( source == mCatcherList )
            {
                mCatcherPopupMenu.show(source, x, y);
            }
        }
    }

    private class CellRenderer extends DefaultTableCellRenderer
    {
        public Component getTableCellRendererComponent(
            JTable table, Object value, boolean isSelected,
            boolean hasFocus, int row, int column)
        {
            super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
            this.setForeground(Color.black);
//			this.setBackground(Color.black);

            if (table == mNetTable)
            {
                if (row < mHostMgr.getNetworkHostCount() )
                {
                    int		srow = mNetSorter.indexes[row];
                    Host	host = (Host)mHostMgr.getNetworkHostAt( srow );
                    if ( host == null )
                    {
                        return this;
                    }
                    switch (host.getStatus())
                    {
                        case Host.sStatusNotConnected:
                            break;

                        case Host.sStatusError:
//							this.setForeground(Color.darkGray);
                            this.setForeground(Color.gray);
                            break;

                        case Host.sStatusConnecting:
                        case Host.sStatusAccepting:
                            this.setForeground(Color.red);
                            break;

                        case Host.sStatusConnected:
                        {
                            this.setForeground(Color.blue);
                            switch (column)
                            {
                                case 2:
                                    if (host.dropPacketsInRed())
                                        this.setForeground(Color.red);
                                    break;
                                case 3:
                                    if (host.sendQueueInRed())
                                        this.setForeground(Color.red);
                                    break;
                                case 4:
                                    if (host.latencyInRed())
                                        this.setForeground(Color.red);
                                    break;
                            }
                            break;
                        }
                    }
                }
            }
            return this;
        }
    }
}
