/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.gui.tabs;

import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import javax.swing.table.*;
import javax.swing.event.*;

import phex.MainFrame;
import phex.query.Search;
import phex.query.SearchContainer;
import phex.config.Cfg;
import phex.utils.TableSorter;
import phex.utils.URLUtil;
import phex.msg.MsgManager;
import phex.download.DownloadManager;
import phex.gui.models.SearchTableModel;
import phex.interfaces.IFind;
import phex.ServiceManager;
import phex.gui.common.*;
import phex.event.SearchChangeListener;
import phex.event.SearchChangeEvent;
import phex.gui.models.SearchComboBoxModel;
import phex.gui.renderer.*;
import phex.gui.dialogs.DownloadConfigDialog;
import phex.download.RemoteFile;
import phex.download.DownloadFile;
import phex.dialogues.DlgAddAsCandidate;
import phex.utils.IPUtils;
import phex.utils.Localizer;

/**
 * The SearchTab Panel.
 */
public class SearchTab extends JPanel implements IFind
{
    private DownloadManager mDownloadMgr = ServiceManager.getDownloadManager();
    private SearchContainer searchContainer;
    private boolean mSearchTextTyped = true;
    private MainFrame mainFrame;
    private JLabel mSearchResultLabel;
    private JTextField mSearchText;
    private JTextField mMaxSearch;
    private JTextField mMinSpeed;
    private JTextField mMinimumSizeFilterText;
    private JTextField mMaximumSizeFilterText;
    private JTextField mSearchFilterText;
    private JButton mSearchStopButton;
    private JButton removeSearchBtn;
    private JButton mSearchButton;
    private JComboBox mSearchResultCombo;
    private JCheckBox mApplyFilterCheck;
    private SearchComboBoxModel mSearchResultModel;
    private JTable mSearchTable;
    private AbstractTableModel mSearchModel;
    private TableSorter mSearchSorter;
    private JPopupMenu mSearchResultPopupMenu;

    public SearchTab( MainFrame frame )
    {
        mainFrame = frame;
        searchContainer = ServiceManager.getQueryManager().getSearchContainer();
        mSearchResultPopupMenu = new JPopupMenu();
        mainFrame.populatePopupMenu(mSearchResultPopupMenu, "SearchResultTable.PopupMenu");
    }

    /////////////// IFind Start ///////////////////////
    public void findInResult(boolean bMatchCase, boolean bFindDown, String searchText)
    {
        int startingIndex;
        int rowCount = mSearchTable.getSelectedRowCount();

        if (rowCount <= 0)
        {
            startingIndex = -1;
        }
        else
        {
            int[] rows = mSearchTable.getSelectedRows();
            startingIndex = rows[0];
        }

        if (bFindDown)
        {
            startingIndex++;
        }
        else
        {
            startingIndex--;
        }

        if (!bMatchCase)
            searchText = searchText.toLowerCase();

        int i = startingIndex;
        rowCount = mSearchSorter.getRowCount();
        int colCount = mSearchSorter.getColumnCount();

        while (true)
        {
            if (bFindDown)
            {
                if (i >= rowCount)
                    break;
            }
            else
            {
                if (i < 0)
                    break;
            }

            for (int j = 0; j < colCount; j++)
            {
                Object	obj = mSearchSorter.getValueAt(i, j);
                if (obj == null)
                    continue;

                int		index;

                if (bMatchCase)
                    index = obj.toString().indexOf(searchText);
                else
                    index = obj.toString().toLowerCase().indexOf(searchText);

                if (index != -1)
                {
                    mSearchTable.clearSelection();
                    mSearchTable.addRowSelectionInterval(i, i);
                    mSearchTable.scrollRectToVisible(mSearchTable.getCellRect(i, 0, true));
                    return;
                }
            }

            if (bFindDown)
            {
                i++;
            }
            else
            {
                i--;
            }
        }

        JOptionPane.showMessageDialog(this, "No more result found.", "Find", JOptionPane.ERROR_MESSAGE);
    }
    /////////////// IFind End ///////////////////////

    public void initComponent()
    {
        GridBagConstraints constraints;
        setLayout(new BorderLayout());
        setBorder(BorderFactory.createLoweredBevelBorder());

        JPanel topSearchPanel = new JPanel( new GridBagLayout() );
        // take over this strange border construct ;-)
        topSearchPanel.setBorder(
            BorderFactory.createCompoundBorder(
                BorderFactory.createCompoundBorder(
                    BorderFactory.createEmptyBorder(8, 8, 8, 8),
                    BorderFactory.createEtchedBorder()),
                BorderFactory.createEmptyBorder(8, 8, 8, 8)));

        JPanel panel = new JPanel( new GridBagLayout() );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.gridwidth = 4;
            constraints.weightx = 1;
            constraints.anchor = GridBagConstraints.NORTHWEST;
        topSearchPanel.add( panel, constraints );

        mSearchText = new JTextField( ServiceManager.sCfg.mSearchLastSearch, 40 );
        mSearchText.addKeyListener( new SearchKeyHandler() );
        GUIUtils.setToolTipText( mSearchText, Localizer.getString( "TTTSearchText" ) );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.weightx = 1;
            constraints.anchor = GridBagConstraints.WEST;
            constraints.insets = new Insets( 0, 0, 3, 5 );
        panel.add( mSearchText, constraints );

        mSearchButton = new JButton( Localizer.getString( "Search") );
        mSearchButton.addActionListener( new SearchButtonHandler() );
        GUIUtils.setToolTipText(mSearchButton, Localizer.getString( "TTTSearch") );
        mainFrame.addRefreshComponent("ActionQuerySearch", mSearchButton);
            constraints = new GridBagConstraints();
            constraints.gridx = 2;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.insets = new Insets( 0, 0, 3, 5 );
        panel.add( mSearchButton, constraints );

        mApplyFilterCheck = new JCheckBox( Localizer.getString( "HideFilteredHosts" ),
            ServiceManager.sCfg.mApplyFilterdHosts);
        GUIUtils.setToolTipText( mApplyFilterCheck,
            Localizer.getString( "TTTHideFilteredHosts" ) );
        mApplyFilterCheck.addActionListener(new ApplyFilterHandler());
            constraints = new GridBagConstraints();
            constraints.gridx = 3;
            constraints.gridy = 0;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.insets = new Insets( 0, 0, 3, 0 );
        panel.add( mApplyFilterCheck, constraints );

        panel = new JPanel( new GridBagLayout() );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 1;
            constraints.weightx = 1;
            constraints.anchor = GridBagConstraints.NORTHWEST;
        topSearchPanel.add( panel, constraints );

        JLabel label = new JLabel( Localizer.getString( "MaxSearchResults" ) + ": ");
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 1;
            constraints.anchor = GridBagConstraints.NORTHEAST;
            constraints.insets = new Insets( 0, 0, 3, 5 );
        panel.add( label, constraints );

        String lastMaxSearch = String.valueOf( ServiceManager.sCfg.mSearchMaxSearch );
        mMaxSearch = new IntegerTextField( lastMaxSearch, 6, 6 );
        mMaxSearch.addKeyListener(new SearchKeyHandler());
        GUIUtils.setToolTipText( mMaxSearch,
            Localizer.getString( "TTTMaxSearchResults" ) );
            constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 1;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.insets = new Insets( 0, 0, 3, 10 );
        panel.add( mMaxSearch, constraints );

        label = new JLabel( Localizer.getString( "ResultFilter" ) + ": " );
            constraints = new GridBagConstraints();
            constraints.gridx = 2;
            constraints.gridy = 1;
            constraints.anchor = GridBagConstraints.NORTHEAST;
            constraints.insets = new Insets( 0, 0, 3, 5 );
        panel.add( label, constraints );

        mSearchFilterText = new JTextField( ServiceManager.sCfg.mSearchFilter, 16 );
        mSearchFilterText.addKeyListener(new SearchKeyHandler());
        GUIUtils.setToolTipText(mSearchFilterText,
            Localizer.getString( "TTTResultFilter" ) );
            constraints = new GridBagConstraints();
            constraints.gridx = 3;
            constraints.gridy = 1;
            constraints.weightx = 1;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.insets = new Insets( 0, 0, 3, 0 );
        panel.add( mSearchFilterText, constraints );

        label = new JLabel( Localizer.getString( "MinSpeed" ) + ": " );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 2;
            constraints.anchor = GridBagConstraints.NORTHEAST;
            constraints.insets = new Insets( 0, 0, 3, 5 );
        panel.add( label, constraints );

        String lastMinSpeed = String.valueOf( ServiceManager.sCfg.mSearchMinSpeed );
        mMinSpeed = new IntegerTextField( lastMinSpeed, 8, 10 );
        mMinSpeed.addKeyListener( new SearchKeyHandler() );
        GUIUtils.setToolTipText(mMinSpeed,
            Localizer.getString( "TTTMinSpeed" ) );
            constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 2;
            constraints.weightx = 1;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.insets = new Insets( 0, 0, 3, 10 );
        panel.add( mMinSpeed, constraints );

        label = new JLabel( Localizer.getString( "MinFileSize" ) + ": ");
            constraints = new GridBagConstraints();
            constraints.gridx = 2;
            constraints.gridy = 2;
            constraints.anchor = GridBagConstraints.NORTHEAST;
            constraints.insets = new Insets( 0, 0, 3, 5 );
        panel.add( label, constraints );

        String firstMinSizeFilterValue = String.valueOf( ServiceManager.sCfg.mMinimumFileSize );
        mMinimumSizeFilterText = new IntegerTextField( firstMinSizeFilterValue, 10, 13 );
        mMinimumSizeFilterText.addKeyListener( new SearchKeyHandler() );
        GUIUtils.setToolTipText( mMinimumSizeFilterText,
            Localizer.getString( "TTTMinFileSize" ) );
            constraints = new GridBagConstraints();
            constraints.gridx = 3;
            constraints.gridy = 2;
            constraints.anchor = GridBagConstraints.NORTHWEST;
            constraints.insets = new Insets( 0, 0, 3, 0 );
        panel.add( mMinimumSizeFilterText, constraints );


/*  I see  no reason to have this visible since people generally don't want to specify. But the work is mostly done if someone wants it.
        JPanel maxSizeFilterPanel = new JPanel( new BorderLayout() );
        String firstMaxSizeFilterValue = String.valueOf( ServiceManager.sCfg.mMaximumFileSize );
        mMaximumSizeFilterText = TextFieldFactory.newTextField( firstMaxSizeFilterValue, 19 );
        mMaximumSizeFilterText.addKeyListener( new SearchKeyHandler() );
        GUIUtils.setToolTipText( mMaximumSizeFilterText, "The results will all be smaller or equal to the maximum file size." );
        maxSizeFilterPanel.add( BorderLayout.WEST, new JLabel( "Maximum file size in Bytes: " ) );
        maxSizeFilterPanel.add( BorderLayout.CENTER, mMaximumSizeFilterText );
*/

        JPanel resultSelectPanel = new JPanel( new GridBagLayout() );
        mSearchResultModel = new SearchComboBoxModel();
        mSearchResultModel.addListDataListener( new SearchUpdateChangeListener() );
        mSearchResultCombo = new JComboBox( mSearchResultModel );
        mSearchResultCombo.setRenderer( new SearchListRenderer() );
        mSearchResultCombo.setEditable(false);
        mSearchResultCombo.addActionListener(new SearchComboActionHandler());
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.weightx = 1;
            constraints.fill = GridBagConstraints.HORIZONTAL;
            constraints.anchor = GridBagConstraints.WEST;
        resultSelectPanel.add( mSearchResultCombo, constraints );

        mSearchStopButton = new JButton("Stop");
        GUIUtils.setToolTipText(mSearchStopButton, "Stop the selected search.");
        mSearchStopButton.setEnabled(false);
        mSearchStopButton.addActionListener( new SearchButtonHandler());
            constraints = new GridBagConstraints();
            constraints.gridx = 1;
            constraints.gridy = 0;
            constraints.insets = new Insets( 0, 6, 0, 0 );
            constraints.anchor = GridBagConstraints.WEST;
        resultSelectPanel.add( mSearchStopButton, constraints );

        removeSearchBtn = new JButton( Localizer.getString( "Remove" ) );
        GUIUtils.setToolTipText( removeSearchBtn,
            Localizer.getString( "RemoveSearchTTT" ) );
        removeSearchBtn.setEnabled( true );
        removeSearchBtn.addActionListener( new SearchButtonHandler() );
            constraints = new GridBagConstraints();
            constraints.gridx = 2;
            constraints.gridy = 0;
            constraints.insets = new Insets( 0, 3, 0, 0 );
            constraints.anchor = GridBagConstraints.WEST;
        resultSelectPanel.add( removeSearchBtn, constraints );


        JPanel resultTablePanel = new JPanel(new BorderLayout());
        mSearchModel = new SearchTableModel( mainFrame );
        mSearchSorter = new TableSorter(mSearchModel);
        mSearchTable = new JTable(mSearchSorter);
        mSearchTable.getSelectionModel().addListSelectionListener(new SelectionHandler());

        // Set up cell renderers. Just use our standard default renderer to class relationships.
        DefaultPhexCellRenderers.setDefaultPhexCellRenderers( mSearchTable );

        GUIUtils.setToolTipText(mSearchTable, "Search results.  Perform sorting by clicking on the column.");
        mSearchTable.addMouseListener(new MouseHandler());
        mSearchSorter.addMouseListenerToHeaderInTable(mSearchTable);
        mSearchResultLabel = new JLabel("Search Result:");

        JPanel	searchButtonsPanel = new JPanel(new FlowLayout());

        JButton	quickDownloadBtn =
            new JButton( Localizer.getString( "QuickDownload" ) );
        GUIUtils.setToolTipText( quickDownloadBtn,
            Localizer.getString( "QuickDownloadTTT" ) );
        quickDownloadBtn.addActionListener( new QuickDownloadHandler() );
        mainFrame.addRefreshComponent("ActionTransferDownload", quickDownloadBtn);
        searchButtonsPanel.add( quickDownloadBtn );

        JButton downloadBtn = new JButton( Localizer.getString( "Download" ) );
        GUIUtils.setToolTipText( downloadBtn,
            Localizer.getString( "DownloadTTT" ) );
        downloadBtn.addActionListener( new DownloadHandler() );
        mainFrame.addRefreshComponent("ActionTransferDownload", downloadBtn);
        searchButtonsPanel.add( downloadBtn );

        JButton	asCandidateButton = new JButton("Add As Candidate");
        GUIUtils.setToolTipText(asCandidateButton, "Add the selected file as download candidate to existing download.");
        asCandidateButton.addActionListener(new AddAsCandidateHandler());
        mainFrame.addRefreshComponent("ActionTransferAddAsCandidate", asCandidateButton);
        searchButtonsPanel.add(asCandidateButton);

        resultTablePanel.add(BorderLayout.NORTH, mSearchResultLabel);
        resultTablePanel.add(BorderLayout.CENTER, new JScrollPane(mSearchTable));
        resultTablePanel.add(BorderLayout.SOUTH, searchButtonsPanel);

        JPanel	resultPanel = new JPanel( new GridBagLayout() );
            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 0;
            constraints.weightx = 1;
            constraints.fill = GridBagConstraints.BOTH;
            constraints.anchor = GridBagConstraints.NORTH;
        resultPanel.add( resultSelectPanel, constraints );

            constraints = new GridBagConstraints();
            constraints.gridx = 0;
            constraints.gridy = 1;
            constraints.weightx = 1;
            constraints.weighty = 1;
            constraints.fill = GridBagConstraints.BOTH;
            constraints.anchor = GridBagConstraints.NORTH;
        resultPanel.add( resultTablePanel, constraints );
        resultPanel.setBorder(
            BorderFactory.createCompoundBorder(
                BorderFactory.createCompoundBorder(
                    BorderFactory.createEmptyBorder(8, 8, 8, 8),
                    BorderFactory.createEtchedBorder()),
                BorderFactory.createEmptyBorder(8, 8, 8, 8)));

        JPanel	searchMainPanel = new JPanel(new BorderLayout());
        searchMainPanel.add(BorderLayout.NORTH, topSearchPanel);
        searchMainPanel.add(BorderLayout.CENTER, resultPanel);

        add(BorderLayout.CENTER, searchMainPanel);
    }

    public Search getSelectedSearchResult()
    {
        int	searchIndex = mSearchResultCombo.getSelectedIndex();
        if (searchIndex >= 0 && searchIndex < searchContainer.getSearchCount())
        {
            return searchContainer.getSearchAt( searchIndex );
        }

        return null;
    }

    public void search()
    {
        String searchStr = mSearchText.getText().trim();
        String maxSearchStr = mMaxSearch.getText().trim();
        String minSpeedStr = mMinSpeed.getText().trim();
        String minSizeStr = mMinimumSizeFilterText.getText().trim();
        //String maxSizeStr = mMaximumSizeFilterText.getText().trim();
        String searchFilter = mSearchFilterText.getText().trim();

        int maxSearch;
        try
        {
            maxSearch = Integer.parseInt( maxSearchStr );
        }
        catch ( NumberFormatException exp )
        {
            JOptionPane.showMessageDialog(this, "Maximum search results should be an integer.",
                "Invalid Maximum Results", JOptionPane.ERROR_MESSAGE);
            return;
        }

        int minSpeed = 0;
        try
        {
            if ( minSpeedStr.length() > 0 )
            {
                minSpeed = Integer.parseInt( minSpeedStr );
            }
        }
        catch ( NumberFormatException exp )
        {
            JOptionPane.showMessageDialog(this, "Minimum speed should be an integer.",
                "Invalid Minimum Speed", JOptionPane.ERROR_MESSAGE);
            return;
        }

        long minSize = 0;
        try
        {
            if ( minSizeStr.length() > 0 )
            {
                minSize = Integer.parseInt( minSizeStr );
            }
        }
        catch ( NumberFormatException exp )
        {
            JOptionPane.showMessageDialog(this, "Minimum size should be an integer.",
                "Invalid Minimum Size", JOptionPane.ERROR_MESSAGE);
            return;
        }

        // PH: All of these need to be fixed!
        ServiceManager.sCfg.mSearchLastSearch = searchStr;
        ServiceManager.sCfg.mSearchMaxSearch = maxSearch;
        ServiceManager.sCfg.mSearchMinSpeed = minSpeed;
        ServiceManager.sCfg.mMinimumFileSize = minSize;
        //ServiceManager.sCfg.mMaximumFileSize = Long.parseLong( maxSizeStr );
        ServiceManager.sCfg.mSearchFilter = searchFilter;
        ServiceManager.sCfg.save();

        if ( searchStr.length() == 0 )
        {
            return;
        }

        if ( searchStr.length() < Cfg.MIN_SEARCH_TERM_LENGTH )
        {
            Object[] objArr = new Object[ 1 ];
            objArr[ 0 ] = new Integer( Cfg.MIN_SEARCH_TERM_LENGTH );
            GUIUtils.showErrorMessage( Localizer.getFormatedString(
                "MinSearchTerm", objArr ) );
            mSearchText.requestFocus();
            return;
        }

        mSearchTextTyped = false;
        mainFrame.refreshAllActions();

        Search.FileSizeConstraints sizeConstraints =
          new Search.FileSizeConstraints( ServiceManager.sCfg.mMinimumFileSize,
          ServiceManager.sCfg.mMaximumFileSize );

        Search search = searchContainer.createSearch( searchStr, minSpeed, sizeConstraints );
    }

    public void stopSearch()
    {
        Search search = getSelectedSearchResult();
        if (search != null)
        {
            search.stopSearching();
        }
    }

    private void removeSearch()
    {
        Search search = getSelectedSearchResult();
        if (search != null)
        {
            searchContainer.removeSearch( search );
        }
    }


    void addAsCandidate()
    {
        Search search = getSelectedSearchResult();
        if (search == null)
            return;
        int				rowCount = mSearchTable.getSelectedRowCount();
        if (rowCount <= 0)
            return;

        int[] rows = mSearchTable.getSelectedRows();
        int row = rows[0];
        row = mSearchSorter.indexes[row];
        RemoteFile	rfile = search.getQueryHit( row );
        RemoteFile	dfile = new RemoteFile(rfile);

        DlgAddAsCandidate dlg = new DlgAddAsCandidate( mainFrame, dfile);
        dlg.setVisible(true);
        if (dlg.getCancel())
            return;

        DownloadFile	download = dlg.getDownloadFile();
        if (download == null)
        {
            return;
        }
        download.addRemoteCandidate(dfile);

        mainFrame.setSelectedTab( 3 );
        mainFrame.refreshAllActions();
    }

    public RemoteFile getSelectedRemoteFile()
    {
        Search search = getSelectedSearchResult();
        if (search == null)
            return null;

        if (mSearchTable.getSelectedRowCount() < 1)
            return null;

        int[]		rows = mSearchTable.getSelectedRows();
        int			row = mSearchSorter.indexes[rows[0]];

        RemoteFile rfile = (RemoteFile)search.getQueryHit( row );
        return rfile;
    }


    public void filterDownloadHost()
    {
        Search search = getSelectedSearchResult();
        if (search == null)
        {
            return;
        }

        int rowCount = mSearchTable.getSelectedRowCount();
        if (rowCount <= 0)
        {
            return;
        }

        int[] rows = mSearchTable.getSelectedRows();
        int addedCount = 0;

        for (int i = 0; i < rowCount; i++)
        {
            int row = rows[i];
            row = mSearchSorter.indexes[row];
            RemoteFile	rfile = search.getQueryHit( row );
            ServiceManager.sCfg.mFilteredSearchHosts.addElement(IPUtils.splitIP2Parts(rfile.getRemoteHost()));
        }
        ServiceManager.sCfg.save();
    }


    public int getResultSelectedCount()
    {
        return (mSearchTable.getSelectedRowCount());
    }

    public void copyURL()
    {
        Search search = getSelectedSearchResult();
        if (search == null)
        {
            return;
        }

        int rowCount = mSearchTable.getSelectedRowCount();
        if (rowCount < 1)
        {
            return;
        }

        int[] rows = mSearchTable.getSelectedRows();
        int row = mSearchSorter.indexes[rows[0]];
        RemoteFile rfile = search.getQueryHit( row );

        String url = URLUtil.encodeURL(rfile.getURL());
        Toolkit.getDefaultToolkit().getSystemClipboard().setContents
                (new StringSelection(url), null);
    }

    public void quickDownload()
    {
        Search search = getSelectedSearchResult();
        if (search == null)
        {
            return;
        }
        int rowCount = mSearchTable.getSelectedRowCount();
        if (rowCount <= 0)
        {
            return;
        }

        int[] rows = mSearchTable.getSelectedRows();
        int addedCount = 0;

        for (int i = 0; i < rowCount; i++)
        {
            int row = rows[i];
            row = mSearchSorter.indexes[row];
            RemoteFile rfile = search.getQueryHit( row );
            RemoteFile dfile = new RemoteFile(rfile);
            if (!mDownloadMgr.addFileToDownload(dfile))
            {
                String	msg = "Another file with the same name '" + dfile.getFilename() + "' is being downloading now.\n";
                msg += "Please wait until the other file has completed its download.";
                JOptionPane.showMessageDialog(ServiceManager.getManager().getMainFrame(),
                                              msg, "Duplicate File", JOptionPane.INFORMATION_MESSAGE);
            }
            else
            {
                addedCount++;
                rfile.setInDownloadQueue( true );
            }
        }

        if (addedCount > 0)
        {
            mainFrame.setSelectedTab( 3 );
        }
        mainFrame.refreshAllActions();
    }


    public boolean getSearchTextTyped()
    {
        return mSearchTextTyped;
    }

    private void updateSearchResultLabel()
    {
        Search	search = getSelectedSearchResult();
        if (search != null)
        {
            updateSearchResultLabel(search);
            return;
        }

        mSearchStopButton.setEnabled(false);
        mainFrame.refreshAllActions();
        mSearchResultLabel.setText("Search Result:");
    }

    private void updateSearchResultLabel(Search search)
    {
        final Search innerSearch = search;
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
            new Runnable()
            {
                public void run()
                {
                    StringBuffer buffer = new StringBuffer( "Search Result:    " );
                    buffer.append( innerSearch.getQueryHitCount() );
                    buffer.append( " returned" );
                    if ( innerSearch.isSearching() )
                    {
                        buffer.append( ", searching..." );
                    }
                    String labelText = buffer.toString();
                    if ( !labelText.equals( mSearchResultLabel.getText() ) )
                    {
                        mSearchStopButton.setEnabled( innerSearch.isSearching() );
                        mainFrame.refreshAllActions();
                        mSearchResultLabel.setText( labelText );
                    }
                }
            });
    }

    //////////////////// inner classes ////////////////////////////////

    private class SearchUpdateChangeListener implements ListDataListener
    {
        public void contentsChanged( final ListDataEvent event )
        {
            if ( event.getType() == ListDataEvent.CONTENTS_CHANGED )
            {
                Search search = getSelectedSearchResult();
                if ( search == null )
                {
                    return;
                }
                updateSearchResultLabel( search );
                GUIUtils.fireTableChanged( mSearchTable, mSearchModel );
            }
            else
            {
                updateSearchResultLabel();
            }
        }

        public void intervalRemoved( ListDataEvent event )
        {
        }

        public void intervalAdded( ListDataEvent event )
        {
        }
    }

    private class SearchKeyHandler extends KeyAdapter
    {
        public void keyReleased(KeyEvent event)
        {
            if (event.getSource() == mSearchText && event.getKeyCode() == KeyEvent.VK_ENTER)
            {
                if (mSearchTextTyped)
                {
                    search();
                }
            }
            else
            {
                mSearchTextTyped = true;
                mainFrame.refreshAllActions();
            }
        }
    }

    private class ApplyFilterHandler implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            ServiceManager.sCfg.mApplyFilterdHosts = mApplyFilterCheck.isSelected();
            ServiceManager.sCfg.save();
        }
    }

    private class SearchComboActionHandler implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            updateSearchResultLabel();
            GUIUtils.fireTableChanged(mSearchTable, mSearchModel);
        }
    }

    private class DownloadHandler implements ActionListener
    {
        public void actionPerformed( ActionEvent e )
        {
            Search search = getSelectedSearchResult();
            if (search == null)
            {
                return;
            }
            int row = mSearchTable.getSelectedRow();
            if ( row == -1 )
            {
                return;
            }

            row = mSearchSorter.indexes[ row ];
            RemoteFile rfile = search.getQueryHit( row );
            RemoteFile dfile = new RemoteFile(rfile);

            DownloadConfigDialog dialog = new DownloadConfigDialog( dfile );
            dialog.show();

/*          if (!mDownloadMgr.addFileToDownload(dfile))
            {
                String	msg = "Another file with the same name '" + dfile.getFilename() + "' is being downloading now.\n";
                msg += "Please wait until the other file has completed its download.";
                JOptionPane.showMessageDialog(ServiceManager.getManager().getMainFrame(),
                                              msg, "Duplicate File", JOptionPane.INFORMATION_MESSAGE);
            }
            else
            {
                rfile.setInDownloadQueue( true );
            }

            mainFrame.refreshAllActions();
            */
        }
    }

    private class QuickDownloadHandler implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            quickDownload();
        }
    }


    private class AddAsCandidateHandler implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            addAsCandidate();
        }
    }

    private class SearchButtonHandler implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            Object source = e.getSource();
            if ( source == mSearchButton )
            {
                search();
            }
            else if ( source == mSearchStopButton )
            {
                stopSearch();
            }
            else if ( source == removeSearchBtn )
            {
                removeSearch();
            }
        }
    }

    private class SelectionHandler implements ListSelectionListener
    {
        public void valueChanged(ListSelectionEvent e)
        {
            mainFrame.refreshAllActions();
        }
    }

    private class MouseHandler extends MouseAdapter implements MouseListener
    {
        public void mouseClicked(MouseEvent e)
        {
            if (e.getClickCount() == 2)
            {
                if (e.getSource() == mSearchTable)
                {
                    quickDownload();
                }
            }
        }


        public void mouseReleased(MouseEvent e)
        {
            if (e.isPopupTrigger())
            {
                popupMenu((Component)e.getSource(), e.getX(), e.getY());
            }
        }


        public void mousePressed(MouseEvent e)
        {
            if (e.isPopupTrigger())
            {
                popupMenu((Component)e.getSource(), e.getX(), e.getY());
            }
        }


        private void popupMenu(Component source, int x, int y)
        {
            if (source == mSearchTable)
            {
                mSearchResultPopupMenu.show(source, x, y);
            }
        }
    }
}
