/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.gui.tabs;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import java.util.Vector;
import phex.MainFrame;
import phex.ServiceManager;
import phex.share.ShareManager;
import phex.share.ShareFile;
import phex.share.UploadFile;
import phex.share.UploadFileContainer;
import phex.share.FileAdministration;
import phex.utils.StrUtil;
import phex.utils.TableSorter;
import phex.utils.Localizer;
import phex.gui.common.GUIUtils;
import phex.gui.models.SharedFilesTableModel;
import phex.gui.models.UploadFilesTableModel;
import phex.gui.renderer.*;
import phex.event.SharedFilesChangeListener;

public class ShareTab extends JPanel
{
    private ShareManager mShareMgr;
    private UploadFileContainer uploadContainer;
    private FileAdministration fileAdmin;
    private MainFrame mainFrame;
    private JTable mUploadTable;
    private AbstractTableModel mUploadModel;
    private TableSorter mUploadSorter;
    private JTable mShareFilesTable;
    private AbstractTableModel mShareFilesModel;
    private TableSorter mShareFilesSorter;
    private JLabel sharedFilesLabel;

    public ShareTab( MainFrame frame )
    {
        mainFrame = frame;
        mShareMgr = ServiceManager.getShareManager();
        fileAdmin = mShareMgr.getFileAdministration();
        uploadContainer = mShareMgr.getUploadFileContainer();
    }

    public void initComponent()
    {
        setLayout(new BorderLayout());
        JPanel uploadTablePanel = new JPanel(new BorderLayout());
        mUploadModel = new UploadFilesTableModel();
        mUploadSorter = new TableSorter(mUploadModel);
        mUploadTable = new JTable(mUploadSorter);
        mUploadTable.getSelectionModel().addListSelectionListener(new SelectionHandler());
        mUploadSorter.addMouseListenerToHeaderInTable(mUploadTable);
        uploadTablePanel.add(BorderLayout.NORTH, new JLabel("Uploads:"));
        uploadTablePanel.add(BorderLayout.CENTER, new JScrollPane(mUploadTable));

        JPanel	uploadButtonsPanel = new JPanel(new FlowLayout());
        JButton	abortUploadButton = new JButton("Abort Upload");
        JButton	clearUploadButton = new JButton("Clear Completed");
        mainFrame.addRefreshComponent("ActionShareAbort", abortUploadButton);
        mainFrame.addRefreshComponent("ActionShareClear", clearUploadButton);

        uploadButtonsPanel.add(abortUploadButton);
        uploadButtonsPanel.add(clearUploadButton);
        abortUploadButton.addActionListener(new AbortUploadHandler());
        clearUploadButton.addActionListener(new ClearUploadHandler());

        JPanel mUpload = new JPanel(new BorderLayout());
        mUpload.add(BorderLayout.CENTER, uploadTablePanel);
        mUpload.add(BorderLayout.SOUTH, uploadButtonsPanel);
        mUpload.setBorder(
            BorderFactory.createCompoundBorder(
                BorderFactory.createCompoundBorder(
                    BorderFactory.createEmptyBorder(8, 8, 8, 8),
                    BorderFactory.createEtchedBorder()),
                BorderFactory.createEmptyBorder(8, 8, 8, 8)));

        JPanel shareTablePanel = new JPanel(new BorderLayout());
        mShareFilesModel = new SharedFilesTableModel();
        mShareFilesSorter = new TableSorter(mShareFilesModel);
        mShareFilesTable = new JTable(mShareFilesSorter);
        mShareFilesTable.getSelectionModel().addListSelectionListener(new SelectionHandler());

        // Set up cell renderers. Just use our standard default renderer to class relationships.
        DefaultPhexCellRenderers.setDefaultPhexCellRenderers( mUploadTable );

        // set up cell renderer to display colour based on share upload status.
        // TODO decouple this cell renderer stuff
        CellRenderer cellRenderer = new CellRenderer();
        mUploadTable.getColumn( mUploadModel.getColumnName( 0 ) ).setCellRenderer( cellRenderer );
        mUploadTable.getColumn( mUploadModel.getColumnName( 1 ) ).setCellRenderer( cellRenderer );
        mUploadTable.getColumn( mUploadModel.getColumnName( 2 ) ).setCellRenderer( cellRenderer );
        mUploadTable.getColumn( mUploadModel.getColumnName( 6 ) ).setCellRenderer( cellRenderer );

        // increase table height a bit to display progress bar string better...
        GUIUtils.adjustTableProgresssBarHeight( mUploadTable );


        mShareFilesSorter.addMouseListenerToHeaderInTable(mShareFilesTable);
        JPanel	shareButtonsPanel = new JPanel(new FlowLayout());
        JButton	shareRescanButton = new JButton("Rescan");
        shareButtonsPanel.add(shareRescanButton);
        shareRescanButton.addActionListener(new ShareRescanHandler());

        sharedFilesLabel = new JLabel( Localizer.getString( "SharedFiles" ) + ":" );
        fileAdmin.addSharedFilesChangeListener(
            new SharedFilesChangeHandler() );
        shareTablePanel.add(BorderLayout.NORTH, sharedFilesLabel );
        shareTablePanel.add(BorderLayout.CENTER, new JScrollPane(mShareFilesTable));
        shareTablePanel.add(BorderLayout.SOUTH, shareButtonsPanel);
        shareTablePanel.setBorder(
            BorderFactory.createCompoundBorder(
                BorderFactory.createCompoundBorder(
                    BorderFactory.createEmptyBorder(8, 8, 8, 8),
                    BorderFactory.createEtchedBorder()),
                BorderFactory.createEmptyBorder(8, 8, 8, 8)));

        mUpload.setPreferredSize(new Dimension(780, 230));
        shareTablePanel.setPreferredSize(new Dimension(780, 200));
        JSplitPane splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
        splitPane.setDividerSize(6);
                splitPane.setOneTouchExpandable(true);
        splitPane.setTopComponent(mUpload);
        splitPane.setBottomComponent(shareTablePanel);

        add(BorderLayout.CENTER, splitPane);
    }

    /**
     * This is overloaded to update the table size for the progress bar on
     * every UI update. Like font size change!
     */
    public void updateUI()
    {
        super.updateUI();
        if ( mUploadTable != null )
        {
            // increase table height a bit to display progress bar string better...
            GUIUtils.adjustTableProgresssBarHeight( mUploadTable );
        }
    }

    public void abortUpload()
    {
        int row = mUploadTable.getSelectedRow();
        if ( row < 0 || row >= uploadContainer.getUploadListSize() )
        {
            return;
        }
        row = mUploadSorter.indexes[row];
        UploadFile file = uploadContainer.getUploadFileAt( row );
        if ( file == null )
        {
            return;
        }
        file.abortUpload();
        mainFrame.refreshAllActions();
    }

    public boolean isUploadSelected()
    {
        return (mainFrame.isUploadTabSelected() &&
                mUploadTable.getSelectedRowCount() > 0 &&
                uploadContainer.getUploadListSize() > 0);
    }

    public void clearUpload()
    {
        uploadContainer.cleanUploadFileList();
        mainFrame.refreshAllActions();
    }

    private class AbortUploadHandler implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            abortUpload();
        }
    }

    private class ClearUploadHandler implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            clearUpload();
        }
    }

    private class ShareRescanHandler implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            mShareMgr.getFileAdministration().rescan();
        }
    }

    private class SelectionHandler implements ListSelectionListener
    {
        public void valueChanged(ListSelectionEvent e)
        {
            mainFrame.refreshAllActions();
        }
    }

    private class SharedFilesChangeHandler implements SharedFilesChangeListener
    {
        /**
         * Called if a shared file changed.
         */
        public void sharedFileChanged( int position )
        {
        }

        /**
         * Called if a shared file was added.
         */
        public void sharedFileAdded( int position )
        {
            updateLabel();
        }

        /**
         * Called if all shared files changed.
         */
        public void allSharedFilesChanged( )
        {
            updateLabel();
        }

        // TODO: Don't want a duplicated initial string. Use this method.
        private void updateLabel()
        {
            StringBuffer buffer = new StringBuffer();
            buffer.append( Localizer.getString( "SharedFiles" ) );
            buffer.append( ": ( " );
            buffer.append( fileAdmin.getFileCount() );
            buffer.append( " / " );
            buffer.append( StrUtil.formatSizeBytes( ((long)fileAdmin.getTotalFileSizeInKb()) * 1024L ) ).append(")");

            sharedFilesLabel.setText( buffer.toString() );
        }
    }

    private class CellRenderer extends DefaultTableCellRenderer
    {
        public Component getTableCellRendererComponent(
            JTable table, Object value, boolean isSelected,
            boolean hasFocus, int row, int column)
        {
            super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
            this.setForeground(Color.black);
//			this.setBackground(Color.black);

            if (table == mUploadTable)
            {
                if ( row < uploadContainer.getUploadListSize() )
                {
                    row = mUploadSorter.indexes[row];
                    switch ( uploadContainer.getUploadFileAt( row ).getStatus() )
                    {
                        case UploadFile.sError:
                        case UploadFile.sAborted:
                            this.setForeground(Color.darkGray);
                            break;
                        case UploadFile.sUploading:
                            this.setForeground(Color.red);
                            break;
                        case UploadFile.sCompleted:
                            this.setForeground(Color.blue);
                            break;
                    }
                }
            }
            return this;
        }
    }
}