
/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *  Copyright (C) 2000 William W. Wong
 *  williamw@jps.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.host;


import java.io.*;
import java.net.*;

import phex.*;
import phex.config.*;
import phex.interfaces.*;
import phex.msg.*;

public class Host
{
    /**
     * The time after which a connection is stable.
     */
    private static final int STABLE_CONNECTION_TIME = 75 * 1000; // 75 seconds

    public static final int		sStatusNotConnected = 0;
    public static final int		sStatusError = 1;
    public static final int		sStatusConnecting = 2;
    public static final int		sStatusAccepting = 3;
    public static final int		sStatusConnected = 4;
    public static final int		sStatusTimeout = 6;

    public static final int		sTypeOutgoing = 1;
    public static final int		sTypeIncoming = 2;
    public static final int		sTypeDownload = 3;
    public static final int		sTypePush = 4;

    public static final int		sLatencyTimeout = 99999;

    private HostManager mManager;
    private SendManager			mSendManager;
    private HostAddress hostAddress;
    private Socket mSock;
    private InputStream mIs;
    private OutputStream mOs;
    private int						mStatus;
    private String					mLastStatusMsg = "";
    private long mStatusTime = 0;
    private int						mType;
    private boolean				mHasWorker;
    private boolean				mSending;
    private int						mReceivedCount;
    private int						mSentCount;
    private int						mDropCount;
    private FQueue mWaitingQueue;
    private MsgInit				mPingMsg = null;
    private long					mPingStartTime = 0;
    private long					mPingNextStartTime = -1;
    private int						mPingLatency = 0;
    private long					mLatencyTimeOutStartTime = 0;
    private long mConnectionStartTime = 0;
    //private DataChanger mStatusChanger;
    private int			         mFileCount = 0;
    private int			         mTotalSize = 0;

    /**
     * Marks if a connection is stable. A connection is stable when a
     * host connection last over STABLE_CONNECTION_TIME seconds.
     */
    private boolean isConnectionStable;

    public Host()
    {
        mManager = ServiceManager.getHostManager();
        mSendManager = ServiceManager.getSendManager();

        mSock = null;
        mIs = null;
        mOs = null;
        mStatus = sStatusNotConnected;
        mType = sTypeOutgoing;
        mHasWorker = false;
        mSending = false;
        mPingMsg = null;
        isConnectionStable = false;

        //mStatusChanger = new DataChanger();
    }

    public Host( HostAddress address )
    {
        this();
        hostAddress = address;
    }

    /*public void addStatusChangedListener(IDataChangedListener listener)
    {
        mStatusChanger.addListener(listener);
    }
*/

    public HostAddress getHostAddress()
    {
        return hostAddress;
    }

/*
    public void setHostAddr(String hostAddr)
    {
        mHostAddr = hostAddr;
    }
    */


    public Socket getSock()
    {
        return mSock;
    }


    public void setSock(Socket sock)
    {
        mSock = sock;
        mReceivedCount = 0;
        mSentCount = 0;
        mDropCount = 0;
    }

    public InputStream getIs()
    {
        return mIs;
    }


    public void setIs(InputStream Is)
    {
        this.mIs = Is;
    }


    public OutputStream getOs()
    {
        return mOs;
    }


    public void setOs(OutputStream os)
    {
        this.mOs = os;
    }


    public int getStatus()
    {
        return mStatus;
    }


    public String getStatusName()
    {
        switch (mStatus)
        {
            case sStatusNotConnected:
                return "(Not connected)";

            case sStatusError:
                return "Error.  " + mLastStatusMsg;

            case sStatusConnecting:
                return "Connecting...  " + mLastStatusMsg;

            case sStatusAccepting:
                return "Accepting...  " + mLastStatusMsg;

            case sStatusConnected:
                if (sendQueueInRed() || latencyInRed())
                    return "Connected (deprecated, no broadcast forwarded)";
                else
                    return "Connected";
            case sStatusTimeout:
                return "Timed out";
        }

        return "Unknown";
    }

    public void setStatus( int status )
    {
        setStatus( status, null, System.currentTimeMillis() );
    }

    public void setStatus( int status, long statusTime )
    {
        setStatus( status, null, statusTime );
    }

    public void setStatus( int status, String msg )
    {
        setStatus( status, msg, System.currentTimeMillis() );
    }

    public void setStatus( int status, String msg, long statusTime )
    {
        if ( mStatus == status )
        {
            return;
        }
        mStatus = status;
        mLastStatusMsg = msg;
        mStatusTime = statusTime;
        mManager.fireNetworkHostChanged( this );
    }

    /**
     * Checks if a connection status is stable. A stable connection
     * is a connection to host that last over STABLE_CONNECTION_TIME seconds.
     * The current time is given out of performance reasons when
     * looping over all hosts.
     */
    public void checkForStableConnection( long currentTime )
    {
        // if we have connected status for at least STABLE_CONNECTION_TIME.
        if ( mStatus == sStatusConnected
             && getConnectionUpTime( currentTime ) > STABLE_CONNECTION_TIME )
        {
            isConnectionStable = true;
            mManager.fireNetworkHostChanged( this );
        }
    }

    public boolean isConnectionStable()
    {
        return isConnectionStable;
    }

    public long getConnectionUpTime( long currentTime )
    {
        if( mStatus == sStatusConnected )
        {
            return currentTime - mStatusTime;
        }
        else
        {
            return 0;
        }
    }

    /**
     * Checks if the try to connect to the host timed out. There are different
     * timeouts for connecting to a host in a LAN and in the internet.
     * Since there is no way of defining a connection timeout this is
     * done through the a timer.
     */
    public void checkForConnectingTimeout( long currentTime )
    {
        if ( mStatus == sStatusConnecting && mType == sTypeOutgoing )
        {
            if (currentTime - mConnectionStartTime > ServiceManager.sCfg.mNetConnectionTimeout)
            {
                setStatus( Host.sStatusTimeout, currentTime );
            }
        }
    }

    public void markConnectionStartTime()
    {
        mConnectionStartTime = System.currentTimeMillis();
    }


    public boolean isErrorStatusExpired( long currentTime )
    {
        if (mStatus == sStatusError || mStatus == sStatusNotConnected || mStatus == sStatusTimeout)
        {
            if ( currentTime - mStatusTime > 5000 )
            {
                return true;
            }
        }
        return false;
    }


    public int getType()
    {
        return mType;
    }


    public String getTypeName()
    {
        switch(mType)
        {
            case sTypeOutgoing:
                return "Outgoing";
            case sTypeIncoming:
                return "Incoming";
            case sTypeDownload:
                return "Download";
            case sTypePush:
                return "Push";
            default:
                return "Unknown";
        }
    }


    public void setType(int Type)
    {
        this.mType = Type;
    }


    public void incReceivedCount()
    {
        mReceivedCount++;
    }


    public int getReceivedCount()
    {
        return mReceivedCount;
    }


    public void incSentCount()
    {
        mSentCount++;
    }


    public int getSentCount()
    {
        return mSentCount;
    }

    public void incDropCount()
    {
        mDropCount++;
    }

    public int getDropCount()
    {
        return mDropCount;
    }

    public int getFileCount()
    {
        return mFileCount;
    }

    /**
     * Returns total size in kBytes.
     */
    public int getTotalSize()
    {
        return mTotalSize;
    }

    public boolean tooManyDropPackets()
    {
        if (!ServiceManager.sCfg.mDisconnectApplyPolicy)
            return false;

        return (mDropCount * 100 / (mReceivedCount + 1) > ServiceManager.sCfg.mDisconnectDropRatio);
    }


    public boolean dropPacketsInRed()
    {
        return (mDropCount * 100 / (mReceivedCount + 1)) > (ServiceManager.sCfg.mDisconnectDropRatio * 3 / 4);
    }

    public void connectForReading()
    {
        if (getType() != Host.sTypeOutgoing)
        {
            return;
        }
        if ( mSock != null )
        {
            return;
        }
        new ReadWorker( this, 0);
    }

    public boolean isConnected( )
    {
        return mSock != null;
    }

    public void disconnect()
    {
        if (mSock != null)
        {
            if (mStatus != sStatusError)
            {
                setStatus( Host.sStatusNotConnected );
            }
        }

        if (mIs != null)
        {
            try
            {
                mIs.close();
            }
            catch (Exception e)
            {
//				e.printStackTrace();
            }
            mIs = null;
        }
        if (mOs != null)
        {
            try
            {
                mOs.close();
            }
            catch (Exception e)
            {
//				e.printStackTrace();
            }
            mOs = null;
        }
        if (mSock != null)
        {
            try
            {
                mSock.close();
            }
            catch (Exception e)
            {
//				e.printStackTrace();
            }
            mSock = null;
        }
    }


    public synchronized boolean acquireByWorker()
    {
        // Test and set, in one block of operation.
        if (mHasWorker)
        {
            return false;
        }
        mHasWorker = true;
        return true;
    }


    public synchronized void releaseFromWorker()
    {
        mHasWorker = false;
    }


    // Test and set and save in queue, in one block of operation.
    public synchronized boolean acquireSendLockOrQueueMsg(HostMsg msgEntry)
    {
        if (mSending)
        {
            // Another SendWorker is sending another msg to the host.
            // The remote host may be slow in reading the msg.  There
            // can be only one send at a time.  Queue up the msgEntry
            // in the Host's private queue so that it doesn't got
            // re-queued in the main mSendQueue and wasting other
            // SendWorkers' cycle.
            initWaitingQueue();
            if (msgEntry.getUrgent())
            {
                mWaitingQueue.addToHead(msgEntry);
            }
            else
            {
                mWaitingQueue.add(msgEntry);
            }
            return false;
        }
        mSending = true;
        return true;
    }


    public synchronized void releaseSendLock()
    {
        // Take one entry and re-queue it back to the main mSendQueue.
        // Notice that I don't re-queue all msgs from the private queue
        // to the main queue.  If the remote host was slow in the last
        // receive, chances are that it is still slow.  Don' want to
        // move everything from the private queue to the main queue and
        // have them moved back into the private queue in the next send.
        initWaitingQueue();
        HostMsg	entry = (HostMsg)mWaitingQueue.remove();
        if (entry != null)
        {
            mSendManager.queueMsgToSend(entry);
        }

        mSending = false;
    }

    public int getSendQueueLength()
    {
        if ( mWaitingQueue == null )
        {
            return 0;
        }
        else
        {
            return mWaitingQueue.getCount();
        }
    }


    public boolean sendQueueTooLong()
    {
        if (!ServiceManager.sCfg.mDisconnectApplyPolicy)
        {
            return false;
        }
        if ( mWaitingQueue == null )
        {
            return false;
        }
        return (mWaitingQueue.getCount() >= ServiceManager.sCfg.mNetMaxSendQueue - 1);
    }

    public boolean sendQueueInRed()
    {
        if ( mWaitingQueue == null )
        {
            return false;
        }
        return (mWaitingQueue.getCount() >= ServiceManager.sCfg.mNetMaxSendQueue * 3 / 4);
    }


    public void pingHost()
    {
        if (mStatus != sStatusConnected)
        {
            return;
        }

        long currTime = System.currentTimeMillis();

        if (mPingMsg != null)
        {
            // Pinging already.  Don't ping again.

            // Check timeout
            if (currTime - mPingStartTime > ServiceManager.sCfg.mPingTimeout)
            {
                // Timeout.
                mPingLatency = sLatencyTimeout;
                mPingMsg = null;
                mLatencyTimeOutStartTime = System.currentTimeMillis();
            }
            return;
        }

        if (currTime < mPingNextStartTime)
        {
            // Wait until the next marked time to start pinging.
            return;
        }

        mPingStartTime = currTime;
        mPingNextStartTime = currTime + ServiceManager.sCfg.mPingFrequency;

        // Send ping msg.
        mPingMsg = new MsgInit();
        mPingMsg.getHeader().setTTL(1);
        mSendManager.queueMsgToSend(this, mPingMsg, true);
    }


    public boolean checkPingResponse(MsgInitResponse responseMsg)
    {
        if (mPingMsg != null &&
            responseMsg.getHeader().getMsgID().equals(mPingMsg.getHeader().getMsgID()))
        {
            mPingMsg = null;
            mPingLatency = (int)(System.currentTimeMillis() - mPingStartTime);
            mFileCount = responseMsg.getFileCount();
            mTotalSize = responseMsg.getTotalSize();

            return true;
        }

        return false;
    }


    public int getLatency()
    {
        return mPingLatency;
    }


    public boolean tooMuchLatency( long currentTime )
    {
        if ( !ServiceManager.sCfg.mDisconnectApplyPolicy )
        {
            return false;
        }

        if (mPingLatency == sLatencyTimeout)
        {
            if (currentTime - mLatencyTimeOutStartTime > ServiceManager.sCfg.mDisconnectLatency)
            {
                return true;
            }
        }
        return false;
    }


    public boolean latencyInRed()
    {
        return (mPingLatency == sLatencyTimeout);
    }


    public void log(String str)
    {
        if (Debug.DbgHost()) Debug.msg("Host " + hostAddress.getHostName()
            + ":" + hostAddress.getPort() + ": " + str);
    }

    public String toString()
    {
        return hostAddress.getHostName() + ":" + hostAddress.getPort();
    }

    /**
     * This method is here to make sure the waiting queue is only generated
     * when it is actually necessary. Generating it in the constructor
     * uses up a big amount of memory for every known Host.
     */
    private void initWaitingQueue()
    {
        if ( mWaitingQueue != null )
        {
            return;
        }
        // Create a queue with max-size, dropping the oldest msg when max reached.
        mWaitingQueue = new FQueue(ServiceManager.sCfg.mNetMaxSendQueue);
    }
}