package phex.host;

import java.net.*;
import phex.ServiceManager;
import phex.utils.IPUtils;

/**
 * Represents a host address.
 */
public class HostAddress
{
    public static String LOCAL_HOST_NAME = "127.0.0.1";
    public static final int DEFAULT_PORT = 6346;
    private String hostName;
    private int port;
    private byte[] hostIP;

    public HostAddress( String aHostName, int aPort )
    {
        hostName = aHostName;
        port = aPort;
    }

    public HostAddress( byte[] aHostIP, int aPort )
    {
        hostIP = aHostIP;
        port = aPort;
        hostName = IPUtils.ip2string( aHostIP );
    }

    public HostAddress( String hostString )
        throws MalformedHostAddressException
    {
        int idx = hostString.indexOf( ':' );
        // no port given
        if ( idx < 0 || idx == hostString.length() - 1 )
        {
            hostName = hostString;
            port = DEFAULT_PORT;
            return;
        }
        if ( idx == 0 )
        {
            throw new MalformedHostAddressException( "No host name: "
                + hostString );
        }
        hostName = hostString.substring( 0, idx );
        String portString = hostString.substring( idx + 1 );
        try
        {
            port = Integer.parseInt( portString );
        }
        catch ( NumberFormatException exp )
        {
            throw new MalformedHostAddressException( "Can't parse port: "
                + portString );
        }
    }

    public void updateAddress( String aHostName, int aPort )
    {
        if ( !aHostName.equals( hostName ) )
        {
            hostName = aHostName;
            hostIP = null;
        }
        port = aPort;
    }

    public String getHostName()
    {
        return hostName;
    }

    public int getPort()
    {
        return port;
    }

    /**
     * The method returns the IP of the host.
     */
    public byte[] getHostIP() throws UnknownHostException
    {
        initHostIP();
        return hostIP;
    }

    public boolean equals( HostAddress address )
    {
        return hostName.equals( address.hostName )
               && port == address.port ;
    }

    /**
     * Checks if the host address is the local one
     */
    public boolean isLocalHost( )
    {
        HostAddress localAddress = ServiceManager.getListener().getLocalAddress();
        boolean isPortEqual = port == localAddress.getPort();

        try
        {
            initHostIP();
        }
        catch ( UnknownHostException exp )
        {
            return false;
        }
        if ( hostIP[0] == (byte) 127 )
        {
            return isPortEqual;
        }
        else
        {
            return localAddress.equals( this );
        }
    }

    private void initHostIP()
        throws UnknownHostException
    {
        if( hostIP == null )
        {
            // does the parsing of an IP or determinse the IP over DNS
            // TODO BUG if socks5 proxy is used the ip can't be determined right!
            hostIP = InetAddress.getByName( hostName ).getAddress();
        }
    }
}