/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package phex.host;

import java.util.ArrayList;
import java.util.Iterator;
import javax.swing.*;
import javax.swing.event.EventListenerList;
import phex.ServiceManager;
import phex.event.NetworkHostsChangeListener;

/**
 * Responsible for holding all hosts of the current network neighbour hood
 */
final public class NetworkHostsContainer
{
    /**
     * The complete neighbour hood. Containing connected and not connected hosts.
     */
    private ArrayList networkHosts;

    /**
     * Only the connected neighbours.
     */
    private ArrayList connectedHosts;

    /**
     * All listeners interested in events.
     */
    private ArrayList listenerList = new ArrayList( 3 );

    public NetworkHostsContainer()
    {
        networkHosts = new ArrayList();
        connectedHosts = new ArrayList();
    }

    public synchronized int getConnectedHostCount()
    {
        return connectedHosts.size();
    }

    /**
     * Checks if we have incomming slots available.
     */
    public synchronized boolean hasIncommingSlotsAvailable()
    {
        // the same rules apply for outgoing connections... the at least
        // 25% outgoing connection rule is removed.
        // TODO have a extra configuration for max incomming connections or min
        // outgoing connections.
        // also we don't response on pings when the slots are full therefore
        // we should not get that many incommings
        return getConnectedHostCount() + 1 <= ServiceManager.sCfg.mNetMaxConnection;
    }

    public synchronized Host getConnectedHostAt( int index )
    {
        if ( index >= connectedHosts.size() )
        {
            return null;
        }
        return (Host) connectedHosts.get( index );
    }

    /**
     * Checks if a connected host is able to keep up...
     * if not it will be removed...
     */
    public synchronized void cleanupConnectedHosts()
    {
        long currentTime = System.currentTimeMillis();
        int size = connectedHosts.size();
        for (int i = 0; i < size; i++)
        {
            Host host = (Host) connectedHosts.get( i );
            if ( host.getStatus() != Host.sStatusError &&
                   (  host.tooManyDropPackets()
                   || host.tooMuchLatency( currentTime )
                   || host.sendQueueTooLong() ) )
            {
                host.setStatus( Host.sStatusError, "Apply disconnect policy.",
                    currentTime );
            }
        }
    }

    public synchronized void addConnectedHost( Host host )
    {
        connectedHosts.add( host );
        ServiceManager.getHostManager().addCaughtHost(
            host.getHostAddress(), CaughtHostsContainer.LOW_PRIORITY );

        // make sure host is in network! this is only for testing!!!
        /*if ( !networkHosts.contains( host ) )
        {
            throw new RuntimeException( "unsync" );
        }*/
    }

    /**
     * Checks for hosts that have a connection timeout...
     */
    public synchronized void periodicallyCheckHosts()
    {
        Host host;
        long currentTime = System.currentTimeMillis();
        Iterator iterator = networkHosts.iterator();
        while ( iterator.hasNext() )
        {
            host = (Host) iterator.next();
            host.checkForConnectingTimeout( currentTime );
        }

        iterator = connectedHosts.iterator();
        while( iterator.hasNext() )
        {
            host = (Host) iterator.next();
            host.checkForStableConnection( currentTime );
        }
    }

    public synchronized Host getNetworkHostAt( int index )
    {
        if ( index >= networkHosts.size() )
        {
            return null;
        }
        return (Host) networkHosts.get( index );
    }

    public synchronized int getNetworkHostCount()
    {
        return networkHosts.size();
    }

    public synchronized void addNetworkHost( Host host )
    {
        int position = networkHosts.size();
        networkHosts.add( position, host );
        fireNetworkHostAdded( position );
    }

    public synchronized void createOutgoingConnectionToHost( String hostAddr )
    {
        HostAddress address = null;
        try
        {
            address = new HostAddress( hostAddr );
        }
        catch ( MalformedHostAddressException exp )
        {
            return;
        }

        // Check for duplicate.
        for (int i = 0; i < networkHosts.size(); i++)
        {
            Host host = (Host)networkHosts.get( i );

            if (host.getType() == Host.sTypeOutgoing &&
                host.getHostAddress().equals( address ) )
            {
                return;
            }
        }

        Host host = new Host( address );
        host.setType( Host.sTypeOutgoing );
        addNetworkHost( host );
        host.connectForReading();
    }

    public synchronized void addIncomingHost( Host host )
    {
        // a incoming host is new for the network and is connected
        addNetworkHost( host );
        connectedHosts.add( host );
    }

    public synchronized void cleanupNetworkHosts()
    {
        if ( !ServiceManager.sCfg.mAutoCleanup )
        {
            return;
        }

        int	size = networkHosts.size();
        // first collect...
        Host[] badHosts = new Host[ size ];
        long currentTime = System.currentTimeMillis();
        int pos = 0;
        int status;
        for (int i = 0; i < size; i++)
        {
            Host host = (Host)networkHosts.get( i );
            status = host.getStatus();
            if ( status != Host.sStatusConnected &&
                 status != Host.sStatusConnecting &&
                 status != Host.sStatusAccepting )
            {
                if ( host.isErrorStatusExpired( currentTime ) )
                {
                    badHosts[ pos ] = host;
                    pos ++;
                }
            }
        }
        // then kill...
        if ( pos > 0 )
        {
            removeNetworkHosts( badHosts );
        }
    }

    public synchronized void removeAllNetworkHosts()
    {
        Host host;
        Iterator iterator = networkHosts.iterator();
        while ( networkHosts.size() > 0 )
        {
            host = (Host) networkHosts.get( 0 );
            internalRemoveNetworkHost( host );
        }
    }

    public synchronized void removeNetworkHosts( Host[] hosts )
    {
        Host host;
        int length = hosts.length;
        for ( int i = 0; i < length; i++ )
        {
            host = hosts[ i ];
            internalRemoveNetworkHost( host );
        }
    }

    /**
     * This is the only way a Host gets diconnected right!
     * The Host disconnect method is only used to clean up the connection.
     */
    public synchronized void removeNetworkHost( Host host )
    {
        internalRemoveNetworkHost( host );
    }

    /**
     * Disconnects from host without firing a event.
     */
    private void internalRemoveNetworkHost( Host host )
    {
        if ( host == null )
        {
            return;
        }
        if ( host.isConnected() )
        {
            connectedHosts.remove( host );
        }
        int position = networkHosts.indexOf( host );
        if ( position >= 0 )
        {
            networkHosts.remove( position );
            host.disconnect();
            fireNetworkHostRemoved( position );
        }

        // This is only for testing!!!
        /*if ( connectedHosts.contains( host ) )
        {
            // go crazy
            throw new RuntimeException ("notrem");
        }*/
    }

    ///////////////////// START event handling methods ////////////////////////
    public void addNetworkHostsChangeListener( NetworkHostsChangeListener listener )
    {
        listenerList.add( listener );
    }

    public void removeNetworkHostsChangeListener( NetworkHostsChangeListener listener )
    {
        listenerList.remove( listener );
    }


    private void fireNetworkHostChanged( final int position )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                NetworkHostsChangeListener listener;
                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (NetworkHostsChangeListener)listeners[ i ];
                    listener.networkHostChanged( position );
                }
            }
        });
    }

    private void fireNetworkHostAdded( final int position )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                NetworkHostsChangeListener listener;
                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (NetworkHostsChangeListener)listeners[ i ];
                    listener.networkHostAdded( position );
                }
            }
        });
    }

    private void fireNetworkHostRemoved( final int position )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                NetworkHostsChangeListener listener;
                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (NetworkHostsChangeListener)listeners[ i ];
                    listener.networkHostRemoved( position );
                }
            }
        });
    }

    public void fireNetworkHostChanged( Host host )
    {
        int position = networkHosts.indexOf( host );
        if ( position >= 0 )
        {
            fireNetworkHostChanged( position );
        }
    }
    ///////////////////// END event handling methods ////////////////////////
}