
/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2000 William W. Wong
 *  williamw@jps.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package phex.msg;


import java.util.*;

import phex.*;
import phex.config.*;
import phex.cqueue.*;
import phex.download.*;
import phex.host.*;
import phex.interfaces.*;
import phex.msg.*;
import phex.query.QueryManager;
import phex.query.Search;
import phex.utils.IPUtils;

final public class MsgManager
{
    private HostManager mHostMgr = ServiceManager.getHostManager();
    private SendManager sendMgr = ServiceManager.getSendManager();
    private QueryManager queryMgr = ServiceManager.getQueryManager();
    private Hashtable			mMsgSeen;
    private Hashtable			mRoutingTable;
    private Hashtable			mPushRoutingTable;
    private CQueue			mMsgLRU;
    private CQueue			mRoutingLRU;
    private CQueue			mPushRoutingLRU;
    private MsgInit			mMyMsgInit;
    private Vector			mPassiveResults = new Vector();
    private String			mPassiveSearching = null;
    private Vector			mPassiveSearchingList = new Vector();
    private IMonitorSearch mMonitorSearchListener;


    public MsgManager()
    {
        mMsgSeen = new Hashtable();
        mRoutingTable = new Hashtable();
        mPushRoutingTable = new Hashtable();
        mMsgLRU = new CQueue(10000, false);		// hold upto 10000 msg
        mRoutingLRU = new CQueue(10000, false);	// hold upto 10000 routing guid
        mPushRoutingLRU = new CQueue(1000, false);	// hold upto 1000 routing guid

        mMyMsgInit = new MsgInit();
        mMyMsgInit.getHeader().setTTL(ServiceManager.sCfg.mNetTTL);

        // Add the InitMsg to the seen list.
        checkAndAddMsgSeen(mMyMsgInit);
    }


    public void setMonitorSearchListener(IMonitorSearch listener)
    {
        mMonitorSearchListener = listener;
    }

    // Test and set in one atomic operation because there are multiple
    // threads handling multiple connection at the same time.
    // If the test and set operations are done separately, there is
    // a chance that both threads would check and got the wrong result
    // before they can set.
    public synchronized boolean checkAndAddMsgSeen(IMsg msg)
    {
        GUID msgID = msg.getHeader().getMsgID();

        if (mMsgSeen.get(msgID.getHashCode()) != null)
        {
            // Seen this msg.
            return true;
        }
        else
        {
            // Hasn't seen this msg.  Save it.
            mMsgSeen.put(msgID.getHashCode(), msg.getHeader());
            try
            {
                // Push into a FIFO LRU queue to keep track of the age of the object.
                mMsgLRU.addToHead( new CQueueElement( msgID ) );
                if (mMsgLRU.isExceedingMax())
                {
                    // Too many guids.  Get rid of the oldest one.
                    CQueueElement oldestObj = (CQueueElement)mMsgLRU.removeFromTail();
                    GUID guid = (GUID)oldestObj.getObj();
                    mMsgSeen.remove(guid.getHashCode());
                }
            }
            catch (Exception e)
            {
                e.printStackTrace();
            }
            return false;
        }
    }


    public synchronized void addToRoutingTable(GUID msgID, Host sender)
    {
        try
        {
            // Add to routing table.
            mRoutingTable.put(msgID.getHashCode(), sender);

            // Push into a FIFO LRU queue to keep track of the age of the object.
            mRoutingLRU.addToHead(new CQueueElement(msgID));
            if (mRoutingLRU.isExceedingMax())
            {
                // Too many guids.  Get rid of the oldest one.
                CQueueElement	oldestObj = (CQueueElement)mRoutingLRU.removeFromTail();
                GUID			guid = (GUID)oldestObj.getObj();
                mRoutingTable.remove(guid.getHashCode());
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }

    }

    public synchronized void addToPushRoutingTable(GUID clientID, Host sender)
    {
        try
        {
            // Add the clientID in the search result of the sharing host
            // to routing table.
            mPushRoutingTable.put(clientID.getHashCode(), sender);

            // Push into a FIFO LRU queue to keep track of the age of the object.
            mPushRoutingLRU.addToHead(new CQueueElement(clientID));
            if (mPushRoutingLRU.isExceedingMax())
            {
                // Too many guids.  Get rid of the oldest one.
                CQueueElement	oldestObj = (CQueueElement)mPushRoutingLRU.removeFromTail();
                GUID			guid = (GUID)oldestObj.getObj();
                mPushRoutingTable.remove(guid.getHashCode());
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }

    }


    public synchronized Host getRouting(GUID msgID)
    {
        return (Host)mRoutingTable.get(msgID.getHashCode());
    }


    public synchronized Host getPushRouting(GUID clientID)
    {
        return (Host)mPushRoutingTable.get(clientID.getHashCode());
    }


    public MsgInit getMyMsgInit()
    {
        return mMyMsgInit;
    }


    public void resetMyMsgInit()
    {
        mMyMsgInit = new MsgInit();
    }


    public synchronized void processQueryResponse(Host remoteHost, MsgQueryResponse msg)
    {
        queryMgr.getSearchContainer().processQueryResponse( remoteHost, msg );
        queryMgr.getBackgroundSearchContainer().processQueryResponse( remoteHost, msg );

        if ( !IPUtils.isHostInUserFilter( msg.getRemoteHost().getHostAddress() ) )
        {
            passiveMonitorResult(msg);
        }

    }

    public String getPassiveSearching()
    {
        return mPassiveSearching;
    }


    public void setPassiveSearching(String str)
    {
        mPassiveSearching = str;
        mPassiveSearchingList.removeAllElements();

        if (str != null)
        {
            StringTokenizer	tokens = new StringTokenizer(str);
            while (tokens.hasMoreElements())
            {
                mPassiveSearchingList.addElement(tokens.nextElement());
            }
        }
    }


    public Vector getPassiveResults()
    {
        return mPassiveResults;
    }


    public void passiveMonitorResult(MsgQueryResponse msg)
    {
        if (mPassiveSearching == null)
            return;

        // resolve: have its own property name.
        if (mPassiveResults.size() >= ServiceManager.sCfg.mSearchMaxSearch)
        {
            // resolve: reset status
            return;
        }

        String	hostStr = msg.getRemoteHostStr();
        int		speed = msg.getRemoteHostSpeed();

        for (int i = 0; i < msg.getRecordCount(); i++)
        {
            MsgResRecord	rec = msg.getMsgRecord(i);

            // Capture everything.
            if (mPassiveSearching.length() == 0)
            {
                RemoteFile	rfile = new RemoteFile(
                    msg.getRemoteClientID(),
                    rec.getFileIndex(),
                    rec.getFilename(),
                    rec.getFileSize(),
                    hostStr,
                    speed);

                if (!passiveResultExists(rfile))
                {
                    mPassiveResults.addElement(rfile);
                    mMonitorSearchListener.passiveResultArrived();
                    continue;
                }
            }

            // Capture using only the filter match keyword.
            for (int j = 0; j < mPassiveSearchingList.size(); j++)
            {
                String filter = (String)mPassiveSearchingList.elementAt(j);
                if (rec.getFilename().indexOf(filter) != -1)
                {
                    RemoteFile	rfile = new RemoteFile(
                        msg.getRemoteClientID(),
                        rec.getFileIndex(),
                        rec.getFilename(),
                        rec.getFileSize(),
                        hostStr,
                        speed);

                    if (!passiveResultExists(rfile))
                    {
                        mPassiveResults.addElement(rfile);
                        mMonitorSearchListener.passiveResultArrived();
                    }
                    break;
                }
            }
        }

    }


    // resolve: this is expensive.
    private boolean passiveResultExists(RemoteFile rfile)
    {
        for (int i = 0; i < mPassiveResults.size(); i++)
        {
            RemoteFile	pfile = (RemoteFile)mPassiveResults.elementAt(i);
            if (pfile.equals(rfile))
                return true;
        }
        return false;
    }


    public void resetPassiveResults()
    {
        mPassiveResults.removeAllElements();
        mMonitorSearchListener.passiveResultArrived();
    }

    /**
     * Forward broadcast messages to neighbors except the fromHost
     */
    public void forwardMsg( IMsg msg, Host fromHost )
    {
        if (decTTL(msg))
        {
            // fromHost.log("TTL expired " + msg);
            return;
        }

        Host remoteHost;
        final int numberOfConnectedHosts = mHostMgr.getConnectedHostCount();

        for (int i = 0; i < numberOfConnectedHosts; i++)
        {
            remoteHost = mHostMgr.getConnectedHostAt( i );
            if ( remoteHost == fromHost )
            {
                continue;
            }
            sendMsgToHost( msg, remoteHost );
        }
    }

    /**
     * Sends a message to all hosts
     */
    public void sendMsgToHosts( IMsg msg )
    {
        Host remoteHost;
        final int numberOfConnectedHosts = mHostMgr.getConnectedHostCount();
        
        for (int i = 0; i < numberOfConnectedHosts; i++)
        {
            remoteHost = mHostMgr.getConnectedHostAt( i );
            sendMsgToHost( msg, remoteHost );
        }
    }

    public void pingNeighbors()
    {
        Host remoteHost;
        final int numberOfConnectedHosts = mHostMgr.getConnectedHostCount();
        
        for (int i = 0; i < numberOfConnectedHosts; i++)
        {
            remoteHost = mHostMgr.getConnectedHostAt( i );
            if ( remoteHost != null )
            {
                remoteHost.pingHost();
            }
        }
    }


    /**
     * Sends a message to a hosts
     */
    public void sendMsgToHost( IMsg msg, Host host )
    {
        if ( host == null)
        {
            return;
        }
        if ( host.sendQueueInRed() || host.latencyInRed() )
        {
            // Neighbor is too slow.  Deprecate the neighbor.
            // Don't send broadcast message to neighbor.
            return;
        }
        
        // Note that the msg is shared among all sending hosts.
        sendMgr.queueMsgToSend( host, msg, false);
    }

    public boolean decTTL(IMsg msg)
    {
        MsgHeader header = msg.getHeader();
        int ttl = header.getTTL() - 1;

        if (ttl <= 0)
        {
            // Expired
            return true;
        }

        if (ttl > ServiceManager.sCfg.mNetMaxTTL)
        {
            ttl = ServiceManager.sCfg.mNetMaxTTL;
        }

        header.setTTL(ttl);
        header.setHopsTaken(header.getHopsTaken() + 1);

        return false;
    }
}