
/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2000 William W. Wong
 *  williamw@jps.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.msg;

import java.net.*;
import java.util.*;

import phex.*;
import phex.interfaces.*;
import phex.utils.*;

public class MsgQueryResponse implements IMsg
{
    private static final boolean INCLUDE_QHD = true;
    /**
     * The unknown flag status.
     */
    private static final short UNKNOWN_FLAG = -1;

    /**
     * The true flag status.
     */
    private static final short TRUE_FLAG = 1;

    /**
     * The false flag status.
     */
    private static final short FALSE_FLAG = 0;

    /**
     * push flag mask for beashare metadata.
     */
    private static final byte PUSH_NEEDED_MASK=(byte)0x01;
    /**
     * busy flag mask for beashare metadata.
     */
    private static final byte SERVER_BUSY_MASK=(byte)0x04;
    /**
     * upload flag mask for beashare metadata.
     */
    private static final byte HAS_UPLOADED_MASK=(byte)0x08;
    /**
     * speed flag mask for beashare metadata.
     */
    private static final byte UPLOAD_SPEED_MASK=(byte)0x10;


    private MsgHeader mHeader;
    /**
     * The body of the query response.
     */
    private byte[] body;
    private short mRemoteListeningPort;
    private InetAddress mRemoteHost;
    private int mRemoteHostSpeed;
    private MsgResRecord[] mRecords;

    private GUID remoteClientID;

    /**
     * Defines the vendor code of the client that offers the file.
     */
    private String vendorCode;

    /**
     * Defines if a push transfer is needed or not or unknown.
     */
    private short pushNeededFlag;

    /**
     * Defines if a server is busy currently or unknown.
     */
    private short serverBusyFlag;

    /**
     * Defines if a the server has already uploaded a file.
     */
    private short hasUploadedFlag;

    /**
     * Defines if the upload speed of a server.
     */
    private short uploadSpeedFlag;

    /**
     * Defines if the body of the query response is already parsed.
     */
    private boolean isParsed;

    private MsgQueryResponse()
    { // Not used
        throw new UnsupportedOperationException();
    }

    /*private MsgQueryResponse( MsgHeader header )
    {
        setHeader( header );
        mRemoteListeningPort = 0;
        mRemoteHost = null;
        mRemoteHostSpeed = 0;
        mRecords = new ArrayList( 1 );
        pushNeededFlag = UNKNOWN_FLAG;
        serverBusyFlag = UNKNOWN_FLAG;
        hasUploadedFlag = UNKNOWN_FLAG;
    }*/

    public MsgQueryResponse( MsgHeader header, GUID clientID,
        InetAddress hostAddress, short hostPort, int speed, MsgResRecord records[] )
    {
        setHeader( header );
        remoteClientID = clientID;
        mRemoteHost = hostAddress;
        mRemoteListeningPort = hostPort;
        mRemoteHostSpeed = speed;
        mRecords = records;

        // when we are behind firewall or have not accepted an incoming connection
        // like defined in the protocol
        boolean isPushNeeded = false;
        if ( !ServiceManager.getListener().hasConnectedIncoming() ||
            ServiceManager.sCfg.isBehindFirewall )
        {
            isPushNeeded = true;
        }

        boolean isServerBusy = ServiceManager.getShareManager().isHostBusy();

        buildBody( isPushNeeded, isServerBusy );
        header.setDataLen( body.length );
        isParsed = true;
    }

    private void buildBody( boolean isPushNeeded, boolean isServerBusy )
    {
        int size = calculateBodySize();
        body = new byte[ size ];

        int recordCount = mRecords.length;
        body[ 0 ] = (byte) recordCount;
        IOUtil.serializeShortLE( mRemoteListeningPort, body, 1 );

        byte[] ipAddress = mRemoteHost.getAddress();
        body[ 3 ] = ipAddress[ 0 ];
        body[ 4 ] = ipAddress[ 1 ];
        body[ 5 ] = ipAddress[ 2 ];
        body[ 6 ] = ipAddress[ 3 ];
        IOUtil.serializeIntLE( mRemoteHostSpeed, body, 7 );

        int offset = 11;

        for (int i = 0; i < recordCount; i++)
        {
            offset = mRecords[ i ].serialize(body, offset);
        }

        if ( INCLUDE_QHD )
        {
            // add vendor code 'PHEX'
            body[ offset++ ] = (byte) 0x50;
            body[ offset++ ] = (byte) 0x48;
            body[ offset++ ] = (byte) 0x45;
            body[ offset++ ] = (byte) 0x58;
            // open data length
            body[ offset ++ ] = (byte) 2;
            // open data flags
            byte isPushNeededByte = (byte) 0;
            if ( isPushNeeded )
            {
                isPushNeededByte = PUSH_NEEDED_MASK;
            }
            byte isServerBusyByte = (byte) 0;
            if ( isServerBusy )
            {
                isServerBusyByte = SERVER_BUSY_MASK;
            }
            body[ offset++ ] = (byte) (
                  isPushNeededByte
                | SERVER_BUSY_MASK
                | 0 //HAS_UPLOADED_MASK we dont know that yet
                // we know we never measured that speed
                | UPLOAD_SPEED_MASK );
            body[ offset++ ] = (byte) (
                  PUSH_NEEDED_MASK
                | isServerBusyByte
                | 0 //(hasUploadedSuccessfully ? HAS_UPLOADED_MASK : 0)
                // we know we never measured that speed
                | 0 );//(isSpeedMeasured ? UPLOAD_SPEED_MASK : 0));
        }
        offset = remoteClientID.serialize( body, offset );
    }

    /**
     * Create a query response with its body. The body is not paresed directly
     * cause some query are just forwarded without the need of beeing completly
     * parsed.
     */
    public MsgQueryResponse( MsgHeader header, byte[] aBody )
    {
        setHeader( header );
        body = aBody;
        header.setDataLen( body.length );
        isParsed = false;
    }

    private void setHeader(MsgHeader header)
    {
        mHeader = header;
        mHeader.setFunction(MsgHeader.sQueryResponse);
    }

    public MsgHeader getHeader()
    {
        return mHeader;
    }

    public int getRecordCount()
    {
        parseBody();
        return mRecords.length;
    }

    public InetAddress getRemoteHost()
    {
        parseBody();
        return mRemoteHost;
    }

    public String getRemoteHostStr()
    {
        return getRemoteHost().getHostAddress() + ":" + mRemoteListeningPort;
    }

    public int getRemoteHostSpeed()
    {
        parseBody();
        return mRemoteHostSpeed;
    }

    public GUID getRemoteClientID()
    {
        if ( remoteClientID == null )
        {
            parseRemoteClientID();
        }
        return remoteClientID;
    }

    public MsgResRecord getMsgRecord( int i )
    {
        parseBody();
        return mRecords[ i ];
    }

    public void computeHeaderLen()
    {
        // nothing to do here already done in constructor
        //mHeader.setDataLen(getSize() - mHeader.getSize());
    }

    public int getSize()
    {
        return mHeader.getSize() + mHeader.getDataLen();
    }

    /**
     * 11 byte query response header size plus file record size
     * plus optional 7 byte QHD plus 16-byte client id
     */
    private int calculateBodySize()
    {
        // header ( # hits, port, ip, speed )
        int	size = 11;
        // record size
        size += getRecordLength();
        // optional 7 byte QHD
        if ( INCLUDE_QHD )
        {
            // vendor, open data length, open data
            size = size + 4 + 1 + 2;
        }

        // 16 byte client id
        size += remoteClientID.getSize();
        return size;
    }

    private int getRecordLength()
    {
        int size = 0;
        int recordCount = mRecords.length;
        for (int i = 0; i < recordCount; i++)
        {
            MsgResRecord rec = mRecords[ i ];
            size += rec.getSize();
        }
        return size;
    }

    public void copy(MsgQueryResponse b)
    {
        mHeader.copy(b.getHeader());
        mRemoteListeningPort = mRemoteListeningPort;
        mRemoteHost = mRemoteHost;
        mRemoteHostSpeed = mRemoteHostSpeed;
        remoteClientID = remoteClientID;
        int recordCount = b.mRecords.length;
        mRecords = new MsgResRecord[ recordCount ];
        for (int i = 0; i < recordCount; i++)
        {
            MsgResRecord rec = new MsgResRecord();
            rec.copy( b.mRecords[ i ] );
            mRecords[ i ] = rec;
        }
        body = b.body;
    }


    public int serialize(byte[] outbuf, int offset)
            throws Exception
    {
        offset = mHeader.serialize(outbuf, offset);
        try
        {
            System.arraycopy( body, 0, outbuf, offset, mHeader.getDataLen() );
        }
        catch ( ArrayIndexOutOfBoundsException exp )
        {
            System.out.println( body.length + "  " + offset +
                "  " + mHeader.getDataLen() + "  " + calculateBodySize() + "\n" + new String(body) + "\n" +
                new String(outbuf) + "\n----");
            exp.printStackTrace();
        }
        /*System.out.println("1 " + new String(outbuf) + "  " + ( offset + mHeader.getDataLen() ) );

        int recordCount = mRecords.length;

        outbuf[offset++] = (byte) recordCount;
        offset = IOUtil.serializeShortLE(mRemoteListeningPort, outbuf, offset);
        System.arraycopy(mRemoteHost.getAddress(), 0, outbuf, offset, 4);
        offset += 4;
        offset = IOUtil.serializeIntLE(mRemoteHostSpeed, outbuf, offset);

        for (int i = 0; i < recordCount; i++)
        {
            offset = mRecords[ i ].serialize(outbuf, offset);
        }

        offset = remoteClientID.serialize(outbuf, offset);

        System.out.println("2 " + new String(outbuf) + "  " + offset);
        System.out.println("---------");*/

        return offset;
    }


    public int deserialize(byte[] inbuf, int offset)
            throws Exception
    {
        return offset;
    }

    private void parseBody()
    {
        if ( isParsed )
        {
            return;
        }
        // Already read the header.
        int offset = 0;
        byte n = body[offset++];
        int recordCount = (n < 0 ? 256 + n : n);
        mRemoteListeningPort = IOUtil.deserializeShortLE(body, offset);
        offset += 2;
        StringBuffer strBuf = new StringBuffer();
        offset = IOUtil.deserializeIP(body, offset, strBuf);
        try
        {
            mRemoteHost = InetAddress.getByName( strBuf.toString() );
            mRemoteHostSpeed = IOUtil.deserializeIntLE(body, offset);
            offset += 4;

            mRecords = new MsgResRecord[ recordCount ];
            for (int i = 0; i < recordCount; i++)
            {
                MsgResRecord rec = new MsgResRecord();
                offset = rec.deserialize(body, offset);
                mRecords[ i ] = rec;
            }

            // Handle Bearshare meta informations. The format is documented in
            // the GnutellaProtocol04.pdf document

            if ( offset <= (mHeader.getDataLen() - 16 - 4 - 2) )
            {
                // parse meta data
                // Use ISO encoding for two bytes characters on some platforms.
                vendorCode = new String( body, offset, 4, "ISO-8859-1");
                offset += 4;

                int openDataLength = IOUtil.unsignedByte2int( body[ offset ] );
                offset += 1;

                // parse upload speed, have uploaded, busy and push
                if ( openDataLength > 1)
                {   // if we have a flag byte
                    byte flag1 = body[ offset ];
                    byte flag2 = body[ offset + 1];

                    // check if push flag is meaningfull do it reversed from other checks
                    if ( ( flag2 & PUSH_NEEDED_MASK ) != 0 )
                    {
                        if ( ( flag1 & PUSH_NEEDED_MASK ) != 0 )
                        {
                            pushNeededFlag = TRUE_FLAG;
                        }
                        else
                        {
                            pushNeededFlag = FALSE_FLAG;
                        }
                    }

                    // check if server busy flag meaningfull
                    if ((flag1 & SERVER_BUSY_MASK) != 0)
                    {
                        if ( (flag2 & SERVER_BUSY_MASK) != 0 )
                        {
                            serverBusyFlag = TRUE_FLAG;
                        }
                        else
                        {
                            serverBusyFlag = FALSE_FLAG;
                        }
                    }

                    // check if has uploaded flag meaningfull
                    if ((flag1 & HAS_UPLOADED_MASK) != 0)
                    {
                        if ( (flag2 & HAS_UPLOADED_MASK) != 0 )
                        {
                            hasUploadedFlag = TRUE_FLAG;
                        }
                        else
                        {
                            hasUploadedFlag = FALSE_FLAG;
                        }
                    }
                    if ((flag1 & UPLOAD_SPEED_MASK) !=0 )
                    {
                        if ( (flag2 & UPLOAD_SPEED_MASK) !=0 )
                        {
                            uploadSpeedFlag = TRUE_FLAG;
                        }
                        else
                        {
                            uploadSpeedFlag = FALSE_FLAG;
                        }
                    }
                }
                offset += openDataLength;
                //System.out.println( (mHeader.getDataLen() -16 - 4 -2) + "  " + offset + "  " +
                //    /*new String( body, offset, mHeader.getDataLen() - offset) + "   " +*/
                //    openDataLength + "  " + pushNeededFlag + "  " + serverBusyFlag + "  " +
                //    uploadSpeedFlag + "  " + hasUploadedFlag + "  " + vendorCode );
            }
            parseRemoteClientID();
            isParsed = true;
        }
        catch ( UnknownHostException exp )
        {
            return;
        }
        catch ( java.io.UnsupportedEncodingException exp )
        {
            return;
        }
    }

    private void parseRemoteClientID()
    {
        if ( remoteClientID == null )
        {
            remoteClientID = new GUID();
        }
        remoteClientID.deserialize(body, mHeader.getDataLen() - GUID.sDataLength);
    }


    public String toString()
    {
        String	str =
                "[" +
                mHeader + " " +
                "NumRecords=" + mRecords.length + ", " +
                "RemoteListeningPort=" + mRemoteListeningPort + ", " +
                "RemoteHost=" + mRemoteHost.getHostAddress() + ", " +
                "RemoteHostSpeed=" + mRemoteHostSpeed + ", " +
                "RemoteClientID=" + remoteClientID + ", " +
                "Records=";

        for (int i = 0; i < mRecords.length; i++)
        {
            str += mRecords[ i ] + " ";
        }
        str += "]";

        return str;
    }
}
