
/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *  Copyright (C) 2000 William W. Wong ( williamw@jps.net )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

package phex.query;

import java.util.*;
import javax.swing.*;
import phex.GUID;
import phex.ServiceManager;
import phex.config.*;
import phex.download.*;
import phex.host.*;
import phex.interfaces.*;
import phex.msg.*;
import phex.utils.SearchEngine;
import phex.utils.IPUtils;
import phex.event.NetworkHostsChangeListener;
import phex.event.SearchChangeListener;
import phex.event.SearchChangeEvent;

public class Search
{
    public static final long DEFAULT_SEARCH_TIMEOUT = 5 * 60 * 1000; // 5 minutes

    private HostManager mHostMgr = ServiceManager.getHostManager();
    private MsgManager mMsgMgr = ServiceManager.getMsgManager();
    private SearchContainer searchContainer;

    private int mMinSpeed;
    private String mQueryStr;
    private FileSizeConstraints mFileSizeConstraints;
    private int mMaxResult;
    private MsgQuery mQuery;
    private ArrayList queryHitList;
    private boolean isSearching = false;
    private long mSearchStartTime;

    /**
     * The number of millis a search is running before it times out.
     * This is introduced since searches are now forwarded to new connections to
     * limit trafic here a search has to time out.
     */
    private long searchTimeout;
    private String fixedDisplayStr;
    private NewConnectedHostListener newHostListener;

    /**
     * All listeners interested in events.
     */
    private ArrayList listenerList = new ArrayList( 2 );


    private Search()
    {
        // disable
    }

    public Search( int minSpeed, String queryStr, FileSizeConstraints fileSizeConstraints )
    {
        this( minSpeed, queryStr, fileSizeConstraints,
            ServiceManager.sCfg.mSearchMaxSearch, DEFAULT_SEARCH_TIMEOUT );
    }

    public Search( int minSpeed, String queryStr, FileSizeConstraints fileSizeConstraints,
        int maxResults, long aSearchTimeout )
    {
        queryHitList = new ArrayList();
        searchContainer = ServiceManager.getQueryManager().getSearchContainer();
        mMinSpeed = minSpeed;
        mQueryStr = queryStr;
        mFileSizeConstraints = fileSizeConstraints;
        mMaxResult = maxResults;
        searchTimeout = aSearchTimeout;


        mQuery = new MsgQuery();
        mQuery.getHeader().setTTL(ServiceManager.sCfg.mNetTTL);
        mQuery.setMinSpeed((short)minSpeed);
        mQuery.setSearchString(queryStr);

        fixedDisplayStr = "[Search=" + queryStr +
                      "]    [Max Result=" + mMaxResult +
                      "]    [Min Speed=" + minSpeed +
                      "]    [Min Size=" + fileSizeConstraints.getLowerBound() + " bytes]";

        newHostListener = new NewConnectedHostListener();
    }


    public int getQueryHitCount()
    {
        synchronized ( queryHitList )
        {
            return queryHitList.size();
        }
    }

    public RemoteFile getQueryHit( int index )
    {
        synchronized ( queryHitList )
        {
            return (RemoteFile) queryHitList.get( index );
        }
    }

    /**
     * for compatibility with RemoteFile of 0.4.6
     * @deprecated don't use. Use getQueryHit() or getQueryHitCount()
     */
    public Vector getQueryResults()
    {
        return new Vector(queryHitList);
    }

    public int getMinSpeed()
    {
        return mMinSpeed;
    }


    public String getQueryStr()
    {
        return mQueryStr;
    }


    public String getDisplayStr()
    {
        return fixedDisplayStr + "    [Results=" + getQueryHitCount() + ']';
    }


    public MsgQuery getQuery()
    {
        return mQuery;
    }

    public boolean isSearching()
    {
        return isSearching;
    }


    public void startSearching()
    {
        //System.out.println( "start " + mQueryStr);
        SwingUtilities.invokeLater( new Runnable()
        {
            public void run()
            {
                // add to seen list.
                mMsgMgr.checkAndAddMsgSeen( mQuery );
                mHostMgr.addNetworkHostsChangeListener( newHostListener );
                mMsgMgr.sendMsgToHosts( mQuery );
                isSearching = true;
                mSearchStartTime = System.currentTimeMillis();
                fireSearchStarted();
            }
        } );
    }

    public void stopSearching()
    {
        if ( !isSearching )
        {// already stoped
            return;
        }
        isSearching = false;
        mHostMgr.removeNetworkHostsChangeListener( newHostListener );
        fireSearchStoped();
    }


    public void processResponse(Host remoteHost, MsgQueryResponse msg)
    {
        // Handle responses that are comming for old searches
        // I see no reason why we should not count them
        // if (!isSearching)
        // {
        //     return;
        // }

        // Is it a response for the query?
        if (!msg.getHeader().getMsgID().equals(mQuery.getHeader().getMsgID()))
        {
            return;
        }

        if ( IPUtils.isHostInUserFilter( msg.getRemoteHost().getHostAddress() ) )
        {
            //System.out.println( "Filter "+ msg.getRemoteHost().getHostAddress());
            return;
        }

        // remoteHost.log("Got response to my query.  " + msg);
        String hostStr = msg.getRemoteHostStr();
        int speed = msg.getRemoteHostSpeed();
        GUID rcID = msg.getRemoteClientID();
        MsgResRecord rec;
        RemoteFile rfile;

        int addStartIdx = queryHitList.size();
        for (int i = 0; i < msg.getRecordCount(); i++)
        {
            rec = msg.getMsgRecord(i);

            long fileSize = rec.getFileSize();
            if( fileSize < mFileSizeConstraints.getLowerBound() ||
                fileSize > mFileSizeConstraints.getUpperBound()
              )
            {
                //remoteHost.log( "Query result " + fileSize + " outside min: " + mFileSizeConstraints.getLowerBound() +
                //                " max: " + mFileSizeConstraintss.getUpperBound() );
                //System.out.println( "Query result " + fileSize + " outside min: " + mFileSizeConstraints.getLowerBound() +
                //                " max: " + mFileSizeConstraints.getUpperBound() );
                continue;
            }

            String filename = rec.getFilename();
            if ( isResultFiltered( filename ) )
            {
                //remoteHost.log( "Query result " + filename + " filtered." );
                continue;
            }

            synchronized ( queryHitList )
            {
                short score = Search.calculateSearchScore( mQueryStr, filename );
                rfile = new RemoteFile(
                    rcID, rec.getFileIndex(), filename, fileSize, hostStr,
                    speed, score );
                queryHitList.add( rfile );
                if ( queryHitList.size() >= mMaxResult )
                {
                    stopSearching();
                    break;
                }
            }
        }
        int addEndIdx = queryHitList.size();
        // if something was added...
        if ( addEndIdx > addStartIdx )
        {
            fireSearchHitsAdded( addStartIdx, addEndIdx );
        }
    }

    public void checkForSearchTimeout( long currentTime )
    {
        if ( currentTime > mSearchStartTime + searchTimeout )
        {
            // timed out stop search
            stopSearching();
        }
    }

    /**
     * This methods calculates the score of a search result. The return value is
     * between 0 and 100. A value of 100 means all terms of the search string
     * are matched 100% in the result string.
     */
    private static short calculateSearchScore( String searchStr, String resultStr )
    {
        double tokenCount = 0;
        double hitCount = 0;
        StringTokenizer tokens = new StringTokenizer( searchStr );
        SearchEngine searchEngine = new SearchEngine();
        searchEngine.setText(resultStr, false);
        while ( tokens.hasMoreTokens() )
        {
            String token = tokens.nextToken();
            tokenCount ++;
            searchEngine.setPattern( token, false );
            if ( searchEngine.match() )
            {
                hitCount ++;
            }
        }
        double perc = hitCount / tokenCount * 100;
        return (short) perc;
    }

    private boolean isResultFiltered(String filename)
    {
        Vector list = ServiceManager.sCfg.mSearchFilterTokens;
        filename = filename.toLowerCase();
        for (int i = 0; i < list.size(); i++)
        {
            if (filename.indexOf(((String)list.elementAt(i)).toLowerCase()) != -1)
            {
                return true;
            }
        }
        return false;
    }


    ///////////////////// START event handling methods ////////////////////////
    public void addSearchChangeListener( SearchChangeListener listener )
    {
        listenerList.add( listener );
    }

    public void removeSearchChangeListener( SearchChangeListener listener )
    {
        listenerList.remove( listener );
    }

    private void fireSearchStarted()
    {
        SearchChangeEvent searchChangeEvent =
            new SearchChangeEvent( this, SearchChangeEvent.SEARCH_STARTED );
        fireSearchChangeEvent( searchChangeEvent );
    }

    private void fireSearchStoped()
    {
        SearchChangeEvent searchChangeEvent =
            new SearchChangeEvent( this, SearchChangeEvent.SEARCH_STOPED );
        fireSearchChangeEvent( searchChangeEvent );
    }

    private void fireSearchHitsAdded( int startIdx, int endIdx)
    {
        SearchChangeEvent searchChangeEvent = new SearchChangeEvent( this,
            SearchChangeEvent.SEARCH_HITS_ADDED, startIdx, endIdx );
        fireSearchChangeEvent( searchChangeEvent );
    }

    private void fireSearchChangeEvent( final SearchChangeEvent searchChangeEvent )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                SearchChangeListener listener;

                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (SearchChangeListener)listeners[ i ];
                    listener.searchChanged( searchChangeEvent );
                }
            }
        });
    }

    /**
     * Class listens to event of the network neighborhood.
     * If a new host is connected the search will be submited to him.
     */
    private class NewConnectedHostListener
        implements NetworkHostsChangeListener
    {
        /**
         * Called if a host of the network neighborhood changed.
         */
        public void networkHostChanged( int position )
        {
            if ( !isSearching )
            {
                return;
            }
            Host newHost = mHostMgr.getConnectedHostAt( position );
            // only use valid hosts with a stable connection status
            if (     newHost == null
                 || !newHost.isConnectionStable() )
            {
                return;
            }
            mMsgMgr.sendMsgToHost( mQuery, newHost );
        }

        /**
         * Called if a host is added to the network neighborhood.
         */
        public void networkHostAdded( int position )
        {// don't care
        }

        /**
         * Called if a host is removed from the network neighborhood.
         */
        public void networkHostRemoved( int position )
        {// don't care
        }
    }
    ///////////////////// END event handling methods ////////////////////////

    /**
     * Class which encapsulates the upper and lower file size boundaries that a
     * search must conform to.
     */
    public final static class FileSizeConstraints
    {
        private FileSizeConstraints() { /* Must be created with a lower and upper */ lower = upper = 0; }
        public FileSizeConstraints( long lower, long upper )
        {
             this.lower = lower;
             this.upper = upper;
        }

        public long getUpperBound() { return upper; }
        public long getLowerBound() { return lower; }

        final private long lower;
        final private long upper;
    }
}