/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.query;

import java.util.*;
import javax.swing.SwingUtilities;
import phex.host.Host;
import phex.event.SearchListChangeListener;
import phex.event.SearchChangeListener;
import phex.event.SearchChangeEvent;
import phex.msg.MsgQueryResponse;
import phex.query.Search;
import phex.ServiceManager;

public class SearchContainer
{
    /**
     * All listeners interested in events.
     */
    private ArrayList listenerList = new ArrayList( 2 );

    // to make this protected is not nice but easy to let the background search
    // container share.
    protected ArrayList searchList;

    private SearchChangeListener searchChangeListener;

    public SearchContainer()
    {
        searchChangeListener = new SingleSearchChangeListener();
        searchList = new ArrayList();
    }

    /** 
     * Create a Search object and call startSearching on it.
     */
    public synchronized Search createSearch( String queryStr, int minSpeed, Search.FileSizeConstraints sizeConstraints )
    {
        Search search = new Search( minSpeed, queryStr, sizeConstraints );
        insertToSearchList( search, 0 );

        if ( searchList.size() > ServiceManager.sCfg.mSearchMaxConcurrent )
        {
            int idx = searchList.size() - 1;
            removeFromSearchList( idx );
        }
        
        search.startSearching();
        
        return search;
    }

    /**
     * The only method allowed to actually add a search to the list.
     */
    protected synchronized void insertToSearchList( Search search, int position )
    {
        search.addSearchChangeListener( searchChangeListener );
        searchList.add( position, search );
        fireSearchAdded( position );
    }

    /**
     * The only method allowed to actually remove a search from the list.
     * The search is stopped before it's removed
     */
    protected synchronized void removeFromSearchList( int index )
    {
        Search search = getSearchAt( index );
        search.stopSearching();
        search.removeSearchChangeListener( searchChangeListener );
        searchList.remove( index  );
        fireSearchRemoved( index );
    }

    public synchronized int getSearchCount()
    {
        return searchList.size();
    }

    public synchronized Search getSearchAt( int index )
    {
        if (index > getSearchCount() - 1)
        {
            return null;
        }
        return (Search)searchList.get( index );
    }

    /**
     * Removes the search from the search list. The search will be stoped before
     * it's removed.
     */
    public synchronized void removeSearch( Search search )
    {
        int index = searchList.indexOf( search );
        // if a search was found.
        if ( index >= 0 )
        {
            removeFromSearchList( index );
        }
    }

    public synchronized void removeSearch( int index )
    {
        removeFromSearchList( index );
    }

    /**
     * Stops all searches where the timeout has passed.
     */
    public synchronized void stopExpiredSearches( long currentTime )
    {
        for (int i = 0; i < searchList.size(); i++)
        {
            ( (Search)searchList.get( i ) ).checkForSearchTimeout( currentTime );
        }
    }

    public synchronized void stopAllSearches()
    {
        for (int i = 0; i < searchList.size(); i++)
        {
            ( (Search)searchList.get( i ) ).stopSearching();
        }
    }

    public synchronized void removeAllSearches()
    {
        for ( int i = searchList.size() - 1; i >= 0; i-- )
        {
            removeFromSearchList( i );
        }
    }

    public synchronized void processQueryResponse( Host remoteHost,
        MsgQueryResponse msg )
    {
        for (int i = 0; i < searchList.size(); i++)
        {
            ( (Search)searchList.get( i ) ).processResponse( remoteHost, msg );
        }
    }

    ///////////////////// START event handling methods ////////////////////////

    public void addSearchListChangeListener( SearchListChangeListener listener )
    {
        listenerList.add( listener );
    }

    public void removeSearchListChangeListener( SearchListChangeListener listener )
    {
        listenerList.remove( listener );
    }

    protected void fireSearchChanged( final int position )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                SearchListChangeListener listener;
                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (SearchListChangeListener)listeners[ i ];
                    listener.searchChanged( position );
                }
            }
        });
    }

    protected void fireSearchAdded( final int position )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                SearchListChangeListener listener;
                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (SearchListChangeListener)listeners[ i ];
                    listener.searchAdded( position );
                }
            }
        });
    }

    protected void fireSearchRemoved( final int position )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                SearchListChangeListener listener;
                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (SearchListChangeListener)listeners[ i ];
                    listener.searchRemoved( position );
                }
            }
        });
    }

    protected void setSearchChangeListener( SearchChangeListener listener )
    {
        searchChangeListener = listener;
    }

    private class SingleSearchChangeListener implements SearchChangeListener
    {
        public void searchChanged( SearchChangeEvent event )
        {
            Search source = (Search) event.getSource();
            fireSearchChanged( source );
        }
    }

    protected void fireSearchChanged( Search search )
    {
        int position = searchList.indexOf( search );
        if ( position >= 0 )
        {
            fireSearchChanged( position );
        }
    }
    ///////////////////// END event handling methods ////////////////////////
}