/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package phex.share;

import java.io.File;
import java.util.*;
import javax.swing.SwingUtilities;
import phex.ServiceManager;
import phex.event.SharedFilesChangeListener;
import phex.config.Cfg;
import phex.utils.ReadWriteLock;

/**
 * A administrator of the shared files. It provides all necessary accessor
 * methods to administrate the shared files.
 */
public class FileAdministration
{
    private final static String aliasFile = "filealias";
    private final static String aliasFilename = aliasFile + ".txt";

    private ReadWriteLock rwLock;

    // needs locking
    private ArrayList sharedDirectories;
    // needs locking
    private ArrayList sharedFiles;
    // needs locking
    private HashMap nameToFileMap;

    // needs no locking
    private ArrayList fileExclusionList;
    // needs no locking
    private ArrayList fileInclusionList;

    private int totalFileSizeKb;
    private ShareManager shareManager;
    /**
     * if this thread is alive a rescan is running.
     */
    private Thread rescanThread;

    /**
     * All listeners interested in events.
     */
    private ArrayList listenerList = new ArrayList( 2 );


    public FileAdministration( ShareManager mgr )
    {
        rwLock = new ReadWriteLock();
        shareManager = mgr;
        sharedDirectories = new ArrayList( 1 );
        sharedFiles = new ArrayList();
        fileExclusionList = new ArrayList();
        fileInclusionList = new ArrayList();
        nameToFileMap = new HashMap();
        totalFileSizeKb = 0;
    }

    /**
     * Gets the file at an index.
     *
     * TODO Actually this method is only for gui representation. I don't like
     * this kind of stuff. It's not really safe to occuring changes. The
     * TableModel should do a better job here to provider better data abstraction.
     */
    public ShareFile getFileAt( int index )
    {
        rwLock.readLock();
        try
        {
            if ( index >= sharedFiles.size() )
            {
                return null;
            }
            return (ShareFile) sharedFiles.get( index );
        }
        finally
        {
            try{ rwLock.readUnlock(); }
            catch (IllegalAccessException exp )
            { exp.printStackTrace(); }
        }
    }

    /**
     * Returns an array of all shared files.
     */
    public ShareFile[] getSharedFiles()
    {
        rwLock.readLock();
        try
        {
            ShareFile[] array = new ShareFile[ sharedFiles.size() ];
            array = (ShareFile[])sharedFiles.toArray( array );
            return array;
        }
        finally
        {
            try{ rwLock.readUnlock(); }
            catch (IllegalAccessException exp )
            { exp.printStackTrace(); }
        }
    }

    /**
     * Returns an array of all shared directories. A Object array is returned instead
     * of an shared file array for performence and memory reasons. This way no
     * array converting is necessary.
     */
    public Object[] getSharedDirectories()
    {
        rwLock.readLock();
        try
        {
            return sharedDirectories.toArray();
        }
        finally
        {
            try{ rwLock.readUnlock(); }
            catch (IllegalAccessException exp )
            { exp.printStackTrace(); }
        }
    }

    /**
     * Returns the shared files count.
     */
    public int getFileCount()
    {
        rwLock.readLock();
        try
        {
            return sharedFiles.size();
        }
        finally
        {
            try{ rwLock.readUnlock(); }
            catch (IllegalAccessException exp )
            { exp.printStackTrace(); }
        }
    }

    /**
     * Returns the a shared file by name.
     */
    public ShareFile getFileByName( String name )
    {
        rwLock.readLock();
        try
        {
            return (ShareFile) nameToFileMap.get( name );
        }
        finally
        {
            try{ rwLock.readUnlock(); }
            catch (IllegalAccessException exp )
            { exp.printStackTrace(); }
        }
    }

    /**
     * Returns a shared file by its index number.
     */
    public ShareFile getFileByIndex( int fileNumber )
        throws IndexOutOfBoundsException
    {
        rwLock.readLock();
        try
        {
            if ( fileNumber >= sharedFiles.size() )
            {
                return null;
            }
            return (ShareFile) sharedFiles.get( fileNumber );
        }
        finally
        {
            try{ rwLock.readUnlock(); }
            catch (IllegalAccessException exp )
            { exp.printStackTrace(); }
        }
    }

    /**
     * Returns the total size of all shared files in KB.
     */
    public int getTotalFileSizeInKb()
    {
        rwLock.readLock();
        try
        {
            return totalFileSizeKb;
        }
        finally
        {
            try{ rwLock.readUnlock(); }
            catch (IllegalAccessException exp )
            { exp.printStackTrace(); }
        }
    }

    /**
     * Rescans the shared files. Locking is done in the called methods to
     * have gaps between a rescanning session.
     */
    public void rescan()
    {
        if ( rescanThread == null || !rescanThread.isAlive() )
        {
            FileRescanRunner runner = new FileRescanRunner();
            rescanThread = new Thread( runner );
            rescanThread.setDaemon( true );
            rescanThread.setPriority( Thread.MIN_PRIORITY );
            rescanThread.start();
        }
    }

    /**
     * In case the user is sharing the download directory, skip the
     * download-in-progress files, index files and alias files. Even though
     * the user should not be able to configure the download directory as shared
     * directory since the new option dialog.
     */
    public boolean isFileInvalid( File file )
    {
        // In case the user is sharing the download directory,
        // skip the download-in-progress files.
        if (file.getName().toLowerCase().endsWith(".dl"))
        {
            return true;
        }
        // Skip PHEX generated index file.
        if (file.getName().equals("files.idx"))
        {
            return true;
        }
        // Skip alias file.
        if (file.getName().equals( aliasFilename ))
        {
            return true;
        }

        rwLock.readLock();
        try
        {
            if ( !isIncluded(file) )
            {
                return true;
            }
            if ( isExcluded(file) )
            {
                return true;
            }
        }
        finally
        {
            try{ rwLock.readUnlock(); }
            catch (IllegalAccessException exp )
            { exp.printStackTrace(); }
        }
        return false;
    }


    ////////////////// private methods //////////////////////////////7

    private void setInclusions(String str)
    {
        StringTokenizer	tokens = new StringTokenizer(str, ";");
        fileInclusionList.clear();
        while (tokens.hasMoreTokens())
        {
            fileInclusionList.add(tokens.nextToken().trim().toLowerCase());
        }
    }

    private boolean isIncluded( File file )
    {
        // use filter only for files not directories
        if ( file.isDirectory() )
        {
            return true;
        }

        String name = file.getName().toLowerCase();

        for (int i = 0; i < fileInclusionList.size(); i++)
        {
            String	pattern = (String) fileInclusionList.get( i );

            if (pattern.equals("*"))
            {
                return true;
            }
            if (pattern.equals("*.*"))
            {
                return true;
            }

            // Take care of *exp and exp* cases.
            if (pattern.startsWith("*"))
            {
                pattern = pattern.substring(1);
            }
            if (pattern.endsWith("*"))
            {
                pattern = pattern.substring(0, pattern.length() - 1);
            }

            if (name.indexOf(pattern) != -1)
            {
                return true;
            }
        }
        return false;
    }

    private void setExclusions(String str)
    {
        StringTokenizer	tokens = new StringTokenizer(str, ";");
        fileExclusionList.clear();

        while (tokens.hasMoreTokens())
        {
            fileExclusionList.add(tokens.nextToken().trim().toLowerCase());
        }
    }

    private boolean isExcluded(File file)
    {
        // use filter only for files not directories
        if ( file.isDirectory() )
        {
            return false;
        }

        String name = file.getName().toLowerCase();

        // Exclude any file contain the "filealias" string.
        if (name.indexOf( aliasFile ) != -1)
        {
            return true;
        }

        for (int i = 0; i < fileExclusionList.size(); i++)
        {
            String pattern = (String) fileExclusionList.get( i );
            if (name.indexOf(pattern) != -1)
            {
                return true;
            }
        }
        return false;
    }

    private void scanDir(File dir, boolean recursive)
    {
        File[] files = dir.listFiles();

        if ( files == null )
        {
            System.out.println("'" + dir + "' is not a directory.");
            return;
        }

        //Properties aliasMapping = loadAliasFile(dir);

        /*
        Properties wordFiles = new Properties();
        if (!ServiceManager.sCfg.mIndexFiles)
        {
            try
            {
                FileInputStream	is = new FileInputStream(new File(dir, "files.idx"));
                wordFiles.load(is);
                is.close();
            }
            catch (IOException e)
            {
                // e.printStackTrace();
            }
        }
        */

        for (int j = 0; j < files.length; j++)
        {
            if ( isFileInvalid( files[j] ) )
            {
                continue;
            }

            if ( files[j].isFile() )
            {
                handleScannedFile( files[j] );
            }
            else if ( files[j].isDirectory() && recursive )
            {
                scanDir(files[j], recursive);
            }
        }

        /*if (ServiceManager.sCfg.mIndexFiles)
        {
            try
            {
                FileOutputStream	os = new FileOutputStream(new File(dir, "files.idx"));
                wordFiles.save(os, "PHEX's Words To Files Index");
                os.close();
            }
            catch (IOException e)
            {
                e.printStackTrace();
            }
        }*/
    }

    private void handleScannedFile( File file )
    {
        /*if (ServiceManager.sCfg.mIndexFiles)
        {
            indexFile(wordFiles, files[j]);
        }*/

        ShareFile shareFile = new ShareFile( file );
        rwLock.writeLock();
        try
        {
            int position = sharedFiles.size();
            shareFile.setFileIndex( position );
            //shareFile.setAlias((String)aliasMapping.get(file.getName()));
            //addWordIndex(file.getName(), fileNumber, wordFiles);

            sharedFiles.add( position, shareFile );
            nameToFileMap.put( file.getAbsolutePath(), shareFile );
            totalFileSizeKb += file.length() / 1024;
            fireSharedFileAdded( position );
        }
        finally
        {
            try{ rwLock.writeUnlock(); }
            catch (IllegalAccessException exp )
            { exp.printStackTrace(); }
        }
    }

    /**
     * Called when rescanning files
     */
    private void setSharedDirectories(String dirs)
    {
        rwLock.writeLock();
        try
        {
            StringTokenizer	tokens = new StringTokenizer(dirs, ";");
            int count = tokens.countTokens();

            sharedDirectories.clear();
            sharedDirectories.ensureCapacity( count );

            while (tokens.hasMoreTokens())
            {
                String dir = tokens.nextToken().trim();
                sharedDirectories.add( dir );
            }
        }
        finally
        {
            try{ rwLock.writeUnlock(); }
            catch (IllegalAccessException exp )
            { exp.printStackTrace(); }
        }
    }

    ///////////////////// START event handling methods ////////////////////////
    public void addSharedFilesChangeListener( SharedFilesChangeListener listener )
    {
        listenerList.add( listener );
    }

    public void removeSharedFilesChangeListener( SharedFilesChangeListener listener )
    {
        listenerList.remove( listener );
    }

    private void fireSharedFileChanged( final int position )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                SharedFilesChangeListener listener;
                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (SharedFilesChangeListener)listeners[ i ];
                    listener.sharedFileChanged( position );
                }
            }
        });
    }

    private void fireSharedFileAdded( final int position )
    {
        Object[] listeners = listenerList.toArray();
        SharedFilesChangeListener listener;
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for ( int i = listeners.length - 1; i >= 0; i-- )
        {
            listener = (SharedFilesChangeListener)listeners[ i ];
            listener.sharedFileAdded( position );
        }
    }

    private void fireAllSharedFilesChanged( )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                SharedFilesChangeListener listener;
                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (SharedFilesChangeListener)listeners[ i ];
                    listener.allSharedFilesChanged( );
                }
            }
        });
    }

    public void fireSharedFileChanged( ShareFile file )
    {
        int position = sharedFiles.indexOf( file );
        if ( position >= 0 )
        {
            fireSharedFileChanged( position );
        }
    }
    ///////////////////// END event handling methods ////////////////////////

    ///////////////////// START inner classes ///////////////////////////////

    class FileRescanRunner implements Runnable
    {
        public void run()
        {
            Cfg cfg = ServiceManager.sCfg;
            setSharedDirectories( cfg.mUploadDir );

            setExclusions( cfg.mUploadFileExclusions );
            setInclusions( cfg.mUploadFileInclusions );

            rwLock.writeLock();
            try
            {
                sharedFiles.clear();
                nameToFileMap.clear();
                totalFileSizeKb = 0;
            }
            finally
            {
                try{ rwLock.writeUnlock(); }
                catch (IllegalAccessException exp )
                { exp.printStackTrace(); }
            }
            fireAllSharedFilesChanged();

            rwLock.readLock();
            try
            {
                Iterator iterator = sharedDirectories.iterator();
                while( iterator.hasNext() )
                {
                    String dirName = (String) iterator.next();
                    File dir = new File( dirName );
                    scanDir(dir, ServiceManager.sCfg.mUploadScanRecursively);
                }
            }
            finally
            {
                try{ rwLock.readUnlock(); }
                catch (IllegalAccessException exp )
                { exp.printStackTrace(); }
            }
        }
    }
}