/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2000 William W. Wong
 *  williamw@jps.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package phex.share;

import java.io.*;
import java.net.*;
import java.util.*;

import phex.*;
import phex.share.UploadFileContainer;
import phex.common.TransferDataProvider;
import phex.host.*;

public class UploadFile implements TransferDataProvider
{
    public static final int	sIdle = 0;
    public static final int	sQueued = 1;
    public static final int	sUploading = 3;
    public static final int	sCompleted = 4;
    public static final int	sError = 5;
    public static final int	sAborted = 6;

    private File			mFile;
    private String			mFilename;
    private Host			mRemoteHost;
    private long			mTransferLength;
    private long			mFileSize;
    private String			mRemoteAppName;
    private int				mStatus = sIdle;
    private String			mStatusText = "";
    private long                        mStatusTime;
    private String			mLog = "";
    private long			mTransferredSize = 0;
    private int transferRate;
    private long startTime;
    private long stopTime;
    private long transferRateTimestamp;
    private int transferRateBytes;
    private UploadFileContainer uploadContainer;
    private Integer currentProgress;

    public UploadFile(File file, Host remoteHost, long transferLength, String remoteAppName)
    {
        mFile = file;
        mRemoteHost = remoteHost;
        mTransferLength = transferLength;
        mRemoteAppName = remoteAppName;
        transferRate = 0;
        transferRateTimestamp = 0;
        transferRateBytes = 0;
        startTime = 0;

        mFilename = mFile.getName();
        mFileSize = mFile.length();
        uploadContainer = ServiceManager.getShareManager().getUploadFileContainer();
        
        currentProgress = new Integer(0);
    }


    public File getFile()
    {
        return mFile;
    }

    public String getFilename()
    {
        return mFilename;
    }


    public Host getRemoteHost()
    {
        return mRemoteHost;
    }

    public long getTransferDataSize()
    {
        return mTransferLength;
    }

    /**
     * Returns the progress in percent. If mStatus == sCompleted will always be 100%.
     */
    // TODO: This is exactly the same implementation as DownloadFile. Should they be combined?
    public Integer getProgress()
    {
        int percentage;
        
        if( mStatus == sCompleted )
        {
            percentage = 100;
        }
        else
        {
            long toTransfer = getTransferDataSize();
            percentage = (int)(getTransferredDataSize() * 100L / (toTransfer == 0L ? 1L : toTransfer));
        }
         
        if ( currentProgress.intValue() != percentage )
        {
            // only create new object if necessary
            currentProgress = new Integer( percentage );
        }
        
        return currentProgress;
    }

    public String getRemoteAppName()
    {
        return mRemoteAppName;
    }


    public int getStatus()
    {
        return mStatus;
    }

    public short getDataTransferStatus()
    {
        switch ( mStatus )
        {
            case sUploading:
                return TransferDataProvider.TRANSFER_RUNNING;
            case sError:
            case sAborted:
                return TransferDataProvider.TRANSFER_ERROR;
            case sCompleted:
                return TransferDataProvider.TRANSFER_COMPLETED;
            default:
                return TransferDataProvider.TRANSFER_NOT_RUNNING;
        }
    }

    public String getStatusName()
    {
        switch (mStatus)
        {
            case sIdle:
                return "Idle";

            case sQueued:
                return "Queued";

            case sUploading:
                return "Uploading";

            case sCompleted:
                return "Completed";

            case sError:
                return "Error " + mStatusText;

            case sAborted:
                return "Aborted";
        }
        return "Unknown";
    }


    public void setStatus(int status)
    {
        setStatus( status, "" );
    }


    public void setStatus(int status, String text)
    {
        mStatus = status;
        mStatusText = text;
        mStatusTime = System.currentTimeMillis();
        uploadContainer.fireUploadFileChanged( this );
    }


    public void setLog(String log)
    {
        mLog = log;
    }


    public void appendLog(String log)
    {
        if (mLog != null)
            mLog += log + "\n";
        else
            mLog = log + "\n";
    }


    public String getLog()
    {
        return mLog;
    }

    /**
     * Indicate that the download is just starting.
     */
    public void setStartingTime( long startingTime )
    {
        transferRateTimestamp = startingTime;
        startTime = startingTime;
        stopTime = 0;
    }
    
    /**
     * Indicate that the download is no longer running.
     */
    public void setStoppingTime( long stoppingTime )
    {
        // Ignore nested calls.
        if( stopTime == 0 )
        {
            stopTime = stoppingTime; 
        }
    }

    /**
     * Indicate how much of the file has been uploaded on this transfer.
     */
    public long getTransferredDataSize()
    {
        return mTransferredSize;
    }
   
    /**
     * Returns the length of time, in seconds, that this transfer has
     * been TRANSFER_RUNNING so that one may calculate average
     * transfer rates.
     */
    public long getTransferTimeInSeconds()
    {
        // There are two cases; a download is still in progress, or it's been stopped.
        // If stopTime is non zero, then it's been stopped.
        if( stopTime != 0 )
        {
            return (stopTime - startTime) / 1000;
        }
        else
        {
            // Get current elapsed time and convert millis into seconds
            // return totalTransferTime / 1000;
            return (System.currentTimeMillis() - startTime) / 1000;
        }
    }

    public void setTransferredDataSize( long transferredSize )
    {
        long diff = transferredSize - mTransferredSize;
        transferRateBytes += diff;

        mTransferredSize = transferredSize;
        uploadContainer.fireUploadFileChanged( this );
    }

    /**
     * To be able to provide a constantly valid data transfer rate the time
     * to calculate the data rate from needs to be updated.
     * If you want your timestamp to be updated regularly you need to register
     * your TransferDataProvider with the TransferRateService.
     */
    public void setTransferRateTimestamp( long timestamp )
    {
        transferRateTimestamp = timestamp;
        transferRateBytes = 0;
    }

    public int getDataTransferRate()
    {
        if ( transferRateTimestamp != 0 )
        {
            double sec = (double)(System.currentTimeMillis() - transferRateTimestamp) / (double)1000;
            // don't drop transfer rate to 0 if we just have a new timestamp and
            // no bytes transfered
            if ( transferRateBytes > 0 || sec > 1 )
            {
                transferRate = (int) ( transferRateBytes / sec );
            }
        }
        return transferRate;
    }

    public synchronized void abortUpload()
    {
        setStatus( UploadFile.sAborted );
        mRemoteHost.disconnect();
    }
}
