/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.share;

import java.util.*;
import javax.swing.*;
import phex.ServiceManager;
import phex.common.TransferRateService;
import phex.event.UploadFilesChangeListener;
import phex.host.Host;

public class UploadFileContainer
{
    /**
     * All listeners interested in events.
     */
    private ArrayList listenerList = new ArrayList( 2 );

    private ArrayList uploadFileList;
    private TransferRateService transferRateService;

    public UploadFileContainer()
    {
        uploadFileList = new ArrayList();
        transferRateService = ServiceManager.getTransferRateService();
    }

    /**
     * Returns the number of all files in the upload list. Also completed...
     */
    public synchronized int getUploadListSize()
    {
        return uploadFileList.size();
    }

    /**
     * Returns only the number of files that are currently getting uploaded.
     * TODO it's better to maintain the number of files in an attribute...
     */
    public synchronized int getUploadFileCount()
    {
        int count = 0;
        for( int i = uploadFileList.size() - 1; i >= 0; i-- )
        {
            UploadFile file = (UploadFile)uploadFileList.get( i );
            if ( file.getStatus() == UploadFile.sUploading )
            {
                count ++;
            }
        }
        return count;
    }

    public synchronized UploadFile getUploadFileAt( int index )
    {
        return (UploadFile) uploadFileList.get( index );
    }

    public synchronized void addUploadFile( UploadFile file )
    {
        int position = uploadFileList.size();
        uploadFileList.add( position, file );
        transferRateService.registerTransferDataProvider( file );
        fireUploadFileAdded( position );
    }

    public synchronized int getUploadCountPerIP( String hostAddr )
    {
        int count = 0;
        Iterator iterator = uploadFileList.iterator();

        while( iterator.hasNext() )
        {
            UploadFile file = (UploadFile) iterator.next();
            if ( file.getStatus() == UploadFile.sUploading &&
                 file.getRemoteHost().getHostAddress().getHostName().equals( hostAddr ) )
            {
                count ++;
            }
        }
        return count;
    }

    public synchronized void cleanUploadFileList()
    {
        for( int i = uploadFileList.size() - 1; i >= 0; i-- )
        {
            UploadFile file = (UploadFile)uploadFileList.get( i );
            if ( file.getStatus() == UploadFile.sQueued ||
                 file.getStatus() == UploadFile.sUploading )
            {
                continue;
            }
            transferRateService.unregisterTransferDataProvider( file );
            uploadFileList.remove( i );
            fireUploadFileRemoved( i );
        }
    }

    ///////////////////// START event handling methods ////////////////////////
    public void addUploadFilesChangeListener( UploadFilesChangeListener listener )
    {
        listenerList.add( listener );
    }

    public void removeUploadFilesChangeListener( UploadFilesChangeListener listener )
    {
        listenerList.remove( listener );
    }

    private void fireUploadFileChanged( final int position )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                UploadFilesChangeListener listener;
                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (UploadFilesChangeListener)listeners[ i ];
                    listener.uploadFileChanged( position );
                }
            }
        });
    }

    private void fireUploadFileAdded( final int position )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                UploadFilesChangeListener listener;
                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (UploadFilesChangeListener)listeners[ i ];
                    listener.uploadFileAdded( position );
                }
            }
        });
    }

    private void fireUploadFileRemoved( final int position )
    {
        // invoke update in event dispatcher
        SwingUtilities.invokeLater(
        new Runnable()
        {
            public void run()
            {
                Object[] listeners = listenerList.toArray();
                UploadFilesChangeListener listener;
                // Process the listeners last to first, notifying
                // those that are interested in this event
                for ( int i = listeners.length - 1; i >= 0; i-- )
                {
                    listener = (UploadFilesChangeListener)listeners[ i ];
                    listener.uploadFileRemoved( position );
                }
            }
        });
    }

    public void fireUploadFileChanged( UploadFile file )
    {
        int position = uploadFileList.indexOf( file );
        if ( position >= 0 )
        {
            fireUploadFileChanged( position );
        }
    }
    ///////////////////// END event handling methods ////////////////////////
}