/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2001 Gregor Koukkoullis ( phex@kouk.de )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package phex.utils;

import java.net.*;
import java.util.*;
import phex.ServiceManager;

public final class IPUtils
{
    private static ArrayList invalidIPs;
    private static ArrayList privateIPs;

    static
    {
        invalidIPs = new ArrayList();
        invalidIPs.add( "0.0.0.0" );
        invalidIPs.add( "224.*.*.*" );
        invalidIPs.add( "225.*.*.*" );
        invalidIPs.add( "226.*.*.*" );
        invalidIPs.add( "227.*.*.*" );
        invalidIPs.add( "228.*.*.*" );
        invalidIPs.add( "229.*.*.*" );
        invalidIPs.add( "230.*.*.*" );
        invalidIPs.add( "231.*.*.*" );
        invalidIPs.add( "232.*.*.*" );
        invalidIPs.add( "233.*.*.*" );
        invalidIPs.add( "234.*.*.*" );
        invalidIPs.add( "235.*.*.*" );
        invalidIPs.add( "236.*.*.*" );
        invalidIPs.add( "237.*.*.*" );
        invalidIPs.add( "238.*.*.*" );
        invalidIPs.add( "239.*.*.*" );

        privateIPs = new ArrayList();
        privateIPs.add( "10.*.*.*" );
        privateIPs.add( "172.16.*.*" );
        privateIPs.add( "172.17.*.*" );
        privateIPs.add( "172.18.*.*" );
        privateIPs.add( "172.19.*.*" );
        privateIPs.add( "172.20.*.*" );
        privateIPs.add( "172.21.*.*" );
        privateIPs.add( "172.22.*.*" );
        privateIPs.add( "172.23.*.*" );
        privateIPs.add( "172.24.*.*" );
        privateIPs.add( "172.25.*.*" );
        privateIPs.add( "172.26.*.*" );
        privateIPs.add( "172.27.*.*" );
        privateIPs.add( "172.28.*.*" );
        privateIPs.add( "172.29.*.*" );
        privateIPs.add( "172.30.*.*" );
        privateIPs.add( "172.31.*.*" );
        privateIPs.add( "192.168.*.*" );
    }

    private IPUtils()
    {
    }

    public static boolean isInvalidIP( String ip )
    {
        Iterator iterator = invalidIPs.iterator();
        return isIPInIterator( ip, iterator );
    }

    /**
     * Performance optimized method compared to isInvalidIP( String )
     */
    public static boolean isInvalidIP( byte[] ip )
    {
        // BearShare seems to use ip with 0.0.0.0
        if ( ip[0] == (byte)0 && ip[1] == (byte)0 && ip[2] == (byte)0 &&
             ip[3] == (byte)0 )
        {
            return true;
        }
        if ( ip[0] >= 224 && ip[0] <= 239 )
        {
            return true;
        }
        return false;
    }


    public static boolean isPrivateIP( String ip )
    {
        Iterator iterator = privateIPs.iterator();
        return isIPInIterator( ip, iterator );
    }

    /**
     * Performance optimized method compared to isPrivateIP( String )
     */
    public static boolean isPrivateIP( byte[] ip )
    {
        //10.*.*.*
        if ( ip[0] == (byte)10 )
        {
            return true;
        }
        //172.16.*.* - 172.31.*.*
        if ( ip[0] == (byte)172 && ip[1] >= (byte)16 && ip[1] <= (byte)31 )
        {
            return true;
        }
        //192.168.*.*
        if ( ip[0] == (byte)192 && ip[1] == (byte)168 )
        {
            return true;
        }
        return false;
    }

    public static boolean isHostInUserIgnoreList( String ip )
    {
        return isIPInIterator(ip, ServiceManager.sCfg.mNetIgnoredHosts.iterator() );
    }

    public static boolean isHostInUserInvalidList( String ip )
    {
        return isIPInIterator(ip, ServiceManager.sCfg.mNetInvalidHosts.iterator() );
    }

    public static boolean isHostInUserFilter( String ip )
    {
        return (ServiceManager.sCfg.mApplyFilterdHosts &&
                isIPInIterator(ip, ServiceManager.sCfg.mFilteredSearchHosts.iterator() ) );
    }

    private static boolean isIPInIterator( String ip, Iterator iterator )
    {
        String[] ipParts = splitIP2Parts( ip );
        while ( iterator.hasNext() )
        {
            Object obj = iterator.next();
            String[] compParts;
            if ( obj instanceof String[] )
            {
                compParts = (String[]) obj;
            }
            else
            {
                compParts = splitIP2Parts( (String) obj );
            }

            int i;
            for (i = 0; i < 4; i++ )
            {
                if (compParts[i].equals("*"))
                {
                    continue;
                }
                if (!compParts[i].equals(ipParts[i]))
                {
                    break;
                }
            }
            if ( i == 4 )
            {
                return true;
            }
        }
        return false;
    }

    public static String[] splitIP2Parts(String ip)
    {
        int	i = ip.indexOf(":");
        if (i != -1)
        {
            ip = ip.substring(0, i);
        }
        StringTokenizer	tokens = new StringTokenizer(ip, ".");
        String[]		parts = new String[4];

        parts[0] = tokens.nextToken();
        parts[1] = tokens.nextToken();
        parts[2] = tokens.nextToken();
        parts[3] = tokens.nextToken();

        return parts;
    }


    /**
     * Converts the given bytes of an IP to a string representation.
     */
    public static String ip2string(byte[] ip)
    {
        StringBuffer buffer = new StringBuffer();
        // TODO This is better then the stuff in IOUtil... change it accordingly...
        buffer.append( ((int)ip[0]) & 0xFF );
        buffer.append( '.' );
        buffer.append( ((int)ip[1]) & 0xFF );
        buffer.append( '.' );
        buffer.append( ((int)ip[2]) & 0xFF );
        buffer.append( '.' );
        buffer.append( ((int)ip[3]) & 0xFF );
        return buffer.toString();
    }
}