/*
 *  PHEX - The pure-java Gnutella-servent.
 *  Copyright (C) 2000 William W. Wong (williamw@jps.net)
 *                2001 Peter Hunnisett (hunnise@yahoo.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package phex.utils;

import java.text.*;
import java.util.*;
import java.math.*;

import phex.*;

/**
* A collection of custom utilities to modify or specially display Strings.
**/

public final class StrUtil
{
    // TODO the blanks are used for a fucking bad solution but probably the most performant one...
    // think about it...
    private static String sBlanks = "                                                                                                           ";

    public final static long s1kB = 1024L;        // 1 kilo Byte for computers only
    public final static long s1MB = s1kB * 1024L; // 1 Mega Byte for computers only
    public final static long s1GB = s1MB * 1024L; // 1 Giga Byte for computers only
    public final static long s1TB = s1GB * 1024L; // 1 Tera Byte for computers only
    public final static long s1PB = s1TB * 1024L; // 1 Peta Byte for computers only

    public static DecimalFormat byteFormat = new DecimalFormat("0.00");
    //public static String OS_NAME = System.getProperty("os.name").toUpperCase();

    /**
     * Returns a string describing the version of the PHEX that is presently running along
     * with any special comments about the version. Will print something like
     * "PHEX 0.6.1 (unstable)"
     */
    public static String getAppNameVersion()
    {
        return "PHEX " + Res.getStr( "Program.Version" ) + " " + Res.getStr( "Program.VersionComment" );
    }

    public static String padStringLeft(String str, int width)
    {
        return getPadStr(str, width) + str;
    }

    public static String padStringRight(String str, int width)
    {
        return str + getPadStr(str, width);
    }

    public static String getPadStr(String str, int width)
    {
        int	len = str.length();

        if (len >= width)
        {
            return " ";
        }
        return sBlanks.substring(0, width - len);
    }

    /**
     * Since we are only supporting J2SE 1.2 and J2SE is not available on
     * MACOS 9 and MACOS X is supporting 255 characters we are shorting only
     * for 255 on MAC.
     *
     * MACOS X filename: 255
     * Windows pathlength: 260
     *         filelength: 255
     */
    public static String convertToLocalSystemFilename(String filename)
    {
        // we generally cut of at 255 it will suit everybody and we have no
        // os comparing stuff to do....
        // TODO we need to improve things here to keep up with the window pathlength
        // handling... but for now we just help the mac guys..
        filename = replaceAllChar( filename, '/', "_" );
        return makeShortName(filename, 255);
        /*if ( OS_NAME.indexOf("MAC") != -1)
        {
            return makeShortName(filename, 255);
        }
        else
        {
            return filename;
        }*/
    }

    /**
     * TODO what the hell is happening here?? is all this necessary??
     */
    public static String makeShortName(String name, int maxLen)
    {
        int len = name.length();
        if (len <= maxLen)
        {
            return name;
        }

        byte xor = (byte)0xFF;
        if (len > 255)
            len = 255;

        for (int i = 0; i < len; i++)
        {
            xor ^= ((byte)name.charAt(i));
        }

        byte[]	buf = new byte[1];

        buf[0] = (byte)len;
        String	lenPart = HexDec.convertBytesToHexString(buf);
        buf[0] = xor;
        String	xorPart = HexDec.convertBytesToHexString(buf);

        int extIndex = name.lastIndexOf(".");
        if (extIndex >= name.length() - 4)
        {
            String	ext = name.substring(extIndex);
            int		off = name.length() - extIndex;
            name = name.substring(0, maxLen - 4 - off);	// -4 for 2 hex bytes of xor and 2 hex bytes of len
            return	name + lenPart + xorPart + ext;
        }
        else
        {
            name = name.substring(0, maxLen - 4);	// -4 for 2 hex bytes of xor and 2 hex bytes of len
            return	name + lenPart + xorPart;
        }
    }

    public static String getFileExt(String filename)
    {
        int index = filename.lastIndexOf(".");

        if (index > -1)
        {
            return filename.substring(index + 1, filename.length());
        }
        else
        {
            return "";
        }
    }

    // Replace all occurrences of c with newString in buffer.
    // This returns the same buffer, now modified.
    public static StringBuffer replaceAllChar(StringBuffer buffer, char c, String newString)
    {
        // Go backwards so that when the new string is substituted,
        // the indexing of won't be messed up.
        for (int i = buffer.length() - 1; i >= 0; --i)
        {
            if (buffer.charAt (i) == c)
            {
                buffer.setCharAt(i, newString.charAt(0));
                if (newString.length() > 1)
                    buffer.insert(i+1, newString.substring(1));
            }
        }

        return (buffer);
    }


    public static String replaceAllChar(String orgStr, char c, String replacingPhrase)
    {
        StringBuffer	buf = new StringBuffer(orgStr);
        replaceAllChar(buf, c, replacingPhrase);
        return buf.toString();
    }


    public static String[] hashtableKeys2Strings(Hashtable htable)
    {
        String[]		result = new String[htable.size()];
        int				i = 0;
        Enumeration		keys = htable.keys();

        while (keys.hasMoreElements())
        {
            result[i++] = (String)keys.nextElement();
        }

        return result;
    }


    // Input host addresses in the forms of:
    //	- host:port
    //	- host:port1-port2
    //	- host:port1,port2
    //	- host:port1-port2,port3
    // This is not used or what???? I guess there is some place where this
    // code is double implemented...
    /*public static ArrayList parseHostAddr(String addrs)
            throws Exception
    {
        ArrayList resultAddrs = new ArrayList();

        if (addrs.indexOf("-") == -1 && addrs.indexOf(",") == -1)
        {
            resultAddrs.add( addrs );
            return resultAddrs;
        }

        int index = addrs.indexOf(":");
        if (index == -1)
        {
            throw new Exception("Invalid host format.");
        }

        String			host = addrs.substring(0, index);
        String			ports = addrs.substring(index + 1);

        StringTokenizer	portTokens = new StringTokenizer(ports, ",");
        while (portTokens.hasMoreTokens())
        {
            String		port = portTokens.nextToken();
            index = port.indexOf("-");
            if (index == -1)
            {
                resultAddrs.add( host + ":" + port );
                continue;
            }

            String port1 = port.substring(0, index);
            String port2 = port.substring(index + 1);

            for (int i = Integer.parseInt(port1), end = Integer.parseInt(port2);
                 i <= end; i++)
            {
                resultAddrs.add( host + ":" + i );
            }
        }
        return resultAddrs;
    }
    */


    public static String parseHost(String hostAddr)
    {
        int	index = hostAddr.indexOf(":");
        if (index == -1)
        {
            return hostAddr;
        }
        return hostAddr.substring(0, index);
    }


    public static String parsePort(String hostAddr)
    {
        int	index = hostAddr.indexOf(":");
        if (index == -1)
        {
            return hostAddr;
        }
        return hostAddr.substring(index + 1);
    }


    public static String parseURLHostPort(String url)
    {
        try
        {
            // http://host:port/ref
            int	index = url.indexOf("//");

            if (index == -1)
                return null;
            index = url.indexOf("/", index + 2);
            return url.substring(0, index);
        }
        catch (Exception e)
        {
            return null;
        }
    }

    /**
     * Format time into a hours:minutes:seconds String like: "12:34:56"
     * indicating 12 hours 34 minutes and 56 seconds.
     */
    public static String formatTime( int seconds )
    {
        int hours = seconds / 3600;
        int minutes = (seconds - hours * 3600) / 60;
        int secs = seconds - hours * 3600 - minutes * 60;
        StringBuffer buf = new StringBuffer(16);

        if (hours > 9)
        {
            buf.append(hours);
        }
        else
        {
            buf.append("0");
            buf.append(hours);
        }
        buf.append(":");

        if (minutes > 9)
        {
            buf.append(minutes);
        }
        else
        {
            buf.append("0");
            buf.append(minutes);
        }
        buf.append(":");

        if (secs > 9)
        {
            buf.append(secs);
        }
        else
        {
            buf.append("0");
            buf.append(secs);
        }

        return buf.toString();
    }

    /**
     * Print only the most significant portion of the time. This is
     * the two most significant units of time. Form will be something
     * like "3h 26m" indicating 3 hours 26 minutes and some insignificant
     * number of seconds.
     */
    public static String formatSignificantElapsedTime( long seconds )
    {
        final long secs    = seconds % 60;
        final long minutes = (seconds / 60) % 60;
        final long hours   = (seconds / 3600) % 24;
        final long days    = (seconds / 86400);

        StringBuffer buf = new StringBuffer(16);

        if( days > 0 ) // Display days and hours
        {
            buf.append(days).append("d ").append(hours).append("h");
        }
        else if( hours > 0 ) // Display hours and minutes
        {
            buf.append(hours).append("h ").append(minutes).append("m");
        }
        else if( minutes > 0 ) // Display minutes and seconds
        {
            buf.append(minutes).append("m ").append(secs).append("s");
        }
        else // Display just seconds
        {
            buf.append(secs).append("s");
        }

        return buf.toString();
    }

    /**
     * Formats the the size as a most significant number of bytes.
     */
    public static String formatSizeBytes( double size )
    {
        StringBuffer buf = new StringBuffer( 16 );

        if (size < s1kB)
        {
            byteFormat.format( size, buf, new FieldPosition( 0 ) ).append(" bytes");
        }
        else if (size < s1MB)
        {
            double d = ((double)size) / s1kB;
            byteFormat.format( d, buf, new FieldPosition( 0 ) ).append(" KB");
        }
        else if (size < s1GB)
        {
            double d = ((double)size) / s1MB;
            byteFormat.format( d, buf, new FieldPosition( 0 ) ).append(" MB");
        }
        else if (size < s1TB)
        {
            double d = ((double)size) / s1GB;
            byteFormat.format( d, buf, new FieldPosition( 0 ) ).append(" GB");
        }
        else if (size < s1PB)
        {
            double d = ((double)size) / s1TB;
            byteFormat.format( d, buf, new FieldPosition( 0 ) ).append(" TB");
        }
        else
        {
            double d = ((double)size) / s1PB;
            byteFormat.format( d, buf, new FieldPosition( 0 ) ).append(" PB");
        }
        return buf.toString();
    }


    /**
    * Converts points and dashes to spacesand removes possible
    * file extensions. This should be used to create search-terms
    * from filenames.
    **/

    public static String createNaturalSearchTerm( String strTerm )
    {
        int idx = strTerm.lastIndexOf(".");
        if ( idx >= 0 )
        {
            strTerm = strTerm.substring(0, idx);
        }
        strTerm = strTerm.replace('-', ' ');
        strTerm = strTerm.replace('.', ' ');
        strTerm = strTerm.replace('_', ' ');
        strTerm = strTerm.replace('(', ' ');
        strTerm = strTerm.replace(')', ' ');
        strTerm = strTerm.replace('[', ' ');
        strTerm = strTerm.replace(']', ' ');
        strTerm.toLowerCase();

        strTerm.trim();

        return strTerm;
    }

/*
    public static void main(String[] args)
    {
        System.out.println(formatSizeBytes(599));
        System.out.println(formatSizeBytes(1023));
        System.out.println(formatSizeBytes(1024));
        System.out.println(formatSizeBytes(2000));
        System.out.println(formatSizeBytes(50000));
        System.out.println(formatSizeBytes(500000));
        System.out.println(formatSizeBytes(5000000));
        System.out.println(formatSizeBytes((1L << 20) + 1));
        System.out.println(formatSizeBytes((1L << 20) + 24200));
        System.out.println(formatSizeBytes((1L << 30) + 38000000));
        System.out.println(formatSizeBytes((2L << 40)));

        System.out.println(formatTime(1));
        System.out.println(formatTime(10));
        System.out.println(formatTime(100));
        System.out.println(formatTime(59));
        System.out.println(formatTime(60));
        System.out.println(formatTime(61));
        System.out.println(formatTime(120));
        System.out.println(formatTime(3599));
        System.out.println(formatTime(3600));
        System.out.println(formatTime(3601));
        System.out.println(formatTime(100000));
        System.out.println(formatTime(1000000000));
    }
*/


/*
    // Unit test
    public static void main(String[] args)
    {
        System.out.println(StrUtil.makeShortName("1234567890", 20));
        System.out.println(StrUtil.makeShortName("123456789012345678901234567890", 20));
        System.out.println(StrUtil.makeShortName("123456789012345678901234567890.txt", 20));
        System.out.println(StrUtil.makeShortName("123456789012345678901234567890.tx", 20));
        System.out.println(StrUtil.makeShortName("123456789012345678901234567890.txt1", 20));
        System.out.println(StrUtil.makeShortName("123456789012345678901234567890", 30));
        System.out.println(StrUtil.makeShortName("123456789012345678901234567890", 25));
        System.out.println(StrUtil.makeShortName("123456789012345678901234567891", 25));
    }
*/

}


